#ifndef SLMOTION_RANDOM
#define SLMOTION_RANDOM

#include <random>
#include <opencv2/opencv.hpp>

namespace slmotion {
  namespace random {
    // Global helper functions that provide easy access to pseudorandom numbers

    typedef std::mt19937_64 engine_t;
    static_assert(std::is_same<engine_t::result_type, uint64_t>::value,
                  "Expected result type = uint64_t but is not");
    
    /**
     * Sets the random number seed.
     */
    void seed(uint64_t seed = time(nullptr));

    /**
     * Generates a random number in [min,max] range, usually [0,2^64-1]
     */
    uint64_t rand();

    /**
     * The minimum value of rand()
     */
    uint64_t min();

    /**
     * The maximum value of rand()
     */
    uint64_t max();

    /**
     * Returns the engine
     */
    engine_t& getEngine();

    /**
     * Helper function: given a distribution, generates a vectorful of stuff
     */
    template <typename T, typename U>
    std::vector<T> generateRandomVector(size_t nSamples, U& distribution) {
      std::vector<T> samples(nSamples);
      engine_t& engine = getEngine();
      for (size_t i = 0; i < nSamples; ++i)
        samples[i] = distribution(engine);
      return samples;
    }

    /**
     * Helper function: given a distribution, generates a matrix of stuff
     */
    template <typename T, typename U>
    cv::Mat_<T> generateRandomMatrix(int rows, int cols, U& distribution) {
      cv::Mat_<T> samples(rows, cols);
      engine_t& engine = getEngine();
      for (int i = 0; i < rows; ++i)
        for (int j = 0; j < cols; ++j)
          samples(i,j) = distribution(engine);
      return samples;
    }

    /**
     * Generates uniformly distributed random numbers
     */
    template <typename T>
    T unifrnd(T a, T b) {
      std::uniform_int_distribution<T> d(a,b);
      return d(getEngine());
    }

    /**
     * Generates uniformly distributed random numbers
     */
    template <>
    float unifrnd(float a, float b);

    /**
     * Generates uniformly distributed random numbers
     */
    template <>
    double unifrnd(double a, double b);

    /**
     * Generates uniformly distributed random numbers
     */
    template <typename T>
    std::vector<T> unifrnd(T a, T b, size_t N) {
      std::uniform_int_distribution<T> d(a,b);
      return generateRandomVector<T>(N, d);
    }

    /**
     * Generates uniformly distributed random numbers
     */
    template <>
    std::vector<float> unifrnd(float a, float b, size_t N);

    /**
     * Generates uniformly distributed random numbers
     */
    template <>
    std::vector<double> unifrnd(double a, double b, size_t N);

    /**
     * Generates uniformly distributed random numbers
     */
    template <typename T>
    cv::Mat_<T> unifrnd(T a, T b, int rows, int cols) {
      std::uniform_int_distribution<T> d(a,b);
      return generateRandomMatrix<T>(rows, cols, d);
    }

    /**
     * Generates uniformly distributed random numbers
     */
    template <>
    cv::Mat_<float> unifrnd(float a, float b, int rows, int cols);

    /**
     * Generates uniformly distributed random numbers
     */
    template <>
    cv::Mat_<double> unifrnd(double a, double b, int rows, int cols);
  }
}

#endif

