/*
 *
 * This file contains a very simple program to perform tests on different
 * types of skin detectors and their parameters
 *
 */

#include <iostream>
#include <cassert>
#include <typeinfo>
#include <boost/lexical_cast.hpp>
#include "slmotion.hpp"

using cv::VideoCapture;
using std::cerr;
using std::endl;
using cv::Mat;
using std::string;
using slmotion::operator<<;
using slmotion::OpenCvAdaptiveSkinDetector;
using slmotion::GaussianSkinDetector;

void printHelp() {
  cerr <<
    "skintest [options] <filename>\n"
    "This is a small helper application for testing skin detector settings\n"
    "Currently supported options are:\n"
    "--frames <n>                           : Process n frames\n"
    "--thresholds [lb1:lb2:lb3:ub1:ub2:ub3] : Use a simple pixel value threshold\n"
    "                                         filter. If the optional colon-separated\n"
    "                                         values are specified, custom thresholds\n"
    "                                         are used.\n"
    "--open-cv-adaptive                     : Use OpenCV adaptive skin detector\n"
    "--gaussian                             : Use a simple multivariate Gaussian\n"
    "                                         detector\n"
    "--train-image <filename>               : A training image for training the\n"
    "                                         detector\n"
    "--disable-visuals                      : Disables visual output\n"
    //    "--find-parameters                      : Attempts to find values for the\n"
    //    "                                         parameters used by the chosen skin\n"
    //    "                                         model.\n"
    "--valid-image <filename>               : Validation image used for parameter\n"
    "                                         selection.\n"
    "--valid-mask <filename>                : Correctly classified mask for the\n"
    "                                         given validation image\n";
}

// returns true on success
// On failure, prints an error message and returns false
// i = argument position
template<typename T>
bool parseArgument(int argc, char** argv, int& i, T& target) {
  if (i + 1 < argc) {
    ++i;
    try {
      target = boost::lexical_cast<T>(argv[i]);
      return true;
    }
    catch (std::exception& e) {
      cerr << "ERROR: " << e.what() << std::endl;
      --i;
    }
  }
  cerr << "ERROR: The option '" << argv[i] <<
    "' expects an " << 
    (typeid(target) == typeid(int) ? "integer argument" :
     typeid(target) == typeid(unsigned int) ? "unsigned integer argument" :
     typeid(target) == typeid(unsigned long) ? "unsigned integer argument" :
     typeid(target) == typeid(string) ? "string argument" :
     "argument of unknown type")
       << endl;
  return false;
}

void trainGaussian(slmotion::GaussianSkinDetector& det,
                   const cv::Mat& trainImage) {
  if (trainImage.empty()) {
    std::cerr << "FUU FUU: Train image was specified but nothing was read." << std::endl;
    std::cerr << "The skin detector won't be trained." << std::endl;
    return;
  }

  Mat samples = det.nonZeroToSamples(trainImage,
                                     Mat(trainImage.size(), CV_8UC1,
                                         cv::Scalar::all(255)));
  det.train(samples, Mat(), false);
}



void trainGaussian(slmotion::GaussianSkinDetector& det,
                   const std::string& filename) {
  if (filename.length() == 0) {
    std::cerr << "FUU FUU: No train image was specified." << std::endl;
    std::cerr << "The skin detector won't be trained." << std::endl;
    return;
  }

  Mat trainImage = cv::imread(filename);
  cv::cvtColor(trainImage, trainImage, CV_BGR2HSV);

  trainGaussian(det, trainImage);
}


void findParameters(slmotion::OpenCvAdaptiveSkinDetector&,
                    const Mat&, const Mat&) {
  std::cout << "No explicit parameters are needed for the OpenCV Adaptive Skin Detector." << std::endl;
}



void findParameters(slmotion::SkinDetector* detector,
                    const string& validImageFile,
                    const string& validMaskFile) {
  Mat validImage = cv::imread(validImageFile);
  Mat validMask = cv::imread(validMaskFile);
  detector->findParameters(validImage, validMask);
}



int main(int argc, char** argv) {
  if (argc == 1) {
    printHelp();
    return 0;
  }

  enum FilterType {
    THRESHOLDS,
    OPENCV_ADAPTIVE,
    GAUSSIAN
  } filterType = OPENCV_ADAPTIVE;
  bool acceptOptions = true;
  bool visuals = true;
  string inFile;
  size_t frames = 0;
  cv::Scalar upperb, lowerb;
  bool defaultThresholds = true;
  string trainImageFile;
  string validImage;
  string validMask;

  for (int i = 1; i < argc; ++i) {
    if (acceptOptions && argv[i][0] == '-') {
      if (!strcmp(argv[i], "--"))
	acceptOptions = false;
      else if (!strcmp(argv[i], "--frames")) {
        if (!parseArgument(argc, argv, i, frames))
          return -1;
      }
      else if (!strcmp(argv[i], "--thresholds")) {
        filterType = THRESHOLDS;
        if (i + 1 < argc) {
          if (sscanf(argv[i+1], "%lf:%lf:%lf:%lf:%lf:%lf", &lowerb[0],
                     &lowerb[1], &lowerb[2], &upperb[0], &upperb[1],
                     &upperb[2]) == 6) {
            defaultThresholds = false;
            ++i;
          }
        }
      }
      else if (!strcmp(argv[i], "--train-image")) {
        if (!parseArgument(argc, argv, i, trainImageFile))
          return -1;
      }
      else if (!strcmp(argv[i], "--valid-image")) {
        if (!parseArgument(argc, argv, i, validImage))
          return -1;
      }
      else if (!strcmp(argv[i], "--valid-mask")) {
        if (!parseArgument(argc, argv, i, validMask))
          return -1;
      }
      else if (!strcmp(argv[i], "--open-cv-adaptive"))
        filterType = OPENCV_ADAPTIVE;
      else if (!strcmp(argv[i], "--gaussian"))
        filterType = GAUSSIAN;
      else if (!strcmp(argv[i], "--disable-visuals"))
        visuals = false;
      else {
	cerr << "ERROR: '" << argv[i] << "' is not a valid option." << endl;
	return -1;
      }
    }
    else if (inFile == "")
      inFile = argv[i];
    else {
      cerr << "ERROR: Only one input file may be specified" << endl;
      return -1;
    }
  }

  if (!(inFile != "" || (validImage != "" && validMask != ""))) {
    cerr << "ERROR: No input file(s) specified." << endl;
    return -1;
  }

  if ((validImage.length() > 0) == !(validMask.length() > 0)) {
    cerr << "ERROR: If either --valid-image or --valid-mask is set, the other one must be set, too." << endl;
    return -1;
  }


  
  Mat frame;
  Mat mask;

  std::auto_ptr<slmotion::SkinDetector> td;

  switch (filterType) {
  case THRESHOLDS:
    if (defaultThresholds)
      td.reset(new slmotion::ThresholdSkinDetector());
    else
      td.reset(new slmotion::ThresholdSkinDetector(lowerb, upperb));
    break;

  case OPENCV_ADAPTIVE:
    td.reset(new slmotion::OpenCvAdaptiveSkinDetector());
    break;

  case GAUSSIAN:
    td.reset(new slmotion::GaussianSkinDetector());
    trainGaussian(*dynamic_cast<slmotion::GaussianSkinDetector*>(td.get()),
                  trainImageFile);
  }

  if (validImage.length() > 0 && validMask.length() > 0) {
    findParameters(td.get(), validImage, validMask);
  }

  if (inFile.length() > 0) {

    VideoCapture capture(inFile);
    if (!capture.isOpened()) {
      cerr << "ERROR: Capture not initialised." << std::endl;
      return -1;
    }

    size_t frnumber = 0;

    if (visuals)
      cv::namedWindow("output", 1);
 
    while ( capture.grab() && (frnumber++ < frames || frames == 0) ) {
      capture.retrieve(frame);
      td->process(frame, mask);
      if (visuals) {
        cv::imshow("output", mask);
        cv::waitKey(5);
      }
    }
  }
}
