#include "regex.hpp"
#include "LimitedPriorityQueue.hpp"
#include "HandConfiguration.hpp"

#include "random.hpp"
#include "math.hpp"
#include "AdvancedString.hpp"

#define private public
#include "BlackBoardDumpWriter.hpp"
#include "DummyFrameSource.hpp"
#include "TempFile.hpp"
#include "SomPakHeaderDocument.hpp"
#include "XmlDocument.hpp"
#include "slmotion.hpp"
#include "ImageSequenceSource.hpp"
#include "VideoFileSource.hpp"
#include "ThresholdSkinDetector.hpp"
#include "GaussianSkinDetector.hpp"
#include "util.hpp"
#include "AnalysisController.hpp"
#include "ColourSpaceConverter.hpp"
#include "AsmTracker.hpp"
#include "Thread.hpp"
#include "RuleSkinDetector.hpp"
#include "OpenCVVideoCaptureVideoFileSource.hpp"
#include "Gaussian.hpp"
#include "OpenCvAdaptiveSkinDetector.hpp"
#include "Pdm.hpp"
#include "Asm.hpp"
#include "BlackBoardExplorer.hpp"
#include "LinePointIterator.hpp"
#include "VidFileSource.hpp"
#include "MultiFrameSource.hpp"
#include "OpenNiOniFileSource.hpp"
#include "configuration.hpp"
#include "FacialLandmarkDetector.hpp"
#include "CsvImporter.hpp"
#include "pythonapi.hpp"
#include "BlackBoardPointer.hpp"
#undef private
#include "HOGDescriptor.hpp"
#include "PHOG.hpp"

#ifdef SLMOTION_WITH_RUBNER_EMD
#include "emd.h"
#endif // SLMOTION_WITH_RUBNER_EMD
#ifdef SLMOTION_WITH_PELE_FASTEMD
#include "FastEMD/emd_hat.hpp"
#endif // SLMOTION_WITH_PELE_FASTEMD

#define BOOST_TEST_MAIN
#define BOOST_TEST_DYN_LINK
#define BOOST_TEST_MODULE SLMotionTestModule
#include <boost/test/unit_test.hpp>
#include <boost/algorithm/string.hpp>
#include <boost/iterator/indirect_iterator.hpp>
#include <boost/math/constants/constants.hpp>

#include <cmath>

#include <opencv2/nonfree/nonfree.hpp>

/**
 *
 * This file can be used for unit testing
 *
 * For running specific tests, please refer to this page:
 * http://www.boost.org/doc/libs/1_42_0/libs/test/doc/html/utf/user-guide/runtime-config/run-by-name.html
 */

using std::deque;
using boost::lexical_cast;
using cv::Mat;
using cv::Scalar;
using cv::Point;
using cv::imshow;
using cv::imread;
using cv::waitKey;
using cv::MatConstIterator_;
using cv::Rect;
using cv::rectangle;
using cv::ellipse;
using cv::Size;
using std::shared_ptr;
using cv::Vec3b;
using cv::Vec4b;
using cv::Vec;
using cv::Point2f;
using std::string;
using std::vector;
using std::cout;
using std::endl;
using std::set;
using cv::Point2d;
using cv::Mat_;
using cv::Vec2i;
using cv::Point2i;
using namespace slmotion;
using namespace slmotion::configuration;
using namespace slmotion::xml;
using namespace slmotion::math;



static double relativeDiff(double a, double b) {
  if (std::min(a,b) == 0)
    return std::abs(a-b);
  else
    return std::abs(a - b)/std::max(std::abs(a),std::abs(b));
}

static bool almostEqual(double a, double b, double epsilon = 1.0e-5) {
  return relativeDiff(a,b) < epsilon;
}

// true if a and b have the same signedness
static bool signEqual(int a, int b) {
  return (a < 0 && b < 0) ||
    (a > 0 && b > 0) ||
    (a == 0 && b == 0);
}

const string HAAR_CASCADE_1 = "haarcascade_frontalface_alt.xml";
const string HAAR_CASCADE_2 = "haarcascade_frontalface_alt2.xml";



  /**
   * Returns a pointer to the first component of the given type if it 
   * exist; otherwise, returns a NULL.
   */
  template<typename T>
  const T* getPreprocessComponent(const Analyser& anal) {
    for(auto it = anal.components.cbegin(); 
        it != anal.components.cend(); ++it) 
      if (dynamic_cast<const T*>(it->get()))
        return dynamic_cast<const T*>(it->get());
    
    return NULL;
  }


  
  vector<shared_ptr<PostProcessFilter>> getDefaultSkinPostProcessFilters() {
  // BOOST_CHECK(config.skinDetectorPostProcessFilterString == "morph(erode,1);watershed(50);morph(close,2)");

    vector<shared_ptr<PostProcessFilter>> filters;
    filters.push_back(shared_ptr<PostProcessFilter>(new PostProcessMorphologicalTransformationFilter(PostProcessMorphologicalTransformationFilter::Type::ERODE, 1)));
    filters.push_back(shared_ptr<PostProcessFilter>(new PostProcessWatershedFilter(50)));
    filters.push_back(shared_ptr<PostProcessFilter>(new PostProcessMorphologicalTransformationFilter(PostProcessMorphologicalTransformationFilter::Type::CLOSE, 2)));
    return filters;
  }



vector<string> getTestVideoFramefilenames() {
    vector<string> r;
    // string prefix = "/share/imagedb/slmotion/tests/00000";
    // string prefix = "/share/imagedb/slmotion/tests/testivideo4frames/00000";
    string prefix = SLMOTION_TEST_DATA_DIRECTORY"/testivideo4frames/00000";
    string postfix = ".png";
    for(int i = 0; i < 10; ++i) 
      r.push_back(prefix + lexical_cast<string>(i) + postfix);
    return r;
}

#ifdef SLMOTION_ENABLE_OPENNI
static std::map<std::string, vector<string>> getOniTestVideoFrameFilenames() {
  std::string oniprefix = string(SLMOTION_TEST_DATA_DIRECTORY"/oni/");
  std::map<std::string, vector<string>> filenames;
  filenames["rgb-skel"] = std::vector<string>(5906);
  filenames["rgb-noskel"] = std::vector<string>(5906);
  filenames["depth-skel"] = std::vector<string>(5906);
  filenames["depth-noskel"] = std::vector<string>(5906);
  for (unsigned int i = 0; i < 5906; ++i) {
    char fn[7];
    sprintf(fn, "%06d", i);
    filenames["rgb-skel"][i] = oniprefix + "rgb-skel-" + fn + ".png";
    filenames["rgb-noskel"][i] = oniprefix + "rgb-noskel-" + fn + ".png";
    filenames["depth-skel"][i] = oniprefix + "depth-skel-" + fn + ".png";
    filenames["depth-noskel"][i] = oniprefix + "depth-noskel-" + fn + ".png";
  }
  return filenames;
}
#endif // SLMOTION_ENABLE_OPENNI

//  const string TESTVIDEOFILE = "/share/imagedb/slmotion/tests/testivideo2.avi";
  const string SUVI_TEST_VIDEO = SLMOTION_TEST_DATA_DIRECTORY"/000100.avi";
  const string KINECT_VIDEO = SLMOTION_TEST_DATA_DIRECTORY"/M_1.avi";
  const string KINECT_DEPTH = SLMOTION_TEST_DATA_DIRECTORY"/K_1.avi";
#ifdef SLMOTION_ENABLE_VIDFILE_SUPPORT
  const string TESTVIDFILE = SLMOTION_TEST_DATA_DIRECTORY"/testivideo5.vid";
#endif // SLMOTION_ENABLE_VIDFILE_SUPPORT
#ifdef SLMOTION_ENABLE_OPENNI
  const string MARKUS_ONI = SLMOTION_TEST_DATA_DIRECTORY"/markus.oni";
  const string CAPTURE_ONI_NOSKEL = SLMOTION_TEST_DATA_DIRECTORY"/oni/Capture_20130222110437213-noskel.oni";
  const string CAPTURE_ONI_SKEL = SLMOTION_TEST_DATA_DIRECTORY"/oni/Capture_20130222110437213.oni";
  const string CAPTURE_ONI_SKEL_CSV = SLMOTION_TEST_DATA_DIRECTORY"/oni/Capture_20130222110437213.csv";
#endif // SLMOTION_ENABLE_OPENNI
  const string TESTVIDEOFILE = SLMOTION_TEST_DATA_DIRECTORY"/testivideo4.avi";
  const string TESTVIDEOFILE_FFV1 = SLMOTION_TEST_DATA_DIRECTORY"/testivideo4_ffv1.avi";
  const string TESTVIDEOFILE2 = SLMOTION_TEST_DATA_DIRECTORY"/testivideo3.avi";
  const string TESTVIDEOFILE3 = SLMOTION_TEST_DATA_DIRECTORY"/testivideo5.avi";

  const string TESTVIDEOFILE6 = SLMOTION_TEST_DATA_DIRECTORY"/testvideo6.avi";

  const string TESTVIDEOFILE7 = SLMOTION_TEST_DATA_DIRECTORY"/testvideo7.avi";


  const vector<Mat> TESTVIDEOFILE_FFV1_FRAMES {
    cv::imread(SLMOTION_TEST_DATA_DIRECTORY"/testivideo4_ffv1frames/00000001.png"),
      cv::imread(SLMOTION_TEST_DATA_DIRECTORY"/testivideo4_ffv1frames/00000002.png"),
      cv::imread(SLMOTION_TEST_DATA_DIRECTORY"/testivideo4_ffv1frames/00000003.png"),	
      cv::imread(SLMOTION_TEST_DATA_DIRECTORY"/testivideo4_ffv1frames/00000004.png"),	
      cv::imread(SLMOTION_TEST_DATA_DIRECTORY"/testivideo4_ffv1frames/00000005.png"),
      cv::imread(SLMOTION_TEST_DATA_DIRECTORY"/testivideo4_ffv1frames/00000006.png"),
      cv::imread(SLMOTION_TEST_DATA_DIRECTORY"/testivideo4_ffv1frames/00000007.png"),
      cv::imread(SLMOTION_TEST_DATA_DIRECTORY"/testivideo4_ffv1frames/00000008.png"),
      cv::imread(SLMOTION_TEST_DATA_DIRECTORY"/testivideo4_ffv1frames/00000009.png"),
      cv::imread(SLMOTION_TEST_DATA_DIRECTORY"/testivideo4_ffv1frames/00000010.png")
      };


const std::string TESTIMAGE = SLMOTION_TEST_DATA_DIRECTORY"/000109.png";
const std::string TESTIMAGE_SKINMASK = SLMOTION_TEST_DATA_DIRECTORY"/000109_skinmask.png";
const std::string TESTIMAGE2 = SLMOTION_TEST_DATA_DIRECTORY"/000036.png";
const std::string TESTIMAGE2_SKINMASK = SLMOTION_TEST_DATA_DIRECTORY"/000036_skinmask.png";

static vector<Mat> getTestVideoFrames() {
  vector<Mat> s;
  // auto r = getTestVideoFramefilenames();
  // for (auto i = r.cbegin(); i < r.cend(); ++i) 
  //   s.push_back(imread(*i));
  cv::VideoCapture vc(TESTVIDEOFILE);
  cv::Mat temp;
  while(vc.read(temp))
    s.push_back(temp.clone());
  return s;
}
static const vector<Mat> testVideoFrames = std::move(getTestVideoFrames());

static vector<Mat> getTestVideoFrames2() {
  vector<Mat> s;
  auto r = getTestVideoFramefilenames();
  for (auto i = r.cbegin(); i < r.cend(); ++i) 
    s.push_back(imread(*i));
  return s;
}
static const vector<Mat> testVideoFrames2 = std::move(getTestVideoFrames2());




/**
 * Gets two frames, and returns a new frame that has the content of the two
 * frames laid side-by-side
 */
// static Mat sideBySide(const cv::Mat& f1, const cv::Mat& f2) {
//   assert(f1.size() == f2.size() && f1.type() == f2.type());   
//   Mat temp2(f1.rows, f1.cols*2, f1.type());
//   Mat temp3 = temp2.colRange(0, f1.cols);
//   f1.copyTo(temp3);
//   temp3 = temp2.colRange(f1.cols, temp2.cols);
//   f2.copyTo(temp3);
//   return temp2;
// }



/**
 * Gets two frames, and computes the absolute difference between them
 */
// static Mat residual(const cv::Mat& f1, const cv::Mat& f2) {
//   assert(f1.size() == f2.size() && f1.type() == f2.type());
//   Mat f3(f1.size(), f1.type());
//   cv::MatConstIterator_<uchar> i = f1.begin<uchar>();
//   cv::MatConstIterator_<uchar> j = f2.begin<uchar>();
//   cv::MatIterator_<uchar> k = f3.begin<uchar>();
//   while(i != f1.end<uchar>() && j != f2.end<uchar>()) {
//     *k = std::max(*i,*j) - std::min(*i,*j);
//     ++i, ++j, ++k;
//   }
//   return f3;
// }

namespace {
  static struct STORE_EXEC {
    STORE_EXEC() {
      storeExecutableName("tests");    
    }
  } STORE_EXEC;

  // a bogus class for checking that the required files exist
  static struct FILECHECKER {
    FILECHECKER() {
      std::vector<std::string> filenames {
        KINECT_VIDEO, 
          KINECT_DEPTH, 
          SUVI_TEST_VIDEO,
#ifdef SLMOTION_ENABLE_VIDFILE_SUPPORT
          TESTVIDFILE, 
#endif // SLMOTION_ENABLE_VIDFILE_SUPPORT
#ifdef SLMOTION_ENABLE_OPENNI
          MARKUS_ONI,
          CAPTURE_ONI_NOSKEL,
          CAPTURE_ONI_SKEL,
          CAPTURE_ONI_SKEL_CSV,
#endif // SLMOTION_ENABLE_OPENNI
          TESTVIDEOFILE,
          TESTVIDEOFILE_FFV1,
          TESTVIDEOFILE2,
          TESTVIDEOFILE3,
          TESTVIDEOFILE6,
          TESTVIDEOFILE7,
          TESTIMAGE,
          TESTIMAGE_SKINMASK,
          TESTIMAGE2,
          TESTIMAGE2_SKINMASK,
          SLMOTION_TEST_DATA_DIRECTORY"/testivideo4_ffv1frames/00000001.png",
          SLMOTION_TEST_DATA_DIRECTORY"/testivideo4_ffv1frames/00000002.png",
          SLMOTION_TEST_DATA_DIRECTORY"/testivideo4_ffv1frames/00000003.png",	
          SLMOTION_TEST_DATA_DIRECTORY"/testivideo4_ffv1frames/00000004.png",	
          SLMOTION_TEST_DATA_DIRECTORY"/testivideo4_ffv1frames/00000005.png",
          SLMOTION_TEST_DATA_DIRECTORY"/testivideo4_ffv1frames/00000006.png",
          SLMOTION_TEST_DATA_DIRECTORY"/testivideo4_ffv1frames/00000007.png",
          SLMOTION_TEST_DATA_DIRECTORY"/testivideo4_ffv1frames/00000008.png",
          SLMOTION_TEST_DATA_DIRECTORY"/testivideo4_ffv1frames/00000009.png",
          SLMOTION_TEST_DATA_DIRECTORY"/testivideo4_ffv1frames/00000010.png",
          slmotion::configuration::getDefaultConfigurationFile(),
          getDefaultHaarCascadeFile(),
          getAnnotationSchemaFile(), // ANNOTATIONSCHEMAFILE,
          getSomPakFeatureSchemaFile(), //SOM_PAK_FEATURE_SCHEMA_FILE,
          getInstallPrefix() + "/share/slmotion/default.ann",
          getEafSchemaFile() // EAF_SCHEMA_FILE
          };
      auto r = getTestVideoFramefilenames();
      filenames.insert(filenames.end(), r.cbegin(), r.cend());
#ifdef SLMOTION_ENABLE_OPENNI
      auto r2 = getOniTestVideoFrameFilenames();
      auto r3 = r2["rgb-noskel"];
      filenames.insert(filenames.end(), r3.cbegin(), r3.cend());
      r3 = r2["depth-noskel"];
      filenames.insert(filenames.end(), r3.cbegin(), r3.cend());
      r3 = r2["depth-skel"];
      filenames.insert(filenames.end(), r3.cbegin(), r3.cend());
      r3 = r2["rgb-skel"];
      filenames.insert(filenames.end(), r3.cbegin(), r3.cend());
#endif // SLMOTION_ENABLE_OPENNI
      std::cout << "Checking the existence of all test input files...";

      bool allGood = true;
      for (auto it = filenames.cbegin(); it != filenames.cend(); ++it) {
        std::ifstream ifs(*it);
        if (!ifs.good()) {
          if (allGood)
            std::cout << std::endl;
          std::cout << "ERROR: Could not open \"" << *it << "\"!" << std::endl;
          allGood = false;
        }
      }
      
      if (allGood)
        std::cout << " OK!" << std::endl;
      else
        exit(EXIT_FAILURE);
    }
  } CHECK_FILES_EXIST;
}



static std::string getRandomString(size_t length = 8) {
  std::vector<char> str(length + 1);
  for (size_t i = 0; i < str.size(); ++i)
    str[i] = 'a' + rand() % 26;
  // str[str.size()-1] = '\0';
  str.back() = '\0';
  
  return std::string(&str[0]);
}



template<typename T> static T getZero();

template<>
cv::Vec3b getZero() {
  return cv::Vec3b(0,0,0);
}



template<typename T> static int countNonZero(const cv::Mat& m) {
  T zero = getZero<T>();
  int k = 0;
  for (int i = 0; i < m.rows; ++i)
    for (int j = 0; j < m.cols; ++j)
      if (m.at<T>(i,j) != zero)
        ++k;
  return k;
}



template<typename T>
static T* getComponent(Analyser& analyser) {
  T* t = NULL;
  for (auto it = analyser.components.begin(); it != analyser.components.end(); ++it)
    if ((t = dynamic_cast<T*>(it->get())))
      return t;
  return t;
}



template<typename T>
static bool bbEqualT(const boost::any& l, const boost::any& r) {
  const T& lh = boost::any_cast<T>(l);
  const T& rh = boost::any_cast<T>(r);
  return lh == rh;
}

template<>
bool bbEqualT<FeatureVector>(const boost::any& l, const boost::any& r) {
  const FeatureVector& lh = boost::any_cast<FeatureVector>(l);
  const FeatureVector& rh = boost::any_cast<FeatureVector>(r);
  if (lh.components != rh.components)
    return false;

  if (lh.features.size() != rh.features.size())
    return false;

  auto it = lh.features.cbegin(), jt = rh.features.cbegin();
  while (it != lh.features.cend()) {
    if (it->size() != jt->size())
      return false;
    
    if (it->size() > 0) {
      auto iit = it->cbegin(), jjt = jt->cbegin();

      while (iit != it->cend()) {
        if (iit->type() != jjt->type())
          return false;
        else if (iit->type() == typeid(int)) {
          if (!bbEqualT<int>(*iit, *jjt))
            return false;
        }
        else if (iit->type() == typeid(double)) {
          if (!bbEqualT<double>(*iit, *jjt))
            return false;
        }
        else if (iit->type() == typeid(bool)) {
          if (!bbEqualT<bool>(*iit, *jjt))
            return false;
        }
        else 
          return false;

        ++iit;
        ++jjt;
      }
    }

    ++it;
    ++jt;
  }

  if (lh.firstFrame != rh.firstFrame)
    return false;

  if (lh.lastFrame != rh.lastFrame)
    return false;

  return true;
}

template<>
bool bbEqualT<Asm>(const boost::any& l, const boost::any& r) {
  const Asm& lh = boost::any_cast<Asm>(l);
  const Asm& rh = boost::any_cast<Asm>(r);
  return *lh.pdm == *rh.pdm &&
    lh.nComponents == rh.nComponents &&
    lh.maxShapeParameterDeviation == rh.maxShapeParameterDeviation;
}

static bool bbEqual(const BlackBoardPointer<boost::any>& lh, const BlackBoardPointer<boost::any>& rh) {
  if (lh->type() != rh->type())
    return false;
 
  if (lh->type() == typeid(cv::Rect))
    return bbEqualT<cv::Rect>(*lh, *rh);
  else if (lh->type() == typeid(cv::Mat))
    return equal(boost::any_cast<cv::Mat>(*lh), boost::any_cast<cv::Mat>(*rh));
  else if (lh->type() == typeid(vector<Blob>))
    return bbEqualT<vector<Blob>>(*lh, *rh);
  else if (lh->type() == typeid(std::list<BodyPart>))
    return bbEqualT<std::list<BodyPart>>(*lh, *rh);
  else if (lh->type() == typeid(std::set<TrackedPoint>))
    return bbEqualT<std::set<TrackedPoint>>(*lh, *rh);
  else if (lh->type() == typeid(Asm::Instance))
    return bbEqualT<Asm::Instance>(*lh, *rh);
  else if (lh->type() == typeid(Asm))
    return bbEqualT<Asm>(*lh, *rh);
  else if (lh->type() == typeid(cv::Point2d))
    return bbEqualT<cv::Point2d>(*lh, *rh);
  else if (lh->type() == typeid(cv::Point))
    return bbEqualT<cv::Point>(*lh, *rh);
  else if (lh->type() == typeid(double))
    return bbEqualT<double>(*lh, *rh);
  else if (lh->type() == typeid(std::vector<cv::Point2f>))
    return bbEqualT<std::vector<cv::Point2f>>(*lh, *rh);
  else if (lh->type() == typeid(std::map<unsigned long long, cv::Point_<float> >))
    return bbEqualT<std::map<unsigned long long, cv::Point_<float> > >(*lh, *rh);
  else if (lh->type() == typeid(std::map<unsigned long long, slmotion::BodyPart::PartName >))
    return bbEqualT<std::map<unsigned long long, slmotion::BodyPart::PartName > >(*lh, *rh);
  else if (lh->type() == typeid(std::map<unsigned long long, slmotion::TrackedPoint>))
    return bbEqualT<std::map<unsigned long long, slmotion::TrackedPoint> >(*lh, *rh);
  else if (lh->type() == typeid(FeatureVector))
    return bbEqualT<FeatureVector>(*lh, *rh);
  else 
    std::cerr << "INVALID TYPE: " << string_demangle(lh->type().name()) << std::endl;

  return false;
}



static bool bbEqual(const BlackBoard& bb1, const BlackBoard& bb2) {
  if (bb1.size() != bb2.size())
    return false;

  if (bb1.frameBoard.size() != bb2.frameBoard.size())
    return false;

  for (auto it = bb1.frameBoard.cbegin(), jt = bb2.frameBoard.cbegin(); 
       it != bb1.frameBoard.cend(); ++it, ++jt) {
    if (it->first != jt->first)
      return false;
    
    if(!bbEqual(it->second->referenceAny(it->second), 
                jt->second->referenceAny(jt->second)))
      return false;
  }

  if (bb1.globalBoard.size() != bb2.globalBoard.size())
    return false;

  for (auto it = bb1.globalBoard.cbegin(), jt = bb2.globalBoard.cbegin(); 
       it != bb1.globalBoard.cend(); ++it, ++jt) {
    if (it->first != jt->first)
      return false;
    if (!bbEqual(it->second->referenceAny(it->second), 
                 jt->second->referenceAny(jt->second)))
      return false;
  }

  return true;
}



static void DUMMYFUNCTION(void*, const char*, ...) {
}



static bool L2Compare(const cv::Point& lh, const cv::Point& rh) {
  return cv::norm(lh) < cv::norm(rh);
}



// video file source type used whilst performing these tests
typedef OpenCVVideoCaptureVideoFileSource vfs_t;



/**
 * Returns true if the matrices are of equivalent type and size, AND
 * the following property holds:
 *  for each pixel m1(x,y)[k] at point (x,y) in colour channel k in matrix
 *   m1, m1(x,y)[k] == m2(x,y)[mappings[k]] 
 */
template<int channels>
static bool checkMatMapping(const Mat& m1, const Mat& m2,
                            Vec<uchar,channels>& mappings) {
  if (m1.size() != m2.size())
    return false;
  if (m1.type() != CV_8UC(channels))
    return false;
  if (m2.type() != CV_8UC(channels))
    return false;
  
  for (int i = 0; i < m1.rows; ++i) {
    for (int j = 0; j < m1.cols; ++j) {
      typedef const Vec<uchar,channels> vec_t;
      vec_t& v1 = m1.at<vec_t>(i,j);
      vec_t& v2 = m2.at<vec_t>(i,j);
      for (int k = 0; k < channels; ++k) 
        if (v1[k] != v2[mappings[k]])
          return false;
    }
  }
  return true;
}



BOOST_AUTO_TEST_SUITE(tests) 


// ugly but editing a file with ~10k lines is tedious
// tests contained in tests_1.cpp are older and should not change that often
#include "tests_1.cpp"



BOOST_AUTO_TEST_CASE( testHandConfiguration ) {
  std::mt19937 engine(time(nullptr));
  std::uniform_real_distribution<double> distribution(-360, 360);
  // create a random configuration
  HandConfiguration hc;

  BOOST_REQUIRE(hc.getLittleFinger().getMcp().first == 0);
  BOOST_REQUIRE(hc.getLittleFinger().getMcp().second == 0);
  BOOST_REQUIRE(hc.getLittleFinger().getPip() == 0);
  BOOST_REQUIRE(hc.getLittleFinger().getDip() == 0);

  BOOST_REQUIRE(hc.getRingFinger().getMcp().first == 0);
  BOOST_REQUIRE(hc.getRingFinger().getMcp().second == 0);
  BOOST_REQUIRE(hc.getRingFinger().getPip() == 0);
  BOOST_REQUIRE(hc.getRingFinger().getDip() == 0);

  BOOST_REQUIRE(hc.getMiddleFinger().getMcp().first == 0);
  BOOST_REQUIRE(hc.getMiddleFinger().getMcp().second == 0);
  BOOST_REQUIRE(hc.getMiddleFinger().getPip() == 0);
  BOOST_REQUIRE(hc.getMiddleFinger().getDip() == 0);

  BOOST_REQUIRE(hc.getIndexFinger().getMcp().first == 0);
  BOOST_REQUIRE(hc.getIndexFinger().getMcp().second == 0);
  BOOST_REQUIRE(hc.getIndexFinger().getPip() == 0);
  BOOST_REQUIRE(hc.getIndexFinger().getDip() == 0);

  BOOST_REQUIRE(hc.getThumb().getCm().first == 0);
  BOOST_REQUIRE(hc.getThumb().getCm().second == 0);
  BOOST_REQUIRE(hc.getThumb().getMcp().first == 0);
  BOOST_REQUIRE(hc.getThumb().getMcp().second == 0);
  BOOST_CHECK_THROW(hc.getThumb().getPip(),
                    std::out_of_range);
  BOOST_REQUIRE(hc.getThumb().getDip() == 0);


  BOOST_REQUIRE(hc.getRc().first == 0);
  BOOST_REQUIRE(hc.getRc().second == 0);

  BOOST_REQUIRE(hc.getBaseMetaJoint().first == 0);
  BOOST_REQUIRE(hc.getBaseMetaJoint().second == 0);

  BOOST_REQUIRE(hc.getDru() == 0);

  cv::Mat m = hc.toMat();
  BOOST_REQUIRE(m.rows == 25);
  BOOST_REQUIRE(m.cols == 1);
  BOOST_REQUIRE(m.type() == CV_64FC1);
  for (int i = 0; i < m.rows; ++i)
    BOOST_REQUIRE(m.at<double>(i) == 0);

  cv::Mat m2 = static_cast<cv::Mat>(hc);
  BOOST_REQUIRE(&m2 != &m);
  BOOST_REQUIRE(m2.data != m.data);
  BOOST_REQUIRE(m2.rows == 25);
  BOOST_REQUIRE(m2.cols == 1);
  BOOST_REQUIRE(m2.type() == CV_64FC1);
  for (int i = 0; i < m2.rows; ++i)
    BOOST_REQUIRE(m2.at<double>(i) == 0);

  // set random data
  std::vector<double> randomData = slmotion::random::unifrnd<double>(-180.,180.,25);
  int i = 0;
  hc.getLittleFinger().getMcp().first = randomData[i++];
  hc.getLittleFinger().getMcp().second = randomData[i++];
  hc.getLittleFinger().getPip() = randomData[i++];
  hc.getLittleFinger().getDip() = randomData[i++];

  hc.getRingFinger().getMcp().first = randomData[i++];
  hc.getRingFinger().getMcp().second = randomData[i++];
  hc.getRingFinger().getPip() = randomData[i++];
  hc.getRingFinger().getDip() = randomData[i++];

  hc.getMiddleFinger().getMcp().first = randomData[i++];
  hc.getMiddleFinger().getMcp().second = randomData[i++];
  hc.getMiddleFinger().getPip() = randomData[i++];
  hc.getMiddleFinger().getDip() = randomData[i++];

  hc.getIndexFinger().getMcp().first = randomData[i++];
  hc.getIndexFinger().getMcp().second = randomData[i++];
  hc.getIndexFinger().getPip() = randomData[i++];
  hc.getIndexFinger().getDip() = randomData[i++];

  hc.getThumb().getCm().first = randomData[i++];
  hc.getThumb().getCm().second = randomData[i++];
  hc.getThumb().getMcp().first = randomData[i++];
  // hc.getThumb().getMcp().second = randomData[i++];
  hc.getThumb().getMcp().second = slmotion::random::unifrnd<double>(-180.,180.);
  BOOST_CHECK_THROW(hc.getThumb().getPip() = slmotion::random::unifrnd<double>(-180.,180.),
                    std::out_of_range);
  hc.getThumb().getDip() = randomData[i++];


  hc.getRc().first = randomData[i++];
  hc.getRc().second = randomData[i++];

  hc.getBaseMetaJoint().first = randomData[i++];
  hc.getBaseMetaJoint().second = randomData[i++];

  hc.getDru() = randomData[i++];

  BOOST_REQUIRE(i == 25);

  cv::Mat m3 = hc.toMat();
  BOOST_REQUIRE(&m3 != &m);
  BOOST_REQUIRE(&m3 != &m2);
  BOOST_REQUIRE(m3.data != m.data);
  BOOST_REQUIRE(m3.data != m2.data);
  BOOST_REQUIRE(m3.rows == 25);
  BOOST_REQUIRE(m3.cols == 1);
  BOOST_REQUIRE(m3.type() == CV_64FC1);
  for (int i = 0; i < m.rows; ++i) 
    BOOST_REQUIRE(m3.at<double>(i) == randomData[i]);
  

  cv::Mat m4 = static_cast<cv::Mat>(hc);
  BOOST_REQUIRE(&m4 != &m);
  BOOST_REQUIRE(&m4 != &m2);
  BOOST_REQUIRE(&m4 != &m3);
  BOOST_REQUIRE(m4.data != m.data);
  BOOST_REQUIRE(m4.data != m2.data);
  BOOST_REQUIRE(m4.data != m3.data);
  BOOST_REQUIRE(m4.rows == 25);
  BOOST_REQUIRE(m4.cols == 1);
  BOOST_REQUIRE(m4.type() == CV_64FC1);
  for (int i = 0; i < m2.rows; ++i)
    BOOST_REQUIRE(m4.at<double>(i) == randomData[i]);

  cv::Mat m5 = slmotion::random::unifrnd<double>(-180.,180.,25,1);
  BOOST_REQUIRE(m5.rows == 25);
  BOOST_REQUIRE(m5.cols == 1);
  BOOST_REQUIRE(m5.type() == CV_64FC1);
  hc = m5;

  i = 0;
  BOOST_REQUIRE(hc.getLittleFinger().getMcp().first == m5.at<double>(i++));
  BOOST_REQUIRE(hc.getLittleFinger().getMcp().second == m5.at<double>(i++));
  BOOST_REQUIRE(hc.getLittleFinger().getPip() == m5.at<double>(i++));
  BOOST_REQUIRE(hc.getLittleFinger().getDip() == m5.at<double>(i++));

  BOOST_REQUIRE(hc.getRingFinger().getMcp().first == m5.at<double>(i++));
  BOOST_REQUIRE(hc.getRingFinger().getMcp().second == m5.at<double>(i++));
  BOOST_REQUIRE(hc.getRingFinger().getPip() == m5.at<double>(i++));
  BOOST_REQUIRE(hc.getRingFinger().getDip() == m5.at<double>(i++));

  BOOST_REQUIRE(hc.getMiddleFinger().getMcp().first == m5.at<double>(i++));
  BOOST_REQUIRE(hc.getMiddleFinger().getMcp().second == m5.at<double>(i++));
  BOOST_REQUIRE(hc.getMiddleFinger().getPip() == m5.at<double>(i++));
  BOOST_REQUIRE(hc.getMiddleFinger().getDip() == m5.at<double>(i++));

  BOOST_REQUIRE(hc.getIndexFinger().getMcp().first == m5.at<double>(i++));
  BOOST_REQUIRE(hc.getIndexFinger().getMcp().second == m5.at<double>(i++));
  BOOST_REQUIRE(hc.getIndexFinger().getPip() == m5.at<double>(i++));
  BOOST_REQUIRE(hc.getIndexFinger().getDip() == m5.at<double>(i++));

  BOOST_REQUIRE(hc.getThumb().getCm().first == m5.at<double>(i++));
  BOOST_REQUIRE(hc.getThumb().getCm().second == m5.at<double>(i++));
  BOOST_REQUIRE(hc.getThumb().getMcp().first == m5.at<double>(i++));
  // BOOST_REQUIRE(hc.getThumb().getMcp().second == m5.at<double>(i++));
  BOOST_REQUIRE(hc.getThumb().getMcp().second == 0);
  BOOST_CHECK_THROW(hc.getThumb().getPip(),
                    std::out_of_range);
  BOOST_REQUIRE(hc.getThumb().getDip() == m5.at<double>(i++));

  BOOST_REQUIRE(hc.getRc().first == m5.at<double>(i++));
  BOOST_REQUIRE(hc.getRc().second == m5.at<double>(i++));

  BOOST_REQUIRE(hc.getBaseMetaJoint().first == m5.at<double>(i++));
  BOOST_REQUIRE(hc.getBaseMetaJoint().second == m5.at<double>(i++));

  BOOST_REQUIRE(hc.getDru() == m5.at<double>(i++));

  BOOST_REQUIRE(i == 25);
}



BOOST_AUTO_TEST_CASE( testPartialDerivative ) {
  using namespace slmotion::math;
  using namespace slmotion::random;
  seed();
  auto f1 = [](const cv::Mat& x) { return exp(x.at<double>(0)); };
  auto pd1 = partialDerivative (f1, 0);
  cv::Mat x(1,1,CV_64FC1);
  for (double d : unifrnd<double>(-5., 100., 100)) {
    x.at<double>(0) = d;
    BOOST_REQUIRE(abs((pd1(x) - exp(d))/exp(d)) < 1e-7);
  }
  auto f2 = [](const cv::Mat& x) { return sin(x.at<double>(0)); };
  auto pd2 = partialDerivative(f2, 0);
  for (double d : unifrnd<double>(-5., 100., 100)) {
    x.at<double>(0) = d;
    BOOST_REQUIRE(abs((pd2(x) - cos(d))/cos(d)) < 1e-7);
  }

  x = cv::Mat(2,1,CV_64FC1);
  auto f3 = [](const cv::Mat& x) { 
    double x1 = x.at<double>(0);
    double x2 = x.at<double>(1);
    return x1*x1*x1*x2 + 5.0*sin(x2) + x1;
  };
  auto pd3 = partialDerivative(f3, 0);

  auto randvals = unifrnd<double>(-5., 100., 200);
  for (int i = 0; i < 100; i+=2) {
    double x1 = randvals[2*i];
    double x2 = randvals[2*i+1];
    x.at<double>(0) = x1;
    x.at<double>(1) = x2;
    double target = 3.0*x1*x1*x2 + 1.0;
    BOOST_REQUIRE(abs((pd3(x) - target)/target) < 1e-7);
  }

  auto pd4 = partialDerivative(f3, 1);
  randvals = unifrnd<double>(-5., 100., 200);
  for (int i = 0; i < 100; i+=2) {
    double x1 = randvals[2*i];
    double x2 = randvals[2*i+1];
    x.at<double>(0) = x1;
    x.at<double>(1) = x2;
    double target = x1*x1*x1 + 5.0*cos(x2);
    BOOST_REQUIRE(abs((pd4(x) - target)/target) < 1e-5);
  }

  // pd5 = ddf3/dx1dx1
  auto pd5 = partialDerivative(pd3, 0);
  randvals = unifrnd<double>(-5., 100., 200);
  for (int i = 0; i < 100; i+=2) {
    double x1 = randvals[2*i];
    double x2 = randvals[2*i+1];
    x.at<double>(0) = x1;
    x.at<double>(1) = x2;
    double target = 6.0*x1*x2;
    BOOST_REQUIRE(abs((pd5(x) - target)/target) < 1e-5);
  }

  // pd6 = ddf3/dx1dx2
  auto pd6 = partialDerivative(pd3, 1);
  randvals = unifrnd<double>(-5., 100., 200);
  for (int i = 0; i < 100; i+=2) {
    double x1 = randvals[2*i];
    double x2 = randvals[2*i+1];
    x.at<double>(0) = x1;
    x.at<double>(1) = x2;
    double target = 3.0*x1*x1;
    BOOST_REQUIRE(abs((pd6(x) - target)/target) < 1e-3);
  }

  // pd7 = ddf3/dx2dx1
  auto pd7 = partialDerivative(pd4, 0);
  randvals = unifrnd<double>(-5., 100., 200);
  for (int i = 0; i < 100; i+=2) {
    double x1 = randvals[2*i];
    double x2 = randvals[2*i+1];
    x.at<double>(0) = x1;
    x.at<double>(1) = x2;
    double target = 3.0*x1*x1;
    BOOST_REQUIRE(abs((pd7(x) - target)/target) < 1e-2);
  }

  // pd8 = ddf3/dx2dx2
  auto pd8 = partialDerivative(pd4, 1);
  randvals = unifrnd<double>(-3.1415, 3.1415, 200);
  for (int i = 0; i < 100; i+=2) {
    double x1 = randvals[2*i];
    double x2 = randvals[2*i+1];
    x.at<double>(0) = x1;
    x.at<double>(1) = x2;
    double target = -5.0*sin(x2);
    BOOST_REQUIRE(abs((pd8(x) - target)/target) < 0.2);
  }

  auto f4 = [](const cv::Mat& m)->double {
    double x = m.at<double>(0);
    double y = m.at<double>(1);
    double z = m.at<double>(2);

    return x*x*z + x*y*z + z*z*z;
  };

  x = cv::Mat(3,1,CV_64FC1);
  cv::Mat& vec = x;
  auto H = hessian(f4,3);
  for (int i = 0; i < 50; ++i) {
    randvals = unifrnd<double>(-3.1415, 3.1415, 300);
    for (int k = 0; k < 300; k += 3) {
      double x = vec.at<double>(0) = randvals[k];
      double y = vec.at<double>(1) = randvals[k+1];
      double z = vec.at<double>(2) = randvals[k+2];
      
      cv::Mat He = H.eval(vec);
      BOOST_REQUIRE(He.type() == CV_64FC1);
      BOOST_REQUIRE(He.cols == 3);
      BOOST_REQUIRE(He.rows == 3);
      BOOST_REQUIRE(almostEqual(He.at<double>(0,0), 2.0*z, 1e-2));
      BOOST_REQUIRE(abs(He.at<double>(1,1)) < 0.8); // should be zero
      BOOST_REQUIRE(almostEqual(He.at<double>(2,2), 6.0*z, 1e-3));
      
      BOOST_REQUIRE(almostEqual(He.at<double>(0,1), z, 1e-4));
      BOOST_REQUIRE(almostEqual(He.at<double>(1,0), z, 1e-4));

      BOOST_REQUIRE(almostEqual(He.at<double>(0,2), 2.0*x+y, 0.1));
      BOOST_REQUIRE(almostEqual(He.at<double>(2,0), 2.0*x+y, 0.1));
      
      BOOST_REQUIRE(almostEqual(He.at<double>(1,2), x, 0.2));
      BOOST_REQUIRE(almostEqual(He.at<double>(2,1), x, 0.2));
    }
  }
 
  auto G = gradient(f4,3);
  for (int i = 0; i < 50; ++i) {
    randvals = unifrnd<double>(-100., 100., 300);
    for (int k = 0; k < 300; k += 3) {
      double x = vec.at<double>(0) = randvals[k];
      double y = vec.at<double>(1) = randvals[k+1];
      double z = vec.at<double>(2) = randvals[k+2];
      
      cv::Mat Ge = G.eval(vec);
      BOOST_REQUIRE(Ge.type() == CV_64FC1);
      BOOST_REQUIRE(Ge.cols == 1);
      BOOST_REQUIRE(Ge.rows == 3);
      BOOST_REQUIRE(almostEqual(Ge.at<double>(0), 2.0*x*z + y*z));
      BOOST_REQUIRE(almostEqual(Ge.at<double>(1), x*z));
      BOOST_REQUIRE(almostEqual(Ge.at<double>(2), x*x+x*y+3.0*z*z));
    }
  }
}



BOOST_AUTO_TEST_CASE( testSortMat ) {
  for (int k = 0; k < 100; ++k) {
    int N = 1 + rand() % 100;
    int M = 1 + rand() % 100;
    cv::Mat vals = slmotion::random::unifrnd<double>(-100.0, 100.0, N, M);
    BOOST_REQUIRE(vals.cols == M);
    BOOST_REQUIRE(vals.rows == N);
    BOOST_REQUIRE(vals.type() == CV_64FC1);
    std::vector<cv::Point> sortedIdx = slmotion::math::sortMat<double>(vals);
    cv::Mat found(vals.size(), CV_8UC1, cv::Scalar::all(0));
    BOOST_REQUIRE(cv::countNonZero(found) == 0);
    BOOST_REQUIRE((int)sortedIdx.size() == M*N);
    for (size_t i = 0; i < sortedIdx.size(); ++i) {
      if (i > 0) {
        BOOST_REQUIRE(vals.at<double>(sortedIdx[i-1]) <= 
                      vals.at<double>(sortedIdx[i]));
      }
      found.at<uchar>(sortedIdx[i]) = 1;
    }
    BOOST_REQUIRE(cv::countNonZero(found) == M*N);
  }

  for (int k = 0; k < 100; ++k) {
    int N = 1 + rand() % 100;
    int M = 1 + rand() % 100;
    cv::Mat vals = slmotion::random::unifrnd<float>(-100.0, 100.0, N, M);
    BOOST_REQUIRE(vals.cols == M);
    BOOST_REQUIRE(vals.rows == N);
    BOOST_REQUIRE(vals.type() == CV_32FC1);
    std::vector<cv::Point> sortedIdx = slmotion::math::sortMat<float>(vals);
    cv::Mat found(vals.size(), CV_8UC1, cv::Scalar::all(0));
    BOOST_REQUIRE(cv::countNonZero(found) == 0);
    BOOST_REQUIRE((int)sortedIdx.size() == M*N);
    for (size_t i = 0; i < sortedIdx.size(); ++i) {
      if (i > 0) {
        BOOST_REQUIRE(vals.at<float>(sortedIdx[i-1]) <= 
                      vals.at<float>(sortedIdx[i]));
      }
      found.at<uchar>(sortedIdx[i]) = 1;
    }
    BOOST_REQUIRE(cv::countNonZero(found) == M*N);
  }
}

static cv::Mat getTestHand1() {
  cv::Rect bb(cv::Point(109,227),cv::Point(241,376));
  cv::Mat img = cv::imread(TESTIMAGE);
  return img(bb).clone();
}

static cv::Mat getTestHand2() {
  cv::Rect bb(207,187,124,133);
  cv::Mat img = cv::imread(TESTIMAGE2);
  return img(bb).clone();
}

static HOGDescriptor getTestHog() {
  cv::Rect bb(cv::Point(109,227),cv::Point(241,376));
  cv::Mat img = cv::imread(TESTIMAGE);
  cv::Mat imgMask = cv::imread(TESTIMAGE_SKINMASK);
  BOOST_REQUIRE(img.size() == imgMask.size() && img.type() == imgMask.type());
  for (int i = 0; i < img.rows; ++i)
    for (int j = 0; j < img.cols; ++j)
      if (imgMask.at<cv::Vec3b>(i,j) == cv::Vec3b(0,0,0))
        img.at<cv::Vec3b>(i,j) = cv::Vec3b(0,0,0);
  cv::Mat handBox = padBy50PercentAndAlign16(img(bb));
  HOGDescriptor h(handBox);
  // cv::imwrite("/home/jmkarppa/dippa/hog_example_input.png", handBox);
  // h.visualise(handBox);
  // cv::imwrite("/home/jmkarppa/dippa/hog_example.png", handBox);
  return h;
}

static HOGDescriptor getTestHogBw() {
  cv::Rect bb(cv::Point(109,227),cv::Point(241,376));
  cv::Mat img = cv::imread(TESTIMAGE);
  cv::Mat imgMask = cv::imread(TESTIMAGE_SKINMASK);
  BOOST_REQUIRE(img.size() == imgMask.size() && img.type() == imgMask.type());
  for (int i = 0; i < img.rows; ++i)
    for (int j = 0; j < img.cols; ++j)
      if (imgMask.at<cv::Vec3b>(i,j) == cv::Vec3b(0,0,0))
        img.at<cv::Vec3b>(i,j) = cv::Vec3b(0,0,0);
  cv::Mat handBox = padBy50PercentAndAlign16(img(bb));
  convertColourInPlace(handBox, CV_BGR2GRAY);
  return HOGDescriptor(handBox);
}


BOOST_AUTO_TEST_CASE( testHog ) {
  HOGDescriptor h = getTestHog();

  typedef HOGDescriptor::Block Block;
  typedef HOGDescriptor::Cell Cell;

  // cv::Mat temp = handBox.clone();
  for (int x = 0; x < h.getNBlocksX(); ++x) {
    for (int y = 0; y < h.getNBlocksY(); ++y) {
      Block b = h.getBlock(x,y);
      BOOST_REQUIRE(b.cells.size() == 4);
      cv::Size cellSize = b.cells[0].size;
      BOOST_REQUIRE(b.tl.x == x * cellSize.width);
      BOOST_REQUIRE(b.tl.y == y * cellSize.height);
      BOOST_REQUIRE(b.br.x == b.tl.x+b.size.width);
      BOOST_REQUIRE(b.br.y == b.tl.y+b.size.height);
      for (const Cell& c : b.cells) {
        BOOST_REQUIRE(c.bins.size() == 9);
        BOOST_REQUIRE(c.tl.x < b.br.x);
        BOOST_REQUIRE(c.tl.y < b.br.y);
        BOOST_REQUIRE(c.br.x > b.tl.x);
        BOOST_REQUIRE(c.br.y > b.tl.y);
      }
    }
  }
  BOOST_REQUIRE(h.getDescriptorSize() == (int)h.getRawValues().size());
  std::vector<bool> checkedIndices(h.getDescriptorSize(), false);
  cv::Mat checkedBlocks(h.getNBlocksX(), h.getNBlocksY(), CV_8UC1, cv::Scalar::all(0));
  std::map<std::pair<int,int>,std::vector<bool> > checkedCells;
  for (int x = 0; x < h.getNBlocksX(); ++x) 
    for (int y = 0; y < h.getNBlocksY(); ++y)   
      checkedCells[std::make_pair(x,y)] = std::vector<bool>(4,false);

  typedef HOGDescriptor::ValueIdentifier ValueIdentifier;

  const std::vector<float>& hogVector = h.getRawValues();
  double radsPerBin = math::PI/9.0;
  for (int i = 0; i < (int)hogVector.size(); ++i) {
    ValueIdentifier vi = h.identify(i);
    int x = vi.blockXIdx;
    int y = vi.blockYIdx;
    checkedBlocks.at<uchar>(x, y) = 255;
    checkedCells[std::make_pair(x,y)][vi.cellIdx] = true;
    BOOST_REQUIRE(vi.tl == h.getBlock(x,y).cells[vi.cellIdx].tl);
    // std::cerr << vi.br << " vs " << h.getBlock(x,y).cells[vi.cellIdx].br << std::endl;
    BOOST_REQUIRE(vi.br == h.getBlock(x,y).cells[vi.cellIdx].br);
    BOOST_REQUIRE(vi.binNumber == i % 9);
    BOOST_REQUIRE(vi.centroid == 0.5*cv::Point2d(vi.tl.x+vi.br.x, vi.tl.y + vi.br.y));
    BOOST_REQUIRE(vi.angle == radsPerBin/2 + vi.binNumber * radsPerBin);

    BOOST_REQUIRE(vi.blockXIdx >= 0);
    BOOST_REQUIRE(vi.blockXIdx < h.getNBlocksX());
    BOOST_REQUIRE(vi.blockYIdx >= 0);
    BOOST_REQUIRE(vi.blockYIdx < h.getNBlocksY());
    BOOST_REQUIRE(vi.cellIdx >= 0);
    BOOST_REQUIRE(vi.cellIdx < 4);
    BOOST_REQUIRE(vi.binNumber >= 0);
    BOOST_REQUIRE(vi.binNumber < 9);

    int index = ((vi.blockXIdx*h.getNBlocksY() + vi.blockYIdx)*h.getCellsPerBlock() + 
                 vi.cellIdx)*h.getNBins() + vi.binNumber;
    if (index != i) {
      std::cerr << "i: " << i << std::endl
                << "index: " << index << std::endl;
    }
    BOOST_REQUIRE(index == i);
    if (hogVector[index] == 0 && h.getBlock(x,y).cells[vi.cellIdx].bins[vi.binNumber] > 0) {
      std::cerr << "index: " << index << std::endl
                << "blockXIdx: " << vi.blockXIdx << std::endl
                << "blockYIdx: " << vi.blockYIdx << std::endl
                << "hogVector[" << index << "]: " << hogVector[index] << std::endl
                << "got: " << h.getBlock(x,y).cells[vi.cellIdx].bins[vi.binNumber] << std::endl;
      for (size_t i = 0; i < hogVector.size(); ++i) {
        if (hogVector[i] == h.getBlock(x,y).cells[vi.cellIdx].bins[vi.binNumber]) {
          std::cerr << "Found " << hogVector[i] << " at " << i << std::endl;
          break;
        }
      }
    }
    BOOST_REQUIRE(hogVector[index] == h.getBlock(x,y).cells[vi.cellIdx].bins[vi.binNumber]);
    checkedIndices[index] = true;
  }

  for (int x = 0; x < h.getNBlocksX(); ++x) {
    for (int y = 0; y < h.getNBlocksY(); ++y) {
      BOOST_REQUIRE(checkedBlocks.at<uchar>(x,y) == 255);
      for (int k = 0; k < 4; ++k)
        BOOST_REQUIRE(checkedCells[std::make_pair(x,y)][k] == true);
    }
  }
  for (bool b : checkedIndices)
    BOOST_REQUIRE(b);

  checkedIndices = std::vector<bool> (h.getDescriptorSize(), false);

  for (int x = 0; x < h.getNBlocksX(); ++x) {
    for (int y = 0; y < h.getNBlocksY(); ++y) {
      Block b = h.getBlock(x,y);
      for (int cellIdx = 0; cellIdx < (int)b.cells.size(); ++cellIdx) {
        const Cell& c = b.cells[cellIdx];
        for (size_t k = 0; k < c.bins.size(); ++k) {
          int blockNumber = h.getNBlocksY()*x+y;
          int index = (blockNumber*h.getCellsPerBlock()+cellIdx)*h.getNBins()+k;
          bool b = (c.bins[k] == hogVector[index]);
          checkedIndices[index] = b;
        }
      }
    }
  }

  for (bool b : checkedIndices)
    BOOST_REQUIRE(b);

  HOGDescriptor hBw = getTestHogBw();

  BOOST_REQUIRE(h.getDescriptorSize() == hBw.getDescriptorSize());

  std::vector<float> hogVectorCopy(hogVector);

  HOGDescriptor hNormalised = h.getBlockNormalisedCopy();

  BOOST_REQUIRE(h.getValuesPerBlock() == hNormalised.getValuesPerBlock());
  BOOST_REQUIRE(h.getNBlocksY() == hNormalised.getNBlocksY());
  BOOST_REQUIRE(h.getNBlocksX() == hNormalised.getNBlocksX());
  BOOST_REQUIRE(h.getNBlocks() == hNormalised.getNBlocks());
  BOOST_REQUIRE(h.getCellsPerBlock() == hNormalised.getCellsPerBlock());
  BOOST_REQUIRE(h.getNBins() == hNormalised.getNBins());
  BOOST_REQUIRE(h.getBlockSize() == hNormalised.getBlockSize());
  BOOST_REQUIRE(h.getCellSize() == hNormalised.getCellSize());
  
  std::vector<float>::const_iterator it = hNormalised.getRawValues().cbegin();
  std::vector<float>::const_iterator jt = h.getRawValues().cbegin();
  for (int i = 0; i < h.getNBlocks(); ++i) {
    float sum1 = std::accumulate(it, it + h.getValuesPerBlock(), 0.0);
    float sum2 = std::accumulate(jt, jt + h.getValuesPerBlock(), 0.0);

    BOOST_REQUIRE(sum1 == 0.0 || almostEqual(sum1, 1.0));

    for (int j = 0; j < h.getValuesPerBlock(); ++j) 
      BOOST_REQUIRE((sum2 == 0.0 && sum1 == 0.0) || 
                    (sum2 != 0.0 && sum1 != 0.0 && 
                     almostEqual(*it++, *jt++ / sum2)));
  }
}

static cv::Mat getHogTestImg() {
  cv::Rect bb(cv::Point(109,227),cv::Point(241,376));
  cv::Mat img = cv::imread(TESTIMAGE);
  cv::Mat imgMask = cv::imread(TESTIMAGE_SKINMASK);
  BOOST_REQUIRE(img.size() == imgMask.size() && img.type() == imgMask.type());
  for (int i = 0; i < img.rows; ++i)
    for (int j = 0; j < img.cols; ++j)
      if (imgMask.at<cv::Vec3b>(i,j) == cv::Vec3b(0,0,0))
        img.at<cv::Vec3b>(i,j) = cv::Vec3b(0,0,0);
  cv::Mat handBox = img(bb);
  return handBox;
}

BOOST_AUTO_TEST_CASE( testPhog ) {
  cv::Mat handBox = getHogTestImg();

  for (int B : { 9, 20 }) {
    int nCells = 1;
    for (int L = 1; L < 10; ++L) {
      PHOG ph(handBox, B, L);
      // std::cerr << "B = " << B << ", L = " << L << ", nCells = " << nCells 
      //           << ", nCells*B = " << nCells*B 
      //           << ", ph.getRawValues().size() = " << ph.getRawValues().size()
      //           << std::endl;
      BOOST_REQUIRE((int)ph.getRawValues().size() == nCells*B);
      if (L < 8)
        nCells += 4 << 2*(L-1);

      float sum = 0;
      for (auto it = ph.getRawValues().cbegin(); it != ph.getRawValues().cend(); ++it)
        sum += *it;

      BOOST_REQUIRE(almostEqual(sum, 1.0));

      auto it = ph.getRawValues().cbegin();
      for (int l = 0; l < L; ++l) {
        if (l < 8) {
          auto level = ph.getLevel(l);
          BOOST_REQUIRE((int)level.size() == B*(1 << (2*l)));
          auto jt = level.begin();
          for (int i = 0; i < (int)level.size(); ++i)
            BOOST_REQUIRE(*it++ == *jt++);
          BOOST_REQUIRE(jt == level.end());
        }
        else {
          BOOST_CHECK_THROW(ph.getLevel(l), std::invalid_argument);
        }
      }

      BOOST_REQUIRE(it == ph.getRawValues().cend());
    }
  }
}

typedef HOGDescriptor::ValueIdentifier ValueIdentifier;

BOOST_AUTO_TEST_CASE( testHogDm ) {
  HOGDescriptor h = getTestHog();

  auto generateDistanceMatrixSlow = [&](double w) {
    int N = h.getRawValues().size();
    cv::Mat m(N,N,CV_32FC1,cv::Scalar::all(0));
    std::vector<ValueIdentifier> valis(N);
    for (int i = 0; i < N; ++i)
      valis[i] = h.identify(i);

    for (int i = 0; i < N; ++i) {
      const ValueIdentifier& vii = valis[i];
      for (int j = 0; j < N; ++j) {
        const ValueIdentifier& vij = valis[j];
        float d = cos(vii.angle-vij.angle);
        m.at<float>(i,j) = cv::norm(vii.centroid-vij.centroid) + w * (1.0-d*d);
      }
    }
    return m;
  };

  // timerOn();
  cv::Mat distanceMatrix = generateDistanceMatrixSlow(0);
  // std::cerr << "Generated distance matrix (slow) in " << timerOff() << std::endl;

  int N = distanceMatrix.cols;

  BOOST_REQUIRE(distanceMatrix.rows == N);
  BOOST_REQUIRE(distanceMatrix.type() == CV_32FC1);

  std::vector<HOGDescriptor::ValueIdentifier> valis(h.getRawValues().size());
  for (int i = 0; i < N; ++i)
    valis[i] = h.identify(i);

  for (int i = 0; i < N; ++i) 
    BOOST_REQUIRE(distanceMatrix.at<float>(i,i) == 0.0);

  cv::Mat distanceMatrixT = distanceMatrix.t();
  BOOST_REQUIRE(memEqual(distanceMatrix, distanceMatrixT));
  distanceMatrixT = cv::Mat();

  cv::Mat norms(N, N, CV_32FC1);

  float* normsPtr = norms.ptr<float>();
  for (int i = 0; i < N; ++i) {
    const auto& vii = valis[i];
    for (int j = 0; j < N; ++j) {
      const auto& vij = valis[j];
      *normsPtr++ = cv::norm(vij.centroid - vii.centroid);
    }
  }

  BOOST_REQUIRE(memEqual(norms, distanceMatrix));

  // timerOn();
  cv::Mat distanceMatrix2 = h.generateDistanceMatrix(0);
  // std::cerr << "Generated distance matrix (faster) in " << timerOff() << std::endl;

  BOOST_REQUIRE(distanceMatrix2.type() == distanceMatrix.type());
  BOOST_REQUIRE(distanceMatrix2.size() == distanceMatrix.size());

  BOOST_REQUIRE(memEqual(distanceMatrix, distanceMatrix2));

  distanceMatrix = cv::Mat(); // flush memory
  distanceMatrix2 = cv::Mat();
  distanceMatrix = generateDistanceMatrixSlow(h.getBlockSize().width);
  distanceMatrix2 = h.generateDistanceMatrix(h.getBlockSize().width);

  BOOST_REQUIRE(distanceMatrix2.rows == N);
  BOOST_REQUIRE(distanceMatrix2.type() == CV_32FC1);
  BOOST_REQUIRE(distanceMatrix.type() == distanceMatrix2.type());
  BOOST_REQUIRE(distanceMatrix.size() == distanceMatrix2.size());

  bool accumulated = true;
  float* dm1Ptr = distanceMatrix.ptr<float>();
  float* dm2Ptr = distanceMatrix2.ptr<float>();
  for (int i = 0; i < distanceMatrix.rows; ++i) {
    for (int j = 0; j < distanceMatrix.cols; ++j) {
      // if (*dm1Ptr != *dm2Ptr) {
      //   std::cerr << i << " " << j << " " << *dm1Ptr << " " << *dm2Ptr 
      //             << std::endl;
      // }

      accumulated = almostEqual(*dm1Ptr++, *dm2Ptr++);
    }
  }
  BOOST_REQUIRE(accumulated);

  // BOOST_REQUIRE(memEqual(distanceMatrix, distanceMatrix2));

  distanceMatrix2 = cv::Mat();

  distanceMatrixT = distanceMatrix.t();
  BOOST_REQUIRE(memEqual(distanceMatrix, distanceMatrixT));
  distanceMatrixT = cv::Mat();

  for (int i = 0; i < N; ++i) 
    BOOST_REQUIRE(distanceMatrix.at<float>(i,i) == 0.0);

  std::vector<bool> adPositive(N*N);
  std::vector<bool>::iterator it = adPositive.begin();
  for (int i = 0; i < N; ++i) {
    double angVii = valis[i].angle;
    for (int j = 0; j < N; ++j) {
      const auto& vij = valis[j];
      *it++ = std::abs(vij.angle - angVii) > 0;
    }
  }

  it = adPositive.begin();
  float* ptr = distanceMatrix.ptr<float>();
  normsPtr = norms.ptr<float>();
  accumulated = true;
  for (int i = 0; i < N*N; ++i) {
    float val = *ptr++;
    float norm = *normsPtr++;
    if (*it++)
      accumulated = accumulated && !(almostEqual(val, norm)) && val > norm;
    else
      accumulated = accumulated && almostEqual(val, norm);
  }

  BOOST_REQUIRE(accumulated);
}

static HOGDescriptor getTestHog2() {
  cv::Rect bb(207,187,124,133);
  cv::Mat img = cv::imread(TESTIMAGE2);
  cv::Mat imgMask = cv::imread(TESTIMAGE2_SKINMASK);
  
  BOOST_REQUIRE(img.size() == imgMask.size() && img.type() == imgMask.type());
  for (int i = 0; i < img.rows; ++i)
    for (int j = 0; j < img.cols; ++j)
      if (imgMask.at<cv::Vec3b>(i,j) == cv::Vec3b(0,0,0))
        img.at<cv::Vec3b>(i,j) = cv::Vec3b(0,0,0);
  cv::Mat handBox = padBy50PercentAndAlign16(img(bb));
  return HOGDescriptor(handBox);
}


BOOST_AUTO_TEST_CASE( testHogDm2 ) {
  HOGDescriptor h1 = getTestHog();
  HOGDescriptor h2 = getTestHog2();
  BOOST_REQUIRE(h1.getDescriptorSize() != h2.getDescriptorSize());
  BOOST_REQUIRE(h1.getDescriptorSize() == (int)h1.getRawValues().size());
  BOOST_REQUIRE(h2.getDescriptorSize() == (int)h2.getRawValues().size());
  BOOST_REQUIRE(h1.getDescriptorSize() > 0);
  BOOST_REQUIRE(h2.getDescriptorSize() > 0);

  int N = h1.getDescriptorSize();
  int M = h2.getDescriptorSize();

  std::vector<ValueIdentifier> vals1(N);
  for (int i = 0; i < N; ++i)
    vals1[i] = h1.identify(i);

  std::vector<ValueIdentifier> vals2(M);
  for (int i = 0; i < M; ++i)
      vals2[i] = h2.identify(i);

  cv::Mat dm1 = h1.generateDistanceMatrix(0, h2);
  cv::Mat dm2 = h2.generateDistanceMatrix(0, h1);
  BOOST_REQUIRE(dm1.rows == N);
  BOOST_REQUIRE(dm1.cols == M);
  BOOST_REQUIRE(dm2.rows == dm1.cols);
  BOOST_REQUIRE(dm2.cols == dm1.rows);
  dm2 = dm2.t();

  BOOST_REQUIRE(memEqual(dm1, dm2));
  dm2 = cv::Mat();
  cv::Mat norms(dm1.size(), CV_32FC1);
  float* normsPtr = norms.ptr<float>();
  for (int i = 0; i < dm1.rows; ++i) 
    for (int j = 0; j < dm1.cols; ++j) 
      *normsPtr++ = cv::norm(vals1[i].centroid - vals2[j].centroid);

  float* dm1Ptr = dm1.ptr<float>();
  // for (int i = 0; i < dm1.rows; ++i) {
  //   for (int j = 0; j < dm1.cols; ++j) {
  //     float val = *dm1Ptr++;
  //     float norm = *normsPtr++;
      // if (val != norm) {
      //   std::cerr << "(" << i << "," << j << "): " << val << " " << norm << std::endl;
      // }
  //   }
  // }

  bool accumulate = true;
  normsPtr = norms.ptr<float>();
  dm1Ptr = dm1.ptr<float>();
  for (int i = 0; i < dm1.rows; ++i) 
    for (int j = 0; j < dm1.cols; ++j) 
      accumulate = accumulate && almostEqual(*dm1Ptr++, *normsPtr++);

  BOOST_REQUIRE(accumulate);
  
  dm1 = cv::Mat();
  dm1 = h1.generateDistanceMatrix(h1.getBlockSize().width, h2);

  BOOST_REQUIRE(dm1.rows == N);
  BOOST_REQUIRE(dm1.cols == M);

  accumulate = true;
  dm1Ptr = dm1.ptr<float>();
  normsPtr = norms.ptr<float>();
  for (int i = 0; i < dm1.rows; ++i) {
    for (int j = 0; j < dm1.cols; ++j) {
      double angd = std::abs(vals1[i].angle - vals2[j].angle);
      float val = *dm1Ptr++;
      float norm = *normsPtr++;
      if (angd == 0)
        accumulate = accumulate && almostEqual(val, norm);
      else
        accumulate = accumulate && !almostEqual(val, norm) && val > norm;
    }
  }
  BOOST_REQUIRE(accumulate);

  norms = cv::Mat();
  // dm2 = cv::Mat();
  dm2 = h2.generateDistanceMatrix(h2.getBlockSize().width, h1);

  BOOST_REQUIRE(dm2.rows == M);
  BOOST_REQUIRE(dm2.cols == N);
  
  dm2 = dm2.t();
  
  BOOST_REQUIRE(memEqual(dm1, dm2));

  // for (int i = 0; i < dm1.rows; ++i) {
  //   for (int j = 0; j < dm1.cols; ++j) {
  //     double eud = cv::norm(vals1[i].centroid - vals2[j].centroid);
  //     double angd = std::abs(vals1[i].angle - vals2[j].angle);
  //     if (dm1.at<double>(i,j) != dm2.at<double>(j,i))
  //       std::cerr << "fail at: i = " << i << ", j = " << j << std::endl;
  //     BOOST_REQUIRE(dm1.at<double>(i,j) == dm2.at<double>(j,i));
  //     if (angd == 0)
  //       BOOST_REQUIRE(dm1.at<double>(i,j) == eud);
  //     else {
  //       BOOST_REQUIRE(dm1.at<double>(i,j) > eud);
  //     }
  //   }
  // }
}

//static cv::Mat* dmPtr = nullptr;
static float* dmPtr = nullptr;
static int dmCols = 0;

inline static float dFunc(int* f1, int* f2) {
  return *(dmPtr + dmCols*(*f1) + *f2);
}

BOOST_AUTO_TEST_CASE( testEmd ) {
  int N = rand() % 500 + 100;
  int M = rand() % 500 + 100;
  cv::Mat dm = cv::Mat(N, M, CV_32FC1, cv::Scalar::all(0)); 
  BOOST_REQUIRE(dm.type() == CV_32FC1);
  BOOST_REQUIRE(dm.cols == M);
  BOOST_REQUIRE(dm.rows == N);

  std::vector<cv::Point2i> points1(N);
  std::vector<cv::Point2i> points2(M);
  for (auto it = points1.begin(); it != points1.end(); ++it)
    *it = cv::Point2i(rand() % 200 - 100,
                      rand() % 200 - 100);
  for (auto it = points2.begin(); it != points2.end(); ++it)
    *it = cv::Point2i(rand() % 200 - 100,
                      rand() % 200 - 100);

  for (int i = 0; i < N; ++i)
    for (int j = 0; j < M; ++j)
      dm.at<float>(i,j) = cv::norm(points1[i] - points2[j]);


  dmPtr = dm.ptr<float>();
  dmCols = dm.cols;

  std::vector<int> indices(std::max(N,M));
  for (size_t i = 0; i < indices.size(); ++i)
    indices[i] = i;

  for (int i = 0; i < dm.rows; ++i) {
    for (int j = 0; j < dm.cols; ++j) {
      BOOST_REQUIRE(dm.at<float>(i,j) == dFunc(&indices[i], &indices[j]));
    }
  }


  auto normalize = [](std::vector<float>& vec) {
    float sum = std::accumulate(vec.cbegin(), vec.cend(), 0.0);
    for (size_t i = 0; i < vec.size(); ++i)
      vec[i] /= sum;
  };

  std::vector<float> weights1 = random::unifrnd<float>(0.0, 1.0, N);
  std::vector<float> weights2 = random::unifrnd<float>(0.0, 1.0, M);
  normalize(weights1);
  normalize(weights2);

  BOOST_REQUIRE(weights1.size() != weights2.size());
  BOOST_REQUIRE((int)weights1.size() == N);
  BOOST_REQUIRE((int)weights2.size() == M);
  BOOST_REQUIRE(dm.rows == N);
  BOOST_REQUIRE(dm.cols == M);
  BOOST_REQUIRE(almostEqual(1.0, std::accumulate(weights1.cbegin(), weights1.cend(), 0.0)));
  BOOST_REQUIRE(almostEqual(1.0, std::accumulate(weights2.cbegin(), weights2.cend(), 0.0)));

  float totalFlow = 0;
  float totalCost = 0;

#ifdef SLMOTION_WITH_RUBNER_EMD
  signature_t sig1 {
    N, &indices[0], &weights1[0]
  };
  signature_t sig2 {
    M, &indices[0], &weights2[0]
  };

  std::vector<flow_t> flow(N+M-1);
  int flowSize = 0;

  // timerOn();
  float emdValueRubner = emd(&sig1, &sig2, dFunc, &flow[0], &flowSize);
  // std::cerr << "Computed emd (Rubner) in " << timerOff() << ": " << emdValueRubner << std::endl;
  for (const flow_t& f : flow) {
    totalFlow += f.amount;
    totalCost += dm.at<float>(f.from, f.to) * f.amount;
  }
  BOOST_REQUIRE(almostEqual(totalFlow, 1.0));
  BOOST_REQUIRE(almostEqual(totalCost, emdValueRubner));
#endif // SLMOTION_WITH_RUBNER_EMD

  cv::Mat flowMat(weights1.size(), weights2.size(), CV_32FC1, cv::Scalar::all(0));
  cv::Mat weights1Mat(weights1.size(), 1, CV_32FC1, &weights1[0]);
  cv::Mat weights2Mat(weights2.size(), 1, CV_32FC1, &weights2[0]);

  for (int i = 0; i < N; ++i)
    BOOST_REQUIRE(weights1Mat.at<float>(i) == weights1[i]);

  for (int i = 0; i < M; ++i)
    BOOST_REQUIRE(weights2Mat.at<float>(i) == weights2[i]);  

  // timerOn();
  float emdValueOpenCv = cv::EMD(weights1Mat, weights2Mat, CV_DIST_USER, dm, nullptr, flowMat);
  // std::cerr << "Computed emd (OpenCV) in " << timerOff() << ": " 
  //           << emdValueOpenCv << std::endl;
  BOOST_REQUIRE(flowMat.type() == CV_32FC1);
  BOOST_REQUIRE(flowMat.cols == dm.cols);
  BOOST_REQUIRE(flowMat.rows == dm.rows);

  totalFlow = 0;
  totalCost = 0;
  for (int i = 0; i < flowMat.rows; ++i) {
    for (int j = 0; j < flowMat.cols; ++j) {
      totalFlow += flowMat.at<float>(i,j);
      totalCost += flowMat.at<float>(i,j) * dm.at<float>(i,j);
    }
  }
  BOOST_REQUIRE(almostEqual(totalFlow, 1.0));
  BOOST_REQUIRE(almostEqual(totalCost, emdValueOpenCv));

#ifdef SLMOTION_WITH_RUBNER_EMD
  BOOST_REQUIRE(almostEqual(emdValueOpenCv, emdValueRubner, 1e-2));
#endif // SLMOTION_WITH_RUBNER_EMD

#if 0
  std::cerr << "N = " << N << ", M = " << M << std::endl;

  // timerOn();
  // emdValue = emd(&sig1, &sig2, dFunc, &flow[0], &flowSize);
  // std::cerr << "Computed emd (Rubner) in " << timerOff() << ": " << emdValue << std::endl;

  // timerOn();
  float emdValueOpenCv = cv::EMD(weights1Mat, weights2Mat, CV_DIST_USER, dm, nullptr, flowMat);
  // std::cerr << "Computed emd (OpenCV) in " << timerOff() << ": " 
  //           << emdValueOpenCv << std::endl;

  // timerOn();
  float emdValue = emdGreedy(weights1, weights2, dm);
  // std::cerr << "Computed emd (naïve) in " << timerOff() << ": "
  //           << emdValueOpenCv << std::endl;
  std::cerr << emdValue << " " << emdValueOpenCv << std::endl;
  BOOST_REQUIRE(almostEqual(emdValue, emdValueOpenCv));


  std::vector<int> rndCounts = random::unifrnd<int>(0, 50, N);
  std::vector<double> weights1d(rndCounts.cbegin(), rndCounts.cend());
  rndCounts = random::unifrnd<int>(0, 50, N);
  std::vector<double> weights2d(rndCounts.cbegin(), rndCounts.cend());

  double sum1 = std::accumulate(weights1d.cbegin(), weights1d.cend(), 0.0);
  double sum2 = std::accumulate(weights2d.cbegin(), weights2d.cend(), 0.0);
  for (int i = 0; i < N; ++i) {
    weights1d[i] /= sum1;
    weights2d[i] /= sum2;
  }

  weights1 = std::vector<float>(weights1d.cbegin(), weights1d.cend());
  weights2 = std::vector<float>(weights2d.cbegin(), weights2d.cend());

  points2 = std::vector<cv::Point2i>(N);
  for (int i = 0; i < N; ++i)
    points2[i] = cv::Point2i(rand() % 200 - 100,
                             rand() % 200 - 100);

  dm = cv::Mat(N, N, CV_32FC1);
  for (int i = 0; i < N; ++i)
    for (int j = 0; j < N; ++j)
      dm.at<float>(i,j) = cv::norm(points1[i] - points2[j]);

  // dm = dm.t();
  // emd_hat_gd_metric<double> peleEmd;
  // std::vector<std::vector<double> > C;
  // for (int i = 0; i < dm.rows; ++i) {
  //   std::vector<double> row(dm.cols);
  //   double* ptr = dm.ptr<double>(i);
  //   for (int j = 0; j < dm.cols; ++j)
  //     row[j] = *ptr++;
  //   C.push_back(row);
  // }

  // for (int i = 0; i < dm.rows; ++i) {
  //   for (int j = 0; j < dm.cols; ++j) {
  //     BOOST_REQUIRE(dm.at<double>(i,j) == C[i][j]);
  //   }
  // }

  // dm = dm.t();

  // sig2.n = N;
  // sig2.Features = sig1.Features;

  weights2Mat = cv::Mat(weights2.size(), 1, CV_32FC1, &weights2[0]);

  // wptr = weights1Mat.ptr<float>();
  // for (auto it = weights1.cbegin(); it != weights1.cend(); ++it)
  //   *wptr++ = *it;
  // wptr = weights2Mat.ptr<float>();
  // for (auto it = weights2.cbegin(); it != weights2.cend(); ++it)
  //   *wptr++ = *it;

  BOOST_REQUIRE((int)weights1.size() == N);
  BOOST_REQUIRE((int)weights2.size() == N);
  BOOST_REQUIRE(weights1Mat.cols == 1);
  BOOST_REQUIRE(weights1Mat.rows == N);
  BOOST_REQUIRE(weights2Mat.cols == 1);
  BOOST_REQUIRE(weights2Mat.rows == N);
  BOOST_REQUIRE(dm.size() == cv::Size(N,N));

  // timerOn();
  // emdValue = emd(&sig1, &sig2, dFunc, &flow[0], &flowSize);
  // std::cerr << "Computed emd (Rubner) in " << timerOff() << ": " << emdValue << std::endl;

  // timerOn();
  emdValueOpenCv = cv::EMD(weights1Mat, weights2Mat, CV_DIST_USER, dm, nullptr, flowMat);
  // std::cerr << "Computed emd (OpenCV) in " << timerOff() << ": " 
  //           << emdValueOpenCv << std::endl;

  // timerOn();
  emdValue = emdGreedy(weights1, weights2, dm);
  // std::cerr << "Computed emd (naïve) in " << timerOff() << ": "
  //           << emdValueOpenCv << std::endl;

  BOOST_REQUIRE(almostEqual(emdValue, emdValueOpenCv));

  // timerOn();
  // double emdValuePele = peleEmd(weights1d, weights2d, C);
  // std::cerr << "Computed emd (pele) in " << timerOff() << ": "
  //           << emdValuePele << std::endl;
#endif
}



static void normalize(std::vector<float>& vec) {
  float sum = std::accumulate(vec.cbegin(), vec.cend(), 0.0);
  for (size_t i = 0; i < vec.size(); ++i)
    vec[i] /= sum;
}

static void normalize(cv::Mat& vec) {
  assert(vec.type() == CV_32FC1);
  assert(vec.rows > 1);
  assert(vec.cols == 1);
  cv::Mat temp;
  cv::reduce(vec, temp, 0, CV_REDUCE_SUM);
  assert(temp.rows == 1);
  assert(temp.cols == 1);
  assert(temp.type() == CV_32FC1);
  float sum = temp.at<float>(0);
  for (int i = 0; i < vec.rows; ++i)
    vec.at<float>(i) /= sum;
}

BOOST_AUTO_TEST_CASE( testEmd2 ) {
  for (int ccc = 0; ccc < 100; ++ccc) {
    int N, M;
    do {
      N = 1 + rand() % 20;
      M = 1 + rand() % 20;
    }
    while (N == M);
    cv::Mat dm = cv::Mat(N, M, CV_32FC1, cv::Scalar::all(0)); 
    BOOST_REQUIRE(dm.type() == CV_32FC1);
    BOOST_REQUIRE(dm.cols == M);
    BOOST_REQUIRE(dm.rows == N);

    std::vector<cv::Point2i> points1(N);
    std::vector<cv::Point2i> points2(M);
    for (auto it = points1.begin(); it != points1.end(); ++it)
      *it = cv::Point2i(rand() % 200 - 100,
                        rand() % 200 - 100);
    for (auto it = points2.begin(); it != points2.end(); ++it)
      *it = cv::Point2i(rand() % 200 - 100,
                        rand() % 200 - 100);

    for (int i = 0; i < N; ++i)
      for (int j = 0; j < M; ++j)
        dm.at<float>(i,j) = cv::norm(points1[i] - points2[j]);

    dmPtr = dm.ptr<float>();
    dmCols = dm.cols;

    std::vector<int> indices(std::max(N,M));
    for (size_t i = 0; i < indices.size(); ++i)
      indices[i] = i;

    std::vector<float> weights1 = random::unifrnd<float>(0.0, 1.0, N);
    std::vector<float> weights2 = random::unifrnd<float>(0.0, 1.0, M);
    normalize(weights1);
    normalize(weights2);

    BOOST_REQUIRE(weights1.size() != weights2.size());
    BOOST_REQUIRE((int)weights1.size() == N);
    BOOST_REQUIRE((int)weights2.size() == M);
    BOOST_REQUIRE(dm.rows == N);
    BOOST_REQUIRE(dm.cols == M);
    BOOST_REQUIRE(almostEqual(1.0, std::accumulate(weights1.cbegin(), weights1.cend(), 0.0)));
    BOOST_REQUIRE(almostEqual(1.0, std::accumulate(weights2.cbegin(), weights2.cend(), 0.0)));

    float totalFlow = 0;
    float totalCost = 0;

#ifdef SLMOTION_WITH_RUBNER_EMD
    signature_t sig1 {
      N, &indices[0], &weights1[0]
        };
    signature_t sig2 {
      M, &indices[0], &weights2[0]
        };

    std::vector<flow_t> flow(N+M-1);
    int flowSize = 0;

    // timerOn();
    float emdValueRubner = emd(&sig1, &sig2, dFunc, &flow[0], &flowSize);
    // std::cerr << "Computed emd (Rubner) in " << timerOff() << ": " << emdValueRubner << std::endl;
    for (const flow_t& f : flow) {
      totalFlow += f.amount;
      totalCost += dm.at<float>(f.from, f.to) * f.amount;
    }
    BOOST_REQUIRE(almostEqual(totalFlow, 1.0));
    BOOST_REQUIRE(almostEqual(totalCost, emdValueRubner));
#endif // SLMOTION_WITH_RUBNER_EMD

    cv::Mat flowMat(weights1.size(), weights2.size(), CV_32FC1, cv::Scalar::all(0));
    cv::Mat weights1Mat(weights1.size(), 1, CV_32FC1, &weights1[0]);
    cv::Mat weights2Mat(weights2.size(), 1, CV_32FC1, &weights2[0]);
    
    for (int i = 0; i < N; ++i)
      BOOST_REQUIRE(weights1Mat.at<float>(i) == weights1[i]);
    
    for (int i = 0; i < M; ++i)
      BOOST_REQUIRE(weights2Mat.at<float>(i) == weights2[i]);  
    
    // timerOn();
    float emdValueOpenCv = cv::EMD(weights1Mat, weights2Mat, CV_DIST_USER, dm, nullptr, flowMat);
    // std::cerr << "Computed emd (OpenCV) in " << timerOff() << ": " 
    //           << emdValueOpenCv << std::endl;
    BOOST_REQUIRE(flowMat.type() == CV_32FC1);
    BOOST_REQUIRE(flowMat.cols == dm.cols);
    BOOST_REQUIRE(flowMat.rows == dm.rows);
    
    totalFlow = 0;
    totalCost = 0;
    for (int i = 0; i < flowMat.rows; ++i) {
      for (int j = 0; j < flowMat.cols; ++j) {
        totalFlow += flowMat.at<float>(i,j);
        totalCost += flowMat.at<float>(i,j) * dm.at<float>(i,j);
      }
    }
    BOOST_REQUIRE(almostEqual(totalFlow, 1.0));
    BOOST_REQUIRE(almostEqual(totalCost, emdValueOpenCv));

#ifdef SLMOTION_WITH_RUBNER_EMD  
    BOOST_REQUIRE(almostEqual(emdValueOpenCv, emdValueRubner));
#endif // SLMOTION_WITH_RUBNER_EMD
  }
}



static std::vector< std::vector<double> > dmToC(const cv::Mat& dm) {
  std::vector< std::vector<double> > C;
  for (int i = 0; i < dm.rows; ++i) {
    std::vector<double> row(dm.cols);
    for (int j = 0; j < dm.cols; ++j)
      row[j] = dm.at<float>(i,j);
    C.push_back(row);
  }
  return C;
}



BOOST_AUTO_TEST_CASE( testEmd3 ) {
  int N = 1;
  std::vector<float> weights1 { 1.0 };
  std::vector<float> weights2 { 1.0 };
  cv::Mat dm(1, 1, CV_32FC1, cv::Scalar::all(1e-10));

  auto C = dmToC(dm);

  // auto printC = [](const std::vector<std::vector<double> >& C) {
  //   for (const std::vector<double>& row : C) {
  //     for (double d : row)
  //       std::cerr << d << " ";
  //     std::cerr << std::endl;
  //   }
  // };

  // printC(C);

  std::vector<double> P(weights1.cbegin(), weights1.cend());
  std::vector<double> Q(weights2.cbegin(), weights2.cend());

#ifdef SLMOTION_WITH_PELE_FASTEMD
  // timerOn();
  double emdValuePele = emd_hat<double>()(P, Q, C, 0.0);

  // std::cerr << "Computed emd (Pele) in " << timerOff() << ": " 
  //           << emdValuePele << std::endl;

  BOOST_REQUIRE(almostEqual(emdValuePele, 1.0e-10));

  for (double d : random::unifrnd<double>(1.0, 50.0, 50)) {
    dm.at<float>(0,0) = d;
    C = dmToC(dm);
    emdValuePele = emd_hat<double>()(P, Q, C);
    BOOST_REQUIRE(almostEqual(emdValuePele, d));
  }
#endif // SLMOTION_WITH_PELE_FASTEMD

  auto generateDm = [](int N)->cv::Mat {
    std::vector<cv::Point2i> points(N);
    for (auto it = points.begin(); it != points.end(); ++it)
      *it = cv::Point2i(rand() % 200 - 100, rand() % 200 - 100);

    cv::Mat dm(N, N, CV_32FC1);
    for (int i = 0; i < N; ++i)
      for (int j = 0; j < N; ++j)
        dm.at<float>(i,j) = cv::norm(points[i] - points[j]);

    return dm;
  };

  const int MAX_N = 1000;
  std::vector<int> indices(MAX_N);
  for (size_t i = 0; i < indices.size(); ++i)
    indices[i] = i;

  for (N = (0x2 | (rand() & 0x1)); N < MAX_N; N = (N << 1) | (rand() & 0x1)) {
    dm = generateDm(N);
    BOOST_REQUIRE(dm.cols == N);
    BOOST_REQUIRE(dm.rows == N);
    BOOST_REQUIRE(dm.type() == CV_32FC1);
    
    for (int i = 0; i < N; ++i) 
      BOOST_REQUIRE(dm.at<float>(i,i) == 0.0);
    
    for (int i = 0; i < dm.rows; ++i) {
      for (int j = 0; j < dm.cols; ++j) {
        BOOST_REQUIRE(dm.at<float>(i,j) == dm.at<float>(j,i));
      }
    }
    
    BOOST_REQUIRE(memEqual(dm, dm.t()));
    weights1 = random::unifrnd<float>(0.0, 1.0, N);
    weights2 = random::unifrnd<float>(0.0, 1.0, N);
    normalize(weights1);
    normalize(weights2);
    
    C = dmToC(dm);
    P = std::vector<double>(weights1.cbegin(), weights1.cend());
    Q = std::vector<double>(weights2.cbegin(), weights2.cend());
    
#ifdef SLMOTION_WITH_PELE_FASTEMD
    // timerOn();
    emdValuePele = emd_hat<double>()(P, Q, C);
#endif // SLMOTION_WITH_PELE_FASTEMD

#ifdef SLMOTION_WITH_RUBNER_EMD 
    signature_t sig1 {
      N, &indices[0], &weights1[0]
    };

    signature_t sig2 {
      N, &indices[0], &weights2[0]
    };

    dmPtr = dm.ptr<float>();
    dmCols = dm.cols;

    // timerOn();
    float emdValueRubner = emd(&sig1, &sig2, dFunc, nullptr, nullptr);
#endif // SLMOTION_WITH_RUBNER_EMD 
 
    cv::Mat weights1Mat(N, 1, CV_32FC1, &weights1[0]);
    cv::Mat weights2Mat(N, 1, CV_32FC1, &weights2[0]);

    // timerOn();
#ifdef SLMOTION_WITH_PELE_FASTEMD
    float emdValueOpenCv = cv::EMD(weights1Mat, weights2Mat, CV_DIST_USER, dm, nullptr);
 
    double epsilon = pow(1, -((1000-N)/300.0));

    BOOST_REQUIRE(almostEqual(emdValueOpenCv, emdValuePele, epsilon));
#ifdef SLMOTION_WITH_RUBNER_EMD
    BOOST_REQUIRE(almostEqual(emdValueRubner, emdValuePele, epsilon));
#endif // SLMOTION_WITH_RUBNER_EMD
#endif // SLMOTION_WITH_PELE_FASTEDM
  }
}



BOOST_AUTO_TEST_CASE( testEmd4 ) {
  for (int ctr = 0; ctr < 100; ++ctr) {
    int N = 2 + rand() % 100;
    int M = 2 + rand() % 100;

    std::vector<cv::Point2i> points1(N);
    std::vector<cv::Point2i> points2(M);

    for (int i = 0; i < N; ++i)
      points1[i] = cv::Point2i(rand() % 200 - 100,
                               rand() % 200 - 100);
    
    for (int i = 0; i < M; ++i)
      points2[i] = cv::Point2i(rand() % 200 - 100,
                               rand() % 200 - 100);  

    cv::Mat dm(N, M, CV_32FC1);
    for (int i = 0; i < N; ++i)
      for (int j = 0; j < M; ++j)
        dm.at<float>(i,j) = cv::norm(points1[i] - points2[j]);

    std::vector<float> weights1 = random::unifrnd<float>(0, 1, N);
    std::vector<float> weights2 = random::unifrnd<float>(0, 1, M);

    cv::Mat P(N, 1, CV_32FC1);
    cv::Mat Q(M, 1, CV_32FC1);

    for (int i = 0; i < N; ++i)
      P.at<float>(i) = weights1[i];

    for (int i = 0; i < M; ++i)
      Q.at<float>(i) = weights2[i];

    P /= math::sum<float>(P);
    Q /= math::sum<float>(Q);

    double emdValue1 = cv::EMD(P, Q, CV_DIST_USER, dm);
    
    cv::Mat P2(N, 3, CV_32FC1);
    cv::Mat Q2(M, 3, CV_32FC1);

    for (int i = 0; i < N; ++i) {
      P2.at<float>(i, 0) = P.at<float>(i);
      P2.at<float>(i, 1) = points1[i].x;
      P2.at<float>(i, 2) = points1[i].y;
    }

    for (int i = 0; i < M; ++i) {
      Q2.at<float>(i, 0) = Q.at<float>(i);
      Q2.at<float>(i, 1) = points2[i].x;
      Q2.at<float>(i, 2) = points2[i].y;
    }
    
    double emdValue2 = cv::EMD(P2, Q2, CV_DIST_L2);

    BOOST_REQUIRE(almostEqual(emdValue1, emdValue2));
  }
}



static cv::Mat generateDm(const std::vector<cv::Point2i>& points1,
                          const std::vector<cv::Point2i>& points2,
                          int type) {
  assert(type == CV_32FC1 || type == CV_64FC1);
  cv::Mat dm(points1.size(), points2.size(), type);
  for (int i = 0; i < dm.rows; ++i) {
    for (int j = 0; j < dm.cols; ++j) {
      double distance = cv::norm(points1[i] - points2[j]);
      if (type == CV_32FC1)
        dm.at<float>(i,j) = distance;
      else if (type == CV_64FC1)
        dm.at<double>(i,j) = distance;
    }
  }
  return dm;
}

static cv::Mat generateDm(int N, int M, int type) {
  assert(type == CV_32FC1 || type == CV_64FC1);
  std::vector<cv::Point2i> points1(N);
  std::vector<cv::Point2i> points2(M);
  for (int i = 0; i < N; ++i)
    points1[i] = cv::Point2i(rand() % 200 - 100,
                             rand() % 200 - 100);
  for (int i = 0; i < M; ++i)
    points2[i] = cv::Point2i(rand() % 200 - 100,
                             rand() % 200 - 100);

  cv::Mat dm(N, M, type);
  for (int i = 0; i < dm.rows; ++i) {
    for (int j = 0; j < dm.cols; ++j) {
      double distance = cv::norm(points1[i] - points2[j]);
      if (type == CV_32FC1)
        dm.at<float>(i,j) = distance;
      else if (type == CV_64FC1)
        dm.at<double>(i,j) = distance;
    }
  }
  return dm;
}



static cv::Mat generateSymmetricDm(int N, int type) {
  assert(type == CV_32FC1 || type == CV_64FC1);
  std::vector<cv::Point2i> points(N);
  for (int i = 0; i < N; ++i)
    points[i] = cv::Point2i(rand() % 200 - 100,
                             rand() % 200 - 100);
  cv::Mat dm(N, N, type);
  for (int i = 0; i < dm.rows; ++i) {
    for (int j = 0; j < dm.cols; ++j) {
      double distance = cv::norm(points[i] - points[j]);
      if (type == CV_32FC1)
        dm.at<float>(i,j) = distance;
      else if (type == CV_64FC1)
        dm.at<double>(i,j) = distance;
    }
  }
  return dm;
}



BOOST_AUTO_TEST_CASE( testEmdHat ) {
  int N, M;
  cv::Mat P, Q, dm;

  N = M = 1;
  P = Q = cv::Mat::ones(1, 1, CV_32FC1);

  float dist = cv::norm(cv::Point2i(rand() % 200 - 100, 
                                    rand() % 200 - 100) -
                        cv::Point2i(rand() % 200 - 100, 
                                    rand() % 200 - 100));

  dm = cv::Mat(1, 1, CV_32FC1, cv::Scalar::all(dist));
  BOOST_REQUIRE(almostEqual(dm.at<float>(0,0), dist));

  BOOST_REQUIRE(almostEqual(cv::EMD(P, Q, CV_DIST_USER, dm), dist));
  BOOST_REQUIRE(almostEqual(emdHat<float>(P, Q, dm), dist));

  N = 2;
  P = random::unifrnd<float>(0, 1, N, 1);
  normalize(P);

  float sum = 0;
  for (int i = 0; i < P.rows; ++i)
    sum += P.at<float>(i);
  BOOST_REQUIRE(almostEqual(sum, 1.0));
  
  dm = generateDm(N, M, CV_32FC1);

  float emdValue = cv::EMD(P, Q, CV_DIST_USER, dm);
  float emdHatValue = emdHat<float>(P, Q, dm);
  BOOST_REQUIRE(almostEqual(emdValue, emdHatValue));

  std::swap(N, M);
  std::swap(P, Q);
  dm = dm.t();
  BOOST_REQUIRE(almostEqual(cv::EMD(P, Q, CV_DIST_USER, dm),
                            emdHat<float>(P, Q, dm)));

  std::vector<int> Ns = random::unifrnd<int>(2, 50, 30);
  std::vector<int> Ms = random::unifrnd<int>(2, 50, 30);
  for (auto it = Ns.cbegin(), jt = Ms.cbegin();
       it != Ns.cend(); ++it, ++jt) {
    N = *it;
    M = *jt;
    dm = generateDm(N, M, CV_32FC1);
    P = random::unifrnd<float>(0, 1, N, 1);
    Q = random::unifrnd<float>(0, 1, M, 1);
    normalize(P);
    normalize(Q);
    BOOST_REQUIRE(almostEqual(cv::EMD(P, Q, CV_DIST_USER, dm),
                              emdHat<float>(P, Q, dm)));

    P = random::unifrnd<float>(0, 1, N, 1);
    Q = random::unifrnd<float>(0, 1, M, 1);
    float sumP = math::sum<float>(P);
    float sumQ = math::sum<float>(Q);

    float sumPvariant = 0;
    for (int i = 0; i < P.rows; ++i)
      sumPvariant += P.at<float>(i);
    float sumQvariant = 0;
    for (int i = 0; i < Q.rows; ++i)
      sumQvariant += Q.at<float>(i);
    BOOST_REQUIRE(almostEqual(sumPvariant, sumP));
    BOOST_REQUIRE(almostEqual(sumQvariant, sumQ));

    float minSum = std::min(sumP, sumQ);
    if (sumP < sumQ) 
      Q = Q * (sumP / sumQ);    
    else 
      P = P * (sumQ / sumP);    

    BOOST_REQUIRE(almostEqual(math::sum<float>(P), math::sum<float>(Q)));
    BOOST_REQUIRE(almostEqual(math::sum<float>(P), minSum));

    BOOST_REQUIRE(almostEqual(cv::EMD(P, Q, CV_DIST_USER, dm) * minSum,
                              emdHat<float>(P,Q,dm)));
                              
    P = random::unifrnd<float>(0, 1, N, 1);
    Q = random::unifrnd<float>(0, 1, M, 1);
    sumP = math::sum<float>(P);
    sumQ = math::sum<float>(Q);
    float sumDiff = std::abs(sumP - sumQ);
    minSum = std::min(sumP, sumQ);

    double maxD = math::max(dm);
    BOOST_REQUIRE(almostEqual(cv::EMD(P, Q, CV_DIST_USER, dm) * 
                              minSum + maxD * sumDiff,
                              emdHat<float>(P, Q, dm)));
    
  }


#ifdef SLMOTION_WITH_PELE_FASTEMD

  for (auto it = Ns.cbegin(); it != Ns.cend(); ++it) {
    N = *it;
    dm = generateSymmetricDm(N, CV_32FC1);
    std::vector< std::vector<double> > C = dmToC(dm);
    std::vector<double> Pvec(N);
    std::vector<double> Qvec(N);
    P = random::unifrnd<float>(0, 1, N, 1);
    Q = random::unifrnd<float>(0, 1, N, 1);

    float sumP = math::sum<float>(P);
    float sumQ = math::sum<float>(Q);
    // if (sumP < sumQ) {
    //   std::swap(P, Q);
    //   std::swap(sumP, sumQ);
    // }

    //   Q *= (sumP/sumQ);
    // else
    //   P *= (sumQ/sumP);

    for (int i = 0; i < N; ++i) {
      Pvec[i] = P.at<float>(i);
      Qvec[i] = Q.at<float>(i);
    }

    BOOST_REQUIRE(P.rows == (int)Pvec.size());
    BOOST_REQUIRE(Q.rows == (int)Qvec.size());
    BOOST_REQUIRE(Qvec.size() == Pvec.size());
    BOOST_REQUIRE(Q.cols == 1);
    BOOST_REQUIRE(P.cols == 1);
    BOOST_REQUIRE(C.size() == Pvec.size());
    for (int i = 0; i < P.rows; ++i) {
      BOOST_REQUIRE(P.at<float>(i,0) == Pvec[i]);
      BOOST_REQUIRE(Q.at<float>(i,0) == Qvec[i]);
      BOOST_REQUIRE(C[i].size() == Pvec.size());
      for (int j = 0; j < Q.rows; ++j) {
        BOOST_REQUIRE(C[i][j] == dm.at<float>(i,j));
      } 
    }

    float emdHatValue = emdHat<float>(P, Q, dm);
    float emdHatPenalty = emdHatValue - emdHat<float>(P, Q, dm, 0.0);
    float emdValue = (emdHatValue - emdHatPenalty) / std::min(sumP, sumQ);

    float emdHatValuePele = emd_hat<double>()(Pvec, Qvec, C);
    float emdHatPenaltyPele = emdHatValuePele - emd_hat<double>()(Pvec, Qvec, C, 0.0);
    float emdValuePele = (emdHatValuePele - emdHatPenaltyPele) / std::max(sumP, sumQ);

    
    BOOST_REQUIRE(almostEqual(emdHatPenalty, emdHatPenaltyPele, 1e-4));
    BOOST_REQUIRE(almostEqual(emdValue, emdValuePele, 1e-4));
    BOOST_REQUIRE(almostEqual(emdHatValue, emdValuePele*std::min(sumP, sumQ) + emdHatPenaltyPele, 1e-4));
    BOOST_REQUIRE(almostEqual(emdValue*std::max(sumP,sumQ) + emdHatPenalty, emdHatValuePele, 1e-4));
  }


  // for (N = 10; N < 1000; N += 10) {
  //   dm = generateSymmetricDm(N, CV_32FC1);
  //   std::vector< std::vector<double> > C = dmToC(dm);
  //   std::vector<double> Pvec(N);
  //   std::vector<double> Qvec(N);
  //   P = random::unifrnd<float>(0, 1, N, 1);
  //   Q = random::unifrnd<float>(0, 1, N, 1);

  //   for (int i = 0; i < N; ++i) {
  //     Pvec[i] = P.at<float>(i);
  //     Qvec[i] = Q.at<float>(i);
  //   }

  //   timerOn();
  //   emdHat<float>(P, Q, dm);
  //   std::cout << "OpenCV\t" << N << '\t' << timerOff() << std::endl;

  //   timerOn();
  //   emd_hat<double>()(Pvec, Qvec, C);
  //   std::cout << "Pele\t" << N << '\t' << timerOff() << std::endl;
  // }
#endif // SLMOTION_WITH_PELE_FASTEMD
}



BOOST_AUTO_TEST_CASE( testTrimmedHogDm ) {
  HOGDescriptor h1 = getTestHog();
  HOGDescriptor h2 = getTestHog2();
  BOOST_REQUIRE(h1.getDescriptorSize() != h2.getDescriptorSize());
  BOOST_REQUIRE(h1.getDescriptorSize() == (int)h1.getRawValues().size());
  BOOST_REQUIRE(h2.getDescriptorSize() == (int)h2.getRawValues().size());
  BOOST_REQUIRE(h1.getDescriptorSize() > 0);
  BOOST_REQUIRE(h2.getDescriptorSize() > 0);

  int N = h1.getDescriptorSize();
  int M = h2.getDescriptorSize();

  cv::Mat dmTrimmed;
  cv::Mat h1Weights, h2Weights;
  //  timerOn();
  h1.generateTrimmedDistanceMatrix(0.0, 1.1, INT_MAX, h2, dmTrimmed, h1Weights, h2Weights);
  //std::cerr << "Generated trimmed distance matrix in " << timerOff() << std::endl;

  //timerOn();
  cv::Mat dmRegular = h1.generateDistanceMatrix(0.0, h2);
  //std::cerr << "Generated regular distance matrix in " << timerOff() << std::endl;

  BOOST_REQUIRE(dmTrimmed.size() == dmRegular.size());
  BOOST_REQUIRE(dmTrimmed.cols == M);
  BOOST_REQUIRE(dmTrimmed.rows == N);
  BOOST_REQUIRE(h1Weights.rows == N);
  BOOST_REQUIRE(h2Weights.rows == M);
  BOOST_REQUIRE(h1Weights.cols == 1);
  BOOST_REQUIRE(h2Weights.cols == 1);
  BOOST_REQUIRE(dmTrimmed.type() == CV_32FC1);
  BOOST_REQUIRE(dmRegular.type() == CV_32FC1);

  BOOST_REQUIRE(memEqual(dmTrimmed, dmRegular));
  for (int i = 0; i < N; ++i)
    BOOST_REQUIRE(h1Weights.at<float>(i) == h1.getRawValues()[i]);
  for (int i = 0; i < M; ++i)
    BOOST_REQUIRE(h2Weights.at<float>(i) == h2.getRawValues()[i]);

  std::vector<int> indices1;
  std::vector<int> indices2;
  dmTrimmed = cv::Mat();
  h1.generateTrimmedDistanceMatrix(0.0, 0.9999, INT_MAX, h2, dmTrimmed, h1Weights, 
                                   h2Weights, &indices1, &indices2);
  BOOST_REQUIRE(h1Weights.rows < h1.getDescriptorSize());
  BOOST_REQUIRE(h2Weights.rows < h2.getDescriptorSize());  
  BOOST_REQUIRE(h1Weights.rows > 1);
  BOOST_REQUIRE(h2Weights.rows > 1);
  BOOST_REQUIRE(h1Weights.cols == 1);
  BOOST_REQUIRE(h2Weights.cols == 1);
  // BOOST_REQUIRE(std::all_of(h1Weights.cbegin(), h1Weights.cend(), [](float f) { return f > 0; }));
  // BOOST_REQUIRE(std::all_of(h2Weights.cbegin(), h2Weights.cend(), [](float f) { return f > 0; }));
  float* p = h1Weights.ptr<float>();
  bool b = true;
  for (int i = 0; i < h1Weights.rows; ++i)
    b = b && *p++ > 0;
  BOOST_REQUIRE(b);

  p = h2Weights.ptr<float>();
  b = true;
  for (int i = 0; i < h2Weights.rows; ++i)
    b = b && *p++ > 0;
  BOOST_REQUIRE(b);
  
  const std::vector<float>& h1RawValues = h1.getRawValues();
  BOOST_REQUIRE(h1RawValues[indices1[0]] == h1Weights.at<float>(0));
  for (size_t i = 1; i < indices1.size(); ++i) {
    BOOST_REQUIRE(indices1[i-1] < indices1[i]);
    BOOST_REQUIRE(h1RawValues[indices1[i]] == h1Weights.at<float>(i));
  }
  std::set<int> indexSet;
  for (int i : indices1)
    indexSet.insert(i);
  BOOST_REQUIRE((int)indexSet.size() == h1Weights.rows);

  const std::vector<float>& h2RawValues = h2.getRawValues();
  BOOST_REQUIRE(h2RawValues[indices2[0]] == h2Weights.at<float>(0));
  for (size_t i = 2; i < indices2.size(); ++i) {
    BOOST_REQUIRE(indices2[i-1] < indices2[i]);
    BOOST_REQUIRE(h2RawValues[indices2[i]] == h2Weights.at<float>(i));
  }
  indexSet.clear();
  for (int i : indices2)
    indexSet.insert(i);
  BOOST_REQUIRE((int)indexSet.size() == h2Weights.rows);
    
  bool accumulate = true;
  for (size_t i = 0; i < indices1.size(); ++i) {
    for (size_t j = 0; j < indices2.size(); ++j) {
      BOOST_REQUIRE(dmTrimmed.at<float>(i,j) == dmRegular.at<float>(indices1[i], indices2[j]));
    }
  }
  BOOST_REQUIRE(accumulate);
  
  std::list<int> sizesN;
  std::list<int> sizesM;
  for (double d = 0.99; d > 0; d -= (1.0 - d)) {
    dmTrimmed = cv::Mat();
    h1.generateTrimmedDistanceMatrix(0.0, d, INT_MAX, h2, dmTrimmed, h1Weights, h2Weights);
    sizesN.push_back(dmTrimmed.rows);
    sizesM.push_back(dmTrimmed.cols);
    BOOST_REQUIRE(dmTrimmed.rows == h1Weights.rows);
    BOOST_REQUIRE(dmTrimmed.cols == h2Weights.rows);
    BOOST_REQUIRE(h1Weights.cols == 1);
    BOOST_REQUIRE(h2Weights.cols == 1);
  }

  std::list<int>::const_iterator prevSize = sizesN.cbegin();
  std::list<int>::const_iterator curSize = sizesN.cbegin();
  ++curSize;
  while (curSize != sizesN.cend()) {
    BOOST_REQUIRE(*curSize < *prevSize);
    ++curSize;
    ++prevSize;
  }

  prevSize = sizesM.cbegin();
  curSize = sizesM.cbegin();
  ++curSize;
  while (curSize != sizesM.cend()) {
    BOOST_REQUIRE(*curSize < *prevSize);
    ++curSize;
    ++prevSize;
  }

  dmTrimmed = cv::Mat();
  h1.generateTrimmedDistanceMatrix(0.0, 0.0, INT_MAX, h2, dmTrimmed, h1Weights, h2Weights);
  BOOST_REQUIRE(dmTrimmed.cols == 0);
  BOOST_REQUIRE(dmTrimmed.rows == 0);
  BOOST_REQUIRE(h1Weights.rows == 0);
  BOOST_REQUIRE(h2Weights.rows == 0);
  BOOST_REQUIRE(h1Weights.cols == 1);
  BOOST_REQUIRE(h2Weights.cols == 1);

  dmTrimmed = cv::Mat();
  dmRegular = cv::Mat();

  dmRegular = h1.generateDistanceMatrix(4.0, h2);

  h1.generateTrimmedDistanceMatrix(4.0, 0.5, INT_MAX, h2, dmTrimmed, h1Weights, 
                                   h2Weights, &indices1, &indices2);

  indexSet.clear();
  for (int i : indices1)
    indexSet.insert(i);
  BOOST_REQUIRE((int)indexSet.size() == h1Weights.rows);
  BOOST_REQUIRE((int)indices1.size() == h1Weights.rows);

  BOOST_REQUIRE(h1RawValues[indices1[0]] == h1Weights.at<float>(0));
  for (size_t i = 1; i < indices1.size(); ++i) {
    BOOST_REQUIRE(indices1[i-1] < indices1[i]);
    BOOST_REQUIRE(h1RawValues[indices1[i]] == h1Weights.at<float>(i));
  }
    

  indexSet.clear();
  for (int i : indices2)
    indexSet.insert(i);
  BOOST_REQUIRE((int)indexSet.size() == h2Weights.rows);
  BOOST_REQUIRE((int)indices2.size() == h2Weights.rows);

  BOOST_REQUIRE(h2RawValues[indices2[0]] == h2Weights.at<float>(0));
  for (size_t i = 1; i < indices2.size(); ++i) {
    BOOST_REQUIRE(indices2[i-1] < indices2[i]);
    BOOST_REQUIRE(h2RawValues[indices2[i]] == h2Weights.at<float>(i));
  }

  BOOST_REQUIRE((int)h2RawValues.size() > h2Weights.rows);
  BOOST_REQUIRE((int)h1RawValues.size() > h1Weights.rows);
  BOOST_REQUIRE(h1Weights.rows == dmTrimmed.rows);
  BOOST_REQUIRE(h2Weights.rows == dmTrimmed.cols);
  BOOST_REQUIRE(dmRegular.cols > dmTrimmed.cols);
  BOOST_REQUIRE(dmRegular.rows > dmTrimmed.rows);

  float h1Min = math::min(h1Weights); //*std::min_element(h1Weights.cbegin(), h1Weights.cend());
  // std::vector<float>::const_iterator it = h1Weights.cbegin(); 
  float* it = h1Weights.ptr<float>();
  std::vector<float>::const_iterator jt = h1.getRawValues().cbegin(); 
  // while (it != h1Weights.cend()) {
  while (jt != h1.getRawValues().cend()) {
    if (*it != *jt) {
      BOOST_REQUIRE(*jt < h1Min);
    }
    else
      ++it;
    ++jt;
  }

  indexSet.clear();
  for (int i : indices2)
    indexSet.insert(i);
  BOOST_REQUIRE((int)indexSet.size() == h2Weights.rows);
  float h2Min = math::min(h2Weights); // *std::min_element(h2Weights.cbegin(), h2Weights.cend());
  BOOST_REQUIRE(h2Min > 0);
  for (int i = 0; i < h2Weights.rows; ++i) {
    BOOST_REQUIRE(h2Weights.at<float>(i) == h2RawValues[indices2[i]]);
  }
  for (size_t i = 0; i < h2RawValues.size(); ++i) {
    if (!indexSet.count(i))
      BOOST_REQUIRE(h2RawValues[i] <= h2Min);
  }

  accumulate = true;
  for (size_t i = 0; i < indices1.size(); ++i) {
    for (size_t j = 0; j < indices2.size(); ++j) {
      BOOST_REQUIRE(dmTrimmed.at<float>(i,j) == dmRegular.at<float>(indices1[i], indices2[j]));
    }
  }
  BOOST_REQUIRE(accumulate);

  h1.generateTrimmedDistanceMatrix(4.0, 2.0, 100, h2, dmTrimmed, h1Weights, 
                                   h2Weights);
  BOOST_REQUIRE(dmTrimmed.rows == 100);
  BOOST_REQUIRE(dmTrimmed.cols == 100);
  BOOST_REQUIRE(h2Weights.rows == 100);
  BOOST_REQUIRE(h1Weights.rows == 100);
  BOOST_REQUIRE(h1Weights.cols == 1);
  BOOST_REQUIRE(h2Weights.cols == 1);
}



BOOST_AUTO_TEST_CASE( testSurfSift ) {
  cv::SIFT sift;

  std::vector<cv::KeyPoint> siftKeypoints1;
  cv::Mat siftDescriptors1;
  cv::Mat hand1 = getTestHand1();

  sift(hand1, cv::noArray(), siftKeypoints1, siftDescriptors1);
  BOOST_REQUIRE(siftKeypoints1.size() > 0);
  BOOST_REQUIRE(siftDescriptors1.rows == (int)siftKeypoints1.size());
  BOOST_REQUIRE(siftDescriptors1.cols == 128);

  std::vector<cv::KeyPoint> siftKeypoints2;
  cv::Mat siftDescriptors2;
  cv::Mat hand2 = getTestHand2();

  sift(hand2, cv::noArray(), siftKeypoints2, siftDescriptors2);
  BOOST_REQUIRE(siftKeypoints2.size() > 0);
  BOOST_REQUIRE(siftDescriptors2.rows == (int)siftKeypoints2.size());
  BOOST_REQUIRE(siftDescriptors2.cols == 128);
  BOOST_REQUIRE(siftKeypoints2.size() != siftKeypoints1.size());

  cv::Mat siftDmL2 = generateDescriptorDmL2(siftDescriptors1, siftDescriptors2);

  BOOST_REQUIRE(siftDmL2.rows == siftDescriptors1.rows);
  BOOST_REQUIRE(siftDmL2.cols == siftDescriptors2.rows);
  BOOST_REQUIRE(siftDescriptors1.cols == siftDescriptors2.cols);
  for (int i = 0; i < siftDescriptors1.rows; ++i) {
    for (int j = 0; j < siftDescriptors2.rows; ++j) {
      double norm = 0;
      for (int k = 0; k < siftDescriptors1.cols; ++k)
        norm += siftDescriptors1.at<float>(i,k) * siftDescriptors2.at<float>(j,k);
      BOOST_REQUIRE(almostEqual(std::sqrt(norm), siftDmL2.at<float>(i,j)));
    }
  }

  cv::Mat siftDmChiSq = generateDescriptorDmChiSq(siftDescriptors1, siftDescriptors2);
  BOOST_REQUIRE(siftDmChiSq.rows == siftDescriptors1.rows);
  BOOST_REQUIRE(siftDmChiSq.cols == siftDescriptors2.rows);
  for (int i = 0; i < siftDescriptors1.rows; ++i) {
    for (int j = 0; j < siftDescriptors2.rows; ++j) {
      double norm = 0;
      for (int k = 0; k < siftDescriptors1.cols; ++k) {
        float p = siftDescriptors1.at<float>(i,k);
        float q = siftDescriptors2.at<float>(j,k);
        norm +=  (p-q)*(p-q)/(p + q + 1e-5);
      }
      BOOST_REQUIRE(almostEqual(0.5 * norm, siftDmChiSq.at<float>(i,j)));
    }
  }

  cv::SURF surf;

  std::vector<cv::KeyPoint> surfKeypoints1;
  cv::Mat surfDescriptors1;

  surf(hand1, cv::noArray(), surfKeypoints1, surfDescriptors1);
  BOOST_REQUIRE(surfKeypoints1.size() > 0);
  BOOST_REQUIRE(surfDescriptors1.rows == (int)surfKeypoints1.size());
  BOOST_REQUIRE(surfDescriptors1.cols == 64);

  std::vector<cv::KeyPoint> surfKeypoints2;
  cv::Mat surfDescriptors2;

  surf(hand2, cv::noArray(), surfKeypoints2, surfDescriptors2);
  BOOST_REQUIRE(surfKeypoints2.size() > 0);
  BOOST_REQUIRE(surfDescriptors2.rows == (int)surfKeypoints2.size());
  BOOST_REQUIRE(surfDescriptors2.cols == 64);
  BOOST_REQUIRE(surfKeypoints2.size() != surfKeypoints1.size());

  cv::Mat surfDmL2 = generateDescriptorDmL2(surfDescriptors1, surfDescriptors2);

  BOOST_REQUIRE(surfDmL2.rows == surfDescriptors1.rows);
  BOOST_REQUIRE(surfDmL2.cols == surfDescriptors2.rows);
  BOOST_REQUIRE(surfDescriptors1.cols == surfDescriptors2.cols);
  for (int i = 0; i < surfDescriptors1.rows; ++i) {
    for (int j = 0; j < surfDescriptors2.rows; ++j) {
      double norm = 0;
      for (int k = 0; k < surfDescriptors1.cols; ++k)
        norm += surfDescriptors1.at<float>(i,k) * surfDescriptors2.at<float>(j,k);
      BOOST_REQUIRE(almostEqual(std::sqrt(norm), surfDmL2.at<float>(i,j)));
    }
  }

  cv::Mat surfDmChiSq = generateDescriptorDmChiSq(surfDescriptors1, surfDescriptors2);
  BOOST_REQUIRE(surfDmChiSq.rows == surfDescriptors1.rows);
  BOOST_REQUIRE(surfDmChiSq.cols == surfDescriptors2.rows);
  for (int i = 0; i < surfDescriptors1.rows; ++i) {
    for (int j = 0; j < surfDescriptors2.rows; ++j) {
      double norm = 0;
      for (int k = 0; k < surfDescriptors1.cols; ++k) {
        float p = surfDescriptors1.at<float>(i,k);
        float q = surfDescriptors2.at<float>(j,k);
        norm +=  (p-q)*(p-q)/(p + q + 1e-13);
      }
      BOOST_REQUIRE(almostEqual(0.5 * norm, surfDmChiSq.at<float>(i,j), 1e-3));
    }
  }  
}



BOOST_AUTO_TEST_CASE( testChamfer ) {
  for (int ctr = 0; ctr < 100; ++ctr) {
    int N = 1 + rand() % 200;
    int M = 1 + rand() % 200;
    cv::Mat X(N, M, CV_8UC1, cv::Scalar::all(0));
    cv::Mat Y(N, M, CV_8UC1, cv::Scalar::all(0));
    for (int i = 0; i < N; ++i) {
      for (int j = 0; j < M; ++j) {
        if (rand() % 50 == 0)
          X.at<uchar>(i,j) = 255;
        if (rand() % 71 == 0)
          Y.at<uchar>(i,j) = 255;
      }
    }


    cv::Mat XDt;
    cv::Mat YDt;
    cv::distanceTransform(cv::Scalar::all(255) - X, XDt, CV_DIST_L2, 5);
    cv::distanceTransform(cv::Scalar::all(255) - Y, YDt, CV_DIST_L2, 5);

    cv::Mat XFloat;
    cv::Mat YFloat;
    X.convertTo(XFloat, CV_32FC1);
    Y.convertTo(YFloat, CV_32FC1);
    XFloat /= 255;
    YFloat /= 255;

    int XSize = cv::countNonZero(X);
    int YSize = cv::countNonZero(Y);

    if (XSize == 0 || YSize == 0)
      continue;

    double dcdXtoY = math::directedChamferDistance(XFloat, XSize, YDt);

    double dcdXtoYalt = 0;
    for (int i = 0; i < N; ++i) {
      for (int j = 0; j < M; ++j) {
        if (X.at<uchar>(i,j) > 0) {
          double minDist = DBL_MAX;
          for (int u = 0; u < N; ++u) {
            for (int v = 0; v < M; ++v) {
              if (Y.at<uchar>(u,v) > 0) {
                double dist = cv::norm(cv::Point(i,j) - cv::Point(u,v));
                if (dist < minDist)
                  minDist = dist;
              }
            }
          }
          dcdXtoYalt += minDist;
        }
      }
    }
    dcdXtoYalt /= XSize;
    BOOST_REQUIRE(almostEqual(dcdXtoY, dcdXtoYalt, 1e-2));


    double dcdYtoX = math::directedChamferDistance(YFloat, YSize, XDt);
    double dcdYtoXalt = 0;
    for (int i = 0; i < N; ++i) {
      for (int j = 0; j < M; ++j) {
        if (Y.at<uchar>(i,j) > 0) {
          double minDist = DBL_MAX;
          for (int u = 0; u < N; ++u) {
            for (int v = 0; v < M; ++v) {
              if (X.at<uchar>(u,v) > 0) {
                double dist = cv::norm(cv::Point(i,j) - cv::Point(u,v));
                if (dist < minDist)
                  minDist = dist;
              }
            }
          }
          dcdYtoXalt += minDist;
        }
      }
    }
    dcdYtoXalt /= YSize;
    BOOST_REQUIRE(almostEqual(dcdYtoX, dcdYtoXalt, 1e-2));

    BOOST_REQUIRE(almostEqual(chamferDistance(XFloat, XSize, XDt, YFloat, YSize, YDt),
                              dcdYtoXalt + dcdXtoYalt, 1e-2));
  }
}



#if 0
BOOST_AUTO_TEST_CASE( testHogEmd ) {
  HOGDescriptor h1 = getTestHog();
  HOGDescriptor h2 = getTestHog2();
  BOOST_REQUIRE(h1.getDescriptorSize() != h2.getDescriptorSize());
  BOOST_REQUIRE(h1.getDescriptorSize() == (int)h1.getRawValues().size());
  BOOST_REQUIRE(h2.getDescriptorSize() == (int)h2.getRawValues().size());
  BOOST_REQUIRE(h1.getDescriptorSize() > 0);
  BOOST_REQUIRE(h2.getDescriptorSize() > 0);

  // std::cout << "Descriptor 1: " << h1.getDescriptorSize() << std::endl;
  // for (float f : h1.getRawValues())
  //   std::cout << f << std::endl;
  
  // std::cout << "Descriptor 2: " << h2.getDescriptorSize() << std::endl;
  // for (float f : h2.getRawValues())
  //   std::cout << f << std::endl;

  timerOn();
  cv::Mat dm = h1.generateDistanceMatrix(h1.getBlockSize().width, h2);  
  std::cerr << "Computed the distance matrix in " << timerOff() << std::endl;

  const std::vector<float>& h1rw = h1.getRawValues();
  std::vector<double> hogValues1(h1rw.size());
  for (size_t i = 0; i < h1rw.size(); ++i)
    hogValues1[i] = h1rw[i];

  const std::vector<float>& h2rw = h2.getRawValues();
  std::vector<double> hogValues2(h2rw.size());
  for (size_t i = 0; i < h2rw.size(); ++i)
    hogValues2[i] = h2rw[i];

  timerOn();
  double emdDistanceNaive = math::emdNaive(hogValues1, hogValues2, dm);
  std::cerr << "Computed naïve EMD distance in " << timerOff() << ": " 
            << emdDistanceNaive << std::endl;

  std::cerr << "Trimming..." << std::endl;
}
#endif

BOOST_AUTO_TEST_SUITE_END()

BOOST_AUTO_TEST_SUITE( multiThreadTests ) 



#ifdef SLMOTION_THREADING
BOOST_AUTO_TEST_CASE( testLock ) {
  Thread::setMaxThreads(3);
  std::mutex m1, m2;

  BOOST_REQUIRE(m1.try_lock());
  BOOST_REQUIRE(m2.try_lock());
  BOOST_REQUIRE(!m1.try_lock());
  BOOST_REQUIRE(!m2.try_lock());
  m1.unlock();

  auto unlockMutex = [&]() {
    std::this_thread::sleep_for(std::chrono::milliseconds(5000));
    m2.unlock();
  };

  auto lockMutexes = [&]() {
    std::this_thread::sleep_for(std::chrono::milliseconds(2000));
    lock(m1,m2);
  };
  Thread t(lockMutexes);

  std::this_thread::sleep_for(std::chrono::milliseconds(1000));
  BOOST_CHECK(m1.try_lock());
  BOOST_CHECK(!m2.try_lock());
  m1.unlock();

  std::this_thread::sleep_for(std::chrono::milliseconds(2000));
  BOOST_CHECK(m1.try_lock());
  BOOST_CHECK(!m2.try_lock());
  m1.unlock();
  m2.unlock();

  std::this_thread::sleep_for(std::chrono::milliseconds(4000));
  BOOST_CHECK(!m1.try_lock());
  BOOST_CHECK(!m2.try_lock());

  t.join();
}
#endif


#ifdef SLMOTION_THREADING
BOOST_AUTO_TEST_CASE( testMultiThread ) {
  // make sure the multithreaded approach provides the same result

  // force the same random number seed
  size_t randomSeed = reinterpret_cast<size_t>(&slmotion::debug);

  slmotion::debug = 0;
  AnalysisResult results1, results2;
  shared_ptr<BlackBoard> bb1, bb2;

  {
    srand(randomSeed);
    cv::theRNG().state = randomSeed;
    vector<Analyser> anals;
    SLMotionConfiguration opts;
    // createAnalysers(assignFilenamesToJobs(vector<string>{ TESTVIDEOFILE6 }),
    //                 vector<string>(), anals, opts);
    createAnalysers(assignFilenamesToJobs(vector<string>{ TESTVIDEOFILE7 }),
                    vector<string>(), anals, opts, 0, 0);

#ifdef SLMOTION_THREADING   
    slmotion::Thread::setMaxThreads(0);
#endif
    
    AnalysisController analCon(anals[0].visualiser, anals[0].slio, false);
    analCon.analyseVideofiles(anals, 0, SIZE_MAX, false, false, "", "", "",
                              opts.visualiserCombinationStyle);
    results1 = FeatureCreator::extractRawResults(0, 
                                                 anals[0].frameSource->size(), 
                                                 *anals[0].blackboard);

    bb1 = anals[0].blackboard;
  }

  {
    srand(randomSeed);
    cv::theRNG().state = randomSeed;
    vector<Analyser> anals;
    SLMotionConfiguration opts;
    // createAnalysers(assignFilenamesToJobs(vector<string>{ TESTVIDEOFILE6 }),
    //                 vector<string>(), anals, opts); 
    createAnalysers(assignFilenamesToJobs(vector<string>{ TESTVIDEOFILE7 }),
                    vector<string>(), anals, opts, 0, 0); 


#ifdef SLMOTION_THREADING   
    slmotion::Thread::setMaxThreads(0);
#endif
   
    AnalysisController analCon(anals[0].visualiser, anals[0].slio, false);
    analCon.analyseVideofiles(anals, 0, SIZE_MAX, false, false, "", "", "",
                              opts.visualiserCombinationStyle);
    results2 = FeatureCreator::extractRawResults(0, 
                                                 anals[0].frameSource->size(), 
                                                 *anals[0].blackboard);
    bb2 = anals[0].blackboard;
  }

#ifdef SLMOTION_THREADING
    slmotion::Thread::setMaxThreads(5);

    AnalysisResult results3;
    shared_ptr<BlackBoard> bb3;

  {
    srand(randomSeed);
    cv::theRNG().state = randomSeed;
    vector<Analyser> anals;
    SLMotionConfiguration opts;
    // createAnalysers(assignFilenamesToJobs(vector<string>{ TESTVIDEOFILE6 }),
    //                 vector<string>(), anals, opts); 
    createAnalysers(assignFilenamesToJobs(vector<string>{ TESTVIDEOFILE7 }),
                    vector<string>(), anals, opts); 
   
    AnalysisController analCon(anals[0].visualiser, anals[0].slio, false);
    analCon.analyseVideofiles(anals, 0, SIZE_MAX, false, false, "", "", "");
    results3 = FeatureCreator::extractRawResults(0, 
                                                 anals[0].frameSource->size(), 
                                                 *anals[0].blackboard);
    bb3 = anals[0].blackboard;
  }
#endif


  auto tpSetVectorsEqual = [](const std::vector<std::set<TrackedPoint>>& a,
                              const std::vector<std::set<TrackedPoint>>& b) {
    if (a.size() != b.size())
      return false;
    
    //    cout << "Vector lengths match" << endl;

    vector<vector<TrackedPoint::point_t>> aPoints(a.size());
    vector<vector<TrackedPoint::point_t>> bPoints(a.size());
    for (size_t i = 0; i < a.size(); ++i) {
      //      cout << "ai " << a[i].size() << " bi " << b[i].size() << endl;

      if (a[i].size() != b[i].size())
        return false;

      for (auto it = a[i].cbegin(); it != a[i].cend(); ++it)
        aPoints[i].push_back(static_cast<TrackedPoint::point_t>(*it));

      for (auto it = b[i].cbegin(); it != b[i].cend(); ++it)
        bPoints[i].push_back(static_cast<TrackedPoint::point_t>(*it));
    }

    //    cout << "Sizes match" << endl;
      
    return aPoints == bPoints;
  };

  auto analResEqual = [&](const AnalysisResult& lh, const AnalysisResult& rh) {
    if (!tpSetVectorsEqual(lh.KLTPoints,rh.KLTPoints))
      return false;
    // cout << "tpSetVectors match" << endl;

    if (lh.headASMs != rh.headASMs)
      return false;
    // cout << "Head ASMs are equal" << endl;

    if (lh.leftHandASMs != rh.leftHandASMs)
      return false;
    // cout << "leftHandASMs are equal" << endl;

    if (lh.rightHandASMs.size() != rh.rightHandASMs.size()) {
      // cout << "rightHandASM vector sizes do not match!" << endl;
      return false;
    }

    if (lh.rightHandASMs != rh.rightHandASMs)
      return false;
    // cout << "rightHandASMs are equal" << endl;

    // cout << lh.firstFrame << " " << rh.firstFrame << endl;
    // cout << lh.lastFrame << " " << rh.lastFrame << endl;
    return lh.firstFrame == rh.firstFrame &&
    lh.lastFrame == rh.lastFrame;
  };

  BOOST_REQUIRE(results1.firstFrame == results2.firstFrame);
  BOOST_REQUIRE(results1.lastFrame == results2.lastFrame);
#ifdef SLMOTION_THREADING
  BOOST_REQUIRE(results1.firstFrame == results3.firstFrame);
  BOOST_REQUIRE(results1.lastFrame == results3.lastFrame);
#endif
  for (size_t i = results1.firstFrame; i < results1.lastFrame; ++i) {
    BOOST_REQUIRE(*bb1->get<cv::Rect>(i, FACEDETECTOR_BLACKBOARD_ENTRY) ==
                  *bb2->get<cv::Rect>(i, FACEDETECTOR_BLACKBOARD_ENTRY));
    BOOST_REQUIRE(equal(*bb1->get<cv::Mat>(i, COLOURSPACECONVERTER_BLACKBOARD_GSIMAGE_ENTRY),
                        *bb2->get<cv::Mat>(i, COLOURSPACECONVERTER_BLACKBOARD_GSIMAGE_ENTRY)));

    BOOST_REQUIRE(equal(*bb1->get<Mat>(i, SKINDETECTOR_BLACKBOARD_MASK_ENTRY),
                        *bb2->get<Mat>(i, SKINDETECTOR_BLACKBOARD_MASK_ENTRY)));

    BOOST_REQUIRE(*bb1->get<vector<Blob>>(i, BLACKBOARD_BLOBS_ENTRY) ==
                  *bb2->get<vector<Blob>>(i, BLACKBOARD_BLOBS_ENTRY));
  }

  BOOST_REQUIRE(analResEqual(results1,results1));
  BOOST_REQUIRE(analResEqual(results1,results2));


#ifdef SLMOTION_THREADING
  for (size_t i = results1.firstFrame; i < results1.lastFrame; ++i) {
    BOOST_REQUIRE(bb1->get<cv::Rect>(i, FACEDETECTOR_BLACKBOARD_ENTRY) ==
                  bb3->get<cv::Rect>(i, FACEDETECTOR_BLACKBOARD_ENTRY));
    BOOST_REQUIRE(equal(bb1->get<cv::Mat>(i, COLOURSPACECONVERTER_BLACKBOARD_GSIMAGE_ENTRY),
                      bb3->get<cv::Mat>(i, COLOURSPACECONVERTER_BLACKBOARD_GSIMAGE_ENTRY)));

    BOOST_REQUIRE(equal(bb1->get<Mat>(i, SKINDETECTOR_BLACKBOARD_MASK_ENTRY),
                      bb3->get<Mat>(i, SKINDETECTOR_BLACKBOARD_MASK_ENTRY)));

    BOOST_REQUIRE(bb1->get<vector<Blob>>(i, BLACKBOARD_BLOBS_ENTRY) ==
                bb3->get<vector<Blob>>(i, BLACKBOARD_BLOBS_ENTRY));
  }
  BOOST_REQUIRE(analResEqual(results1,results3));
#endif
}
#endif



#ifdef SLMOTION_THREADING
// this test has been tailored for data race huntery
BOOST_AUTO_TEST_CASE( helGrindTest ) {
  slmotion::Thread::setMaxThreads(5);
  
  vector<Analyser> anals;
  SLMotionConfiguration opts;
  createAnalysers(assignFilenamesToJobs(vector<string>{ TESTVIDEOFILE6 }),
                  vector<string>(), anals, opts); 
  
  AnalysisController analCon(anals[0].visualiser, anals[0].slio, false);
  analCon.analyseVideofiles(anals, 0, SIZE_MAX, false, false, "", "", "");
}
#endif

BOOST_AUTO_TEST_SUITE_END()

