#include <fstream>
#include <boost/algorithm/string.hpp>
#include "util.hpp"

#include "regex.hpp"

#ifdef __GNUG__
#include <cxxabi.h>
#endif // __GNUG__

#include <sys/timeb.h>



using cv::Rect;
using cv::Point;
using cv::Mat;
using std::string;
using std::vector;
using std::shared_ptr;
using boost::split;
using boost::is_any_of;
using std::type_info;



namespace slmotion {
  static std::string execName = "EXECUTABLE NAME UNSET (this is a bug!)";

  // #if (__GNUC__ >= 4 && __GNUC_MINOR__ >= 5)
  static long timer = 0;
  
  void timerOn() {
    struct timeb time;
    ftime(&time);
    timer = time.time*1000 + time.millitm;
  }

  time_t timerOff() {
    struct timeb time;
    ftime(&time);
    return time.time*1000 + time.millitm - timer;
  }

  std::ostream& operator<<(std::ostream& ofs, const TrackedPoint& p) {
    return ofs << p.getId() << "[" 
               << static_cast<TrackedPoint::point_t>(p) << "]";
  }
  // #endif

  void printTime(std::ostream& os, time_t time) {
    time_t h = time/3600;
    time_t m = (time - h*3600)/60;
    time_t s = time - h*3600 - m*60;
    os << h << ":" << m << ":" << s;
  }



  std::string getInstallPrefix() {
    char* c = getenv("SLMOTION_INSTALL_PREFIX");
    return c ? c : SLMOTION_INSTALL_PREFIX;
  }



#ifdef SLMOTION_WITH_INTRAFACE
  std::string getIntrafacePrefix() {
    char* c = getenv("SLMOTION_INTRAFACE_PREFIX");
    return c ? c : SLMOTION_INTRAFACE_PREFIX;
  }
#endif // SLMOTION_WITH_INTRAFACE



  PixelToBodyCoordinateTransformer::PixelToBodyCoordinateTransformer(const BodyToPixelCoordinateTransformer& other) :
    bodyWidth(other.bodyWidth), throatLocation(other.throatLocation) {}

  BodyToPixelCoordinateTransformer::BodyToPixelCoordinateTransformer(const PixelToBodyCoordinateTransformer& other) :
    bodyWidth(other.bodyWidth), throatLocation(other.throatLocation) {}


  std::ostream& operator<<(std::ostream& os, const cv::Mat& m) {
    assert(m.type() == CV_64FC1 || m.type() == CV_8UC1 || m.type() == CV_32SC1 || m.type() == CV_32FC1);
    switch(m.type()) {
    case CV_64FC1:
      os << cv::Mat_<double>(m);
      break;
    case CV_32SC1:
      os << cv::Mat_<int>(m);
      break;
    case CV_8UC1:
      os << cv::Mat_<uchar>(m);
      break;
    case CV_32FC1:
      os << cv::Mat_<float>(m);
      break;
    default:
      throw std::invalid_argument("Unsupported matrix element type or invalid number of channels.");
    }
    return os;
  }



  std::ostream& operator<<(std::ostream& os, const Rect& r) {
    return os << '(' << r.x << ',' << r.y << ")->("
              << (r.x+r.width) << ',' << (r.y+r.height) << ')';
  }



  namespace {
    std::vector<std::string> argvVec;
  }



  void setArgV(int argc, char** argv) {
    argvVec = std::vector<std::string>(argv, argv + argc);
  }



  std::string getArgV() {
    return boost::join(argvVec, " ");
  }



  void drawPolygon(cv::Mat& target,
                   const std::vector<cv::Point>& contour,
                   const cv::Scalar colour, const Point& translation) {
    const Point* p = &contour[0];
    int i = contour.size();
    fillPoly(target, &p, &i, 1, colour, 8, 0, translation);
  }



  void drawPolygon(cv::Mat& target,
                   const std::vector < std::vector<cv::Point> >& contours,
                   const cv::Scalar colour) {
    vector<const Point*> pts(contours.size());
    vector<int> npts(contours.size());
    for (size_t i = 0; i < contours.size(); ++i) {
      pts[i] = &contours[i][0];
      npts[i] = contours[i].size();
    }
    fillPoly(target, &pts[0], &npts[0], contours.size(), colour);
  }



  std::string getExtension(const string& filename) {
    // regex for matching file extensions
    std::regex extRegex(".*(\\.[a-z0-9]{2,4})$", 
                        std::regex_constants::icase);
    std::smatch m;
    if (regex_match(filename, m, extRegex))
      return m[1];
    return "";
  }



  std::ostream& operator<<(std::ostream& os, const cv::Size& s) {
    return os << s.width << " " << s.height;
  }


  
  static double getMaxAbsDiff(const cv::Mat& X, const cv::Mat& Y) {
    Mat out;
    cv::absdiff(X, Y, out);
    double maxVal = 0;
    double maxMax = 0;
    vector<Mat> mats;
    split(out, mats);
    for (auto it = mats.cbegin(); it != mats.cend(); ++it) {
      minMaxLoc(*it, NULL, &maxVal);
      maxMax = std::max(maxVal, maxMax);
    }
    return maxMax;
  }



  bool equal(const cv::Mat& X, const cv::Mat& Y) {
    if (X.empty() && Y.empty())
      return true;
    else if (X.empty() || Y.empty())
      return false;

    if (X.size() != Y.size())
      return false;
    if (X.type() != Y.type())
      return false;

    return getMaxAbsDiff(X,Y) == 0;
  }


  
  bool memEqual(const cv::Mat& X, const cv::Mat& Y) {
    if (X.empty() && Y.empty())
      return true;
    else if (X.empty() || Y.empty())
      return false;

    if (X.size() != Y.size())
      return false;
    if (X.type() != Y.type())
      return false;

    size_t size = X.total()*X.elemSize();
    if (Y.total()*Y.elemSize() != size)
      return false;

    if (!X.isContinuous() || !Y.isContinuous()) 
      throw std::invalid_argument("This function can only be used on continuous matrices!");

    return memcmp(X.data, Y.data, size) == 0;
  }



  bool almostEqual(const cv::Mat& X, const cv::Mat& Y, double epsilon) {
    if (X.empty() && Y.empty())
      return true;
    else if (X.empty() || Y.empty())
      return false;

    if (X.size() != Y.size())
      return false;
    if (X.type() != Y.type())
      return false;

    return getMaxAbsDiff(X,Y) < epsilon;
  }



  void readFramesAndGT(const std::string& videoGtFile, 
                       const std::string& prefix,
                       vector<Mat>& outFrames,
                       vector<Mat>& outGroundTruths) {
    outFrames.clear();
    outGroundTruths.clear();

    std::ifstream ifs(videoGtFile);
    string videoFile, groundTruthFile;
    size_t frames = 0;

    Mat temp;

    while (ifs >> videoFile >> groundTruthFile >> frames) {
      cv::VideoCapture vc1(prefix + videoFile);
      cv::VideoCapture vc2(prefix + groundTruthFile);
      size_t i = 0;
 
      std::cout << prefix + videoFile + " "
                << prefix + groundTruthFile << std::endl;

      while (i < frames && vc1.grab() && vc2.grab()) {
        vc1.retrieve(temp);
        outFrames.push_back(temp.clone());
        vc2.retrieve(temp);
        outGroundTruths.push_back(temp.clone());
        std::cout << '\r' << i;
        ++i;
      }
      std::cout << std::endl;
    }
  }



  std::vector<Point> chainApproxSimpleToChainApproxNone(const std::vector<Point>& src) {
    // no points, what can you do?
    if (src.size() == 0)
      return std::vector<Point>();

    std::vector<Point> chain; 
    Point first = src[0];
    for (size_t i = 0; i < src.size(); ++i) {
      // use the first element as the element following the last element in 
      // order to make the curve a closed loop
      const Point& second = i < src.size() - 1 ? src[i+1] : src[0];

      Point delta = second - first;
      if (delta.x != 0)
        delta.x /= std::abs(delta.x);
      if (delta.y != 0)
        delta.y /= std::abs(delta.y);

      do {
        chain.push_back(first);
        first += delta;
      } while (first != second);
    }

    return chain;
  }



  static PostProcessMorphologicalTransformationFilter createMorphFilter(const std::string& s) {
    vector<string> tokens;
    split(tokens, s, is_any_of(","));
    if (tokens.size() != 2)
      throw ConfigurationFileException("'morph' filter must receive two arguments: a type string and a number of iterations");

    typedef slmotion::PostProcessMorphologicalTransformationFilter::Type Type;
    Type t;
    if (tokens[0] == "open")
      t = Type::OPEN;
    else if (tokens[0] == "close")
      t = Type::CLOSE;
    else if (tokens[0] == "dilate")
      t = Type::DILATE;
    else if (tokens[0] == "erode")
      t = Type::ERODE;
    else
      throw ConfigurationFileException("'" + tokens[0] + "' is not a valid morphological transformation.");

    int n = boost::lexical_cast<int>(tokens[1]);

    return PostProcessMorphologicalTransformationFilter(t, n);
  }



  vector<std::shared_ptr<PostProcessFilter>> parseSkinPostprocess(const std::string& s) {
    std::regex entryRegex("([a-z]+)\\(([^\\)]*)\\)", 
                          std::regex_constants::icase);
    std::smatch m;
    vector<string> tokens;
    vector<shared_ptr<PostProcessFilter>> filters;
    if (s.length() > 0)
      split(tokens, s, is_any_of(";"));

    for(auto it = tokens.cbegin(); it != tokens.end(); ++it) {
      if (regex_match(*it, m, entryRegex)) {
        try {
          if (m[1] == "crop")
            filters.push_back(shared_ptr<PostProcessFilter>(slmotion::PostProcessCropFilter(createFromTokens<cv::Rect, int>(m[2], ",")).clone()));
          else if (m[1] == "fillHoles")
            filters.push_back(shared_ptr<PostProcessFilter>(slmotion::PostProcessHoleFillerFilter().clone()));
          else if (m[1] == "morph")
            filters.push_back(shared_ptr<PostProcessFilter>(createMorphFilter(m[2]).clone()));
          else if (m[1] == "watershed")
            filters.push_back(shared_ptr<PostProcessFilter>(slmotion::PostProcessWatershedFilter(boost::lexical_cast<size_t>(m[2])).clone()));
          else if (m[1] == "removeSmallComponents")
            filters.push_back(shared_ptr<PostProcessFilter>(slmotion::SmallComponentRemoverFilter(boost::lexical_cast<size_t>(m[2])).clone()));
          else
            throw ConfigurationFileException("'" + m[1].str() + "' is not a valid filter.");
        }
        catch (ConfigurationFileException& e) {
          throw ConfigurationFileException("Error while parsing skin detector post-processing filter '" + m[1].str() + "': " + e.what());
        }
      }
      else
        throw ConfigurationFileException("Malformed skin detector post-processing filter option; a semicolon-separated list of filters expected.");
    }

    return filters;
  }



  CustomColourSpace createCustomColourSpace(const std::string& s) {
    std::regex regex("custom:(.*)", std::regex_constants::icase);
    std::smatch m;
    vector<string> tokens;
    slmotion::CustomColourSpace c;
    if (!regex_match(s, m, regex))  
      throw ConfigurationFileException("'" + s + "' is not a valid colour space creation string.");
    string t = m[1];
    split(tokens, t, is_any_of(","));
    for(auto it = tokens.cbegin(); it != tokens.cend(); ++it) {
      if (*it == "y")
        c.addComponent(ColourSpace::Y);
      else if (*it == "cr")
        c.addComponent(ColourSpace::CR);
      else if (*it == "cb")
        c.addComponent(ColourSpace::CB);
      else if (*it == "h")
        c.addComponent(ColourSpace::H);
      else if (*it == "s")
        c.addComponent(ColourSpace::S);
      else if (*it == "v")
        c.addComponent(ColourSpace::V);
      else if (*it == "r")
        c.addComponent(ColourSpace::R);
      else if (*it == "g")
        c.addComponent(ColourSpace::G);
      else if (*it == "b")
        c.addComponent(ColourSpace::B);
      else if (*it == "x")
        c.addComponent(ColourSpace::X);
      else if (*it == "xyzy")
        c.addComponent(ColourSpace::XYZ_Y);
      else if (*it == "z")
        c.addComponent(ColourSpace::Z);
      else 
        throw ConfigurationFileException("'" + *it + "' is not a valid colour component");
    }

    return c;
  }



  std::ostream& operator<<(std::ostream& os, const Pdm::PoseParameter& pose) {
    return os << "(" << pose.theta << "," << pose.scale << "," << pose.tx 
              << "," << pose.ty << ")";
  }



#ifdef __GNUG__
  /**
   * Returns a demangled string (GCC specific!)
   *
   * The code is from here: http://stackoverflow.com/questions/281818/unmangling-the-result-of-stdtype-infoname/4541470#4541470
   */
  std::string string_demangle(const char* name) {

    int status = -4;

    char* res = abi::__cxa_demangle(name, NULL, NULL, &status);

    const char* const demangled_name = (status==0)?res:name;

    std::string ret_val(demangled_name);

    free(res);

    return ret_val;
  }
#endif // __GNUG__



  // void debugDump(FILE* file, const xml::XmlDocument& doc) {
  //   xmlElemDump(file, doc.getInternalDoc(), xmlDocGetRootElement(doc.getInternalDoc()));
  // }



  // void debugDump(std::ostream& os, const xml::XmlDocument& doc) {
  //   // Dump the document into memory, then write using standard C++ API
  //   char* memdata;
  //   size_t len;
  //   FILE* memstream = open_memstream(&memdata, &len);
  //   xmlElemDump(memstream, doc.getInternalDoc(), xmlDocGetRootElement(doc.getInternalDoc()));
  //   fclose(memstream);
  //   os.write(memdata, len);
  // }



  std::string getCanonicalPath(const std::string& relativePath) {
    vector<char> canonicalPath(4096); // 4096 here is a Stetson-Harrison constant
    if (!realpath(relativePath.c_str(), &canonicalPath[0]))
      throw IOException("No such path!");
    return &canonicalPath[0];
  }



  template<typename T>
  static std::string anyVectorToString(const boost::any& any) {
    const std::vector<T>& v = boost::any_cast<std::vector<T>>(any);
    std::ostringstream oss;
    oss << "[";
    for (const T& t : v)
      oss << " " << t;
    oss << " ]";
    return oss.str();
  }



  std::string anyToString(const boost::any& any) {
    if (any.type() == typeid(string))
      return boost::any_cast<string>(any);
    if (any.type() == typeid(double))
      return boost::lexical_cast<string>(boost::any_cast<double>(any));
    if (any.type() == typeid(unsigned long int))
      return boost::lexical_cast<string>(boost::any_cast<unsigned long int>(any));
    if (any.type() == typeid(unsigned int))
      return boost::lexical_cast<string>(boost::any_cast<unsigned int>(any));
    if (any.type() == typeid(int))
      return boost::lexical_cast<string>(boost::any_cast<int>(any));
    if (any.type() == typeid(bool))
      return boost::any_cast<bool>(any) ? "true" : "false";
    if (any.type() == typeid(std::vector<std::string>)) {
      vector<string> v = boost::any_cast<std::vector<string>>(any);
      if (v.size() == 0)
        return "";
      std::ostringstream oss;
      std::vector<string>::const_iterator it = v.begin();
      oss << *it;
      while(++it != v.end())
        oss << std::endl << *it;
      return oss.str();
    }
    if (any.type() == typeid(std::vector<std::vector<cv::Point2f>>))
      return anyVectorToString<std::vector<cv::Point2f>>(any);
    if (any.type() == typeid(std::vector<cv::Rect>))
      return anyVectorToString<cv::Rect>(any);
    if (any.type() == typeid(std::vector<float>))
      return anyVectorToString<float>(any);

    if (any.type() == typeid(cv::Rect)) {
      const cv::Rect& r = boost::any_cast<cv::Rect>(any);
      std::ostringstream oss;
      oss << r;
      return oss.str();
    }
      
    return "No known string conversion for type " + string_demangle(any.type().name()) + "";
  }



  bool matchRegex(const std::string& regexS, const std::string& string) {
    std::regex regex(regexS);
    std::smatch m;
    return regex_match(string, m, regex);
  }

  std::string replaceRegex(const std::string& regex, const std::string& replacement,
                           const std::string& string) {
    std::regex rx(regex);
    return std::regex_replace(string, rx, replacement);
  }



  void storeExecutableName(const std::string& executableName) {
    size_t pos = executableName.find_last_of('/');
    execName = pos == std::string::npos ? executableName : executableName.substr(pos+1);
  }



  void printWarning(const std::string& message) {
    std::cerr << execName << ": warning: " << message 
              << std::endl;
  } 



  cv::Mat convertToMarkusFormat(const cv::Mat& depthMat) {
    cv::Mat depthImage(depthMat.size(), CV_8UC1);
    double minVal;
    double maxVal;
    cv::minMaxLoc(depthMat, &minVal, &maxVal);
    cv::Mat subdepth;
    cv::subtract(depthMat, minVal, subdepth);
    subdepth.convertTo(depthImage, CV_8UC3, 255.0/(maxVal-minVal) );
    cv::Mat depthImage2 = cv::Mat(depthMat.size(),CV_8UC3);
    cv::cvtColor(depthImage, depthImage2, CV_GRAY2BGR);
    // this is to address a bug in Markus' code
    depthImage2.at<cv::Vec3b>(0,0) = cv::Vec3b(0,255,255);
    depthImage2.at<cv::Vec3b>(0,1) = cv::Vec3b(0,255,255);
    depthImage2.at<cv::Vec3b>(0,2) = cv::Vec3b(0,255,255);
    depthImage2.at<cv::Vec3b>(1,0) = cv::Vec3b(0,255,255);
    depthImage2.at<cv::Vec3b>(1,1) = cv::Vec3b(0,255,255);
    depthImage2.at<cv::Vec3b>(2,0) = cv::Vec3b(0,255,255);
    return depthImage2;
  }



  std::string cvMatTypeToString(int type) {
    switch (type) { 
    case CV_8UC1:
      return "CV_8UC1";
    case CV_8UC3:
      return "CV_8UC3";
    case CV_8UC4:
      return "CV_8UC4";
    case CV_16SC1:
      return "CV_16SC1";
    case CV_16UC1:
      return "CV_16UC1";
    case CV_32SC1:
      return "CV_32SC1";
    case CV_32FC1:
      return "CV_32FC1";
    case CV_64FC1:
      return "CV_64FC1";
    default:
      return "Unknown matrix type!";
    }
  }



  static std::unordered_map<size_t, std::string>* hashCodesToTypeNames = nullptr;
  void addHashCodeToTypeName(const type_info& typeInfo) {
    if (!hashCodesToTypeNames)
      hashCodesToTypeNames = new std::unordered_map<size_t, std::string>();

    if (!hashCodesToTypeNames->count(typeInfo.hash_code()))
      (*hashCodesToTypeNames)[typeInfo.hash_code()] = string_demangle(typeInfo.name());
  }


  std::string hashCodeToTypeName(size_t dataType) {
    if (hashCodesToTypeNames->count(dataType))
      return (*hashCodesToTypeNames)[dataType];
    else
      return "UNKNOWN TYPE";
  }

  static struct HashCodesToTypeNamesDeleter {
    ~HashCodesToTypeNamesDeleter() {
      delete hashCodesToTypeNames;
    }
  } hashCodesToTypeNamesDeleter;



  std::vector<float> toFloatVector(const boost::any& any) {
      std::vector<float> floats;

    if (any.type() == typeid(std::vector<cv::Point2f>)) {
      const std::vector<cv::Point2f>& points = boost::any_cast<std::vector<cv::Point2f> >(any);
      floats = std::vector<float>(points.size() * 2);
      for (size_t i = 0; i < points.size(); ++i) {
        floats[2*i] = points[i].x;
        floats[2*i+1] = points[i].y;
      }
      return floats;
    }
    else if (any.type() == typeid(cv::Mat)) {
      const cv::Mat& m = boost::any_cast<cv::Mat>(any);
      const uchar* ucharp = m.ptr<uchar>();
      const char* charp = m.ptr<char>();
      const short* shortp = m.ptr<short>();
      const unsigned short* ushortp = m.ptr<unsigned short>();
      const int* intp = m.ptr<int>();
      const float* floatp = m.ptr<float>();
      const double* doublep = m.ptr<double>();

      switch(m.depth()) {
      case CV_8U:
        floats = vector<float>(ucharp, ucharp + m.total());
        break;

      case CV_8S:
        floats = vector<float>(charp, charp + m.total());
        break;

      case CV_16U:
        floats = vector<float>(ushortp, ushortp + m.total());
        break;

      case CV_16S:
        floats = vector<float>(shortp, shortp + m.total());
        break;

      case CV_32S:
        floats = vector<float>(intp, intp + m.total());
        
      case CV_32F:
        floats = vector<float>(floatp, floatp + m.total());
        break;

      case CV_64F:
        floats = vector<float>(doublep, doublep + m.total());
        break;

      default:
      throw std::invalid_argument("Unsupported matrix element type.");
      }
    }

    return floats;
  }



  void convertColourInPlace(cv::Mat& target, int colourCode) {
    cv::Mat temp;
    cv::cvtColor(target, temp, colourCode);
    target = temp;
  }



  std::vector<std::string> readFileLines(const std::string& filename) {
    std::ifstream ifs(filename);
    std::string s;
    std::vector<std::string> vec;
    while (getline(ifs, s))
      vec.push_back(s);

    return vec;
  }



  cv::Mat extractOuterContour(const cv::Mat& m, int type) {
    assert(m.type() == CV_8UC1 || m.type() == CV_8UC3);
    assert(type == CV_8U || type == CV_32F);
    cv::Mat temp;
    if (m.type() == CV_8UC3)
      ColourSpace::GREY->convertColour(m, temp);
    else
      temp = m;
    cv::Mat temp2;
    threshold(temp, temp2, 1.0, type == CV_8U ? 255. : 1.0, cv::THRESH_BINARY);

    assert(temp2.type() == CV_8UC1);

    vector< vector<Point> > contours;
    cv::findContours(temp2, contours, CV_RETR_EXTERNAL, CV_CHAIN_APPROX_SIMPLE);

    size_t largestContourSize = 0;
    size_t largestContourIndex = 0;
    for (size_t i = 0; i < contours.size(); ++i) {
      if (contours[i].size() > largestContourSize) {
        largestContourSize = contours[i].size();
        largestContourIndex = i;
      }
    }

    if (largestContourIndex + 1 < contours.size())
      contours.erase(contours.begin() + largestContourIndex + 1, 
                     contours.end());
    if (largestContourIndex > 0)
      contours.erase(contours.begin(), contours.begin() + largestContourIndex);

    temp = cv::Mat(m.size(), type == CV_8U ? CV_8UC1 : CV_32FC1, cv::Scalar::all(0));
    drawContours(temp, contours, -1, type == CV_8U ? cv::Scalar::all(255) :
                 cv::Scalar::all(1.0), 1);
    return temp;
  }



  cv::Mat sideBySide(const cv::Mat& lh, const cv::Mat& rh) {
    assert(lh.type() == rh.type());
    cv::Mat result(std::max(lh.rows, rh.rows), lh.cols + rh.cols, lh.type(), 
                   cv::Scalar::all(0));
    cv::Mat temp = result.colRange(0, lh.cols);
    lh.copyTo(temp);
    temp = result.colRange(lh.cols, result.cols);
    rh.copyTo(temp);
    return result;
  }



  cv::Mat translate(const cv::Mat& m, int tx, int ty) {
    assert(m.type() == CV_8UC3);
    cv::Mat temp(m.size(), m.type());
    for (int i = 0; i < temp.rows; ++i)
      for (int j = 0; j < temp.cols; ++j)
        if (i - ty < 0 || j - tx < 0 || 
            i - ty >= temp.rows || j - tx >= temp.cols)
          temp.at<cv::Vec3b>(i,j) = cv::Vec3b(0,0,0);
        else
          temp.at<cv::Vec3b>(i,j) = m.at<cv::Vec3b>(i-ty, j-tx);

    return temp;
  }



  // pads the matrix by 50 % and aligns its dimensions to be a multiple of 16
  cv::Mat padBy50PercentAndAlign16(const cv::Mat& m) {
    int newWidth = m.cols * 3 / 2;
    if (newWidth % 16 > 0)
      newWidth += 16 - newWidth % 16;

    int newHeight = m.rows * 1.5;
    if (newHeight % 16 > 0)
      newHeight += 16 - newHeight % 16;

    cv::Mat newMat(newHeight, newWidth, m.type(), cv::Scalar::all(0));
    cv::Mat center = newMat(cv::Rect(m.cols/4, m.rows/4, m.cols, m.rows));
    m.copyTo(center);
    return newMat;
  }



  cv::Rect padBy50PercentAndAlign16(const cv::Rect& r) {
    int newWidth = r.width * 3 / 2;
    if (newWidth % 16 > 0)
      newWidth += 16 - newWidth % 16;
    int widthDifference = newWidth - r.width;

    int newHeight = r.height * 3 / 2;
    if (newHeight % 16 > 0)
      newHeight += 16 - newHeight % 16;
    int heightDifference = newHeight - r.height;
    return cv::Rect(r.x - widthDifference / 2,
                    r.y - heightDifference / 2,
                    newWidth, newHeight);
  }
}
