#ifndef SLMOTION_UTIL
#define SLMOTION_UTIL

#include "EafDocument.hpp"
#include "Component.hpp"
#include "Pdm.hpp"
#include "Filter.hpp"
#include "ColourSpace.hpp"
#include "exceptions.hpp"
#include <memory>
#include "opencv.hpp"
#include <boost/algorithm/string.hpp>
#include <typeinfo>

#include "TrackedPoint.hpp"

#include <fstream>

namespace slmotion {
  /**
   * Converts the matrix from one colour space to another in-place (at
   * least from the user's point of view, a temporary matrix may be
   * constructed internally)
   */
  void convertColourInPlace(cv::Mat& target, int colourCode);



  /**
   * Starts a global timer
   */
  void timerOn();

  /**
   * Stops the global timer.
   *
   * @return The number of seconds since the last call to timerOn().
   */
  time_t timerOff();

  /**
   * Prints the given time to the given ostream.
   */
  void printTime(std::ostream& os, time_t time);

  /**
   * Given a type_info hash code, attempts to guess the type name or
   * returns "UNKNOWN TYPE" if the type is not known
   */
  std::string hashCodeToTypeName(size_t dataType);

  void addHashCodeToTypeName(const std::type_info& typeInfo);

  template<typename T>
  void addHashCodeToTypeName() {
    addHashCodeToTypeName(typeid(T));
  }


  /**
   * Attempts to locate the component of given type from the component 
   * pointer vector
   */
  template<typename T>
  T* findComponentNoThrow(std::vector<std::shared_ptr<Component>>& comps) {
    auto it = std::find_if(comps.begin(), comps.end(),
                           [](std::shared_ptr<Component>& comp) {
                             return dynamic_cast<T*>(comp.get()) != NULL;
                           });
    if (it == comps.end())
      return NULL;

    return dynamic_cast<T*>(it->get());
  }


  /**
   * Attempts to locate the component of given type from the component 
   * pointer vector
   */
  template<typename T>
  T* findComponent(std::vector<std::shared_ptr<Component>>& comps) {
    T* t = findComponentNoThrow<T>(comps);
    if (!t)
      throw AnalysisException("Requested a component that does not "
                              "exist.");

    return t;
  }



  /**
   * Stores the name of the executable by stripping the directory
   * information from argv[0]. Note that this function should be
   * called by main().
   */
  void storeExecutableName(const std::string& executableName);



  /**
   * Prints a formatted warning to stderr
   */
  void printWarning(const std::string& message);



  /**
   * Prints the point somehow
   */
  std::ostream& operator<<(std::ostream& ofs, const TrackedPoint& p);



  /**
   * Given the parameters, performs the coordinate transformation from
   * Euclidean pixel co-ordinates into body-specific Euclidean
   * co-ordinates wrt.  the throat and the horizontal width of the
   * body.
   *
   * @param pixelCoordinates A two-dimensional pixel coordinate pair
   * @param bodyWidth Width of the body (in pixels)
   * @param throatLocation Location of the throat (in pixel coordinates)
   *
   * @return The body coordinates
   */
  inline cv::Point2d transformPixelToBodyCoordinates(const cv::Point2d& pixelCoordinates, 
                                                     double bodyWidth, 
                                                     const cv::Point2d& throatLocation) {
    assert(bodyWidth > 0);
    return (pixelCoordinates - throatLocation) * (1./bodyWidth);
  }



  /**
   * Given the parameters, performs the inverse coordinate
   * transformation from body-specific Euclidean co-ordinates wrt.
   * the throat and the horizontal width of the body into Euclidean
   * pixel co-ordinates.
   *
   * @param bodyCoordinates A two-dimensional body coordinate pair
   * @param bodyWidth Width of the body (in pixels)
   * @param throatLocation Location of the throat (in pixel coordinates)
   *
   * @return The pixel coordinates
   */
  inline cv::Point2d transformBodyToPixelCoordinates(const cv::Point2d& bodyCoordinates, 
                                                     double bodyWidth, 
                                                     const cv::Point2d& throatLocation) {
    assert(bodyWidth > 0);
    return bodyCoordinates*bodyWidth + throatLocation;
  }



  class BodyToPixelCoordinateTransformer;

  /**
   * This class can be used as a functor for performing co-ordinate
   * transformations from Euclidean pixel co-ordinates into
   * body-specific Euclidean co-ordinates wrt.  the throat and the
   * horizontal width of the body.
   */
  class PixelToBodyCoordinateTransformer {
  public:
    /**
     * Constructs the transformer
     *
     * @param bodyWidth Width of the body (in pixels)
     * @param throatLocation Throat location (in pixel coordinates)
     */
    PixelToBodyCoordinateTransformer(double bodyWidth, 
                                      const cv::Point2d throatLocation) :
      bodyWidth(bodyWidth), throatLocation(throatLocation) {}

    PixelToBodyCoordinateTransformer(const BodyToPixelCoordinateTransformer&);

    /**
     * Performs the transformation
     */
    inline cv::Point2d operator()(const cv::Point2d& pixelCoordinates) const {
      return transformPixelToBodyCoordinates(pixelCoordinates,
                                             bodyWidth, throatLocation);
    }


    PixelToBodyCoordinateTransformer(const PixelToBodyCoordinateTransformer&) = default;
    PixelToBodyCoordinateTransformer& operator=(const PixelToBodyCoordinateTransformer&) = default;

    friend class BodyToPixelCoordinateTransformer;

  private:
    double bodyWidth;
    cv::Point2d throatLocation;
  };



  class BodyToPixelCoordinateTransformer {
  public:
    /**
     * Constructs the transformer
     *
     * @param bodyWidth Width of the body (in pixels)
     * @param throatLocation Throat location (in pixel coordinates)
     */
    BodyToPixelCoordinateTransformer(double bodyWidth, 
                                      const cv::Point2d throatLocation) :
      bodyWidth(bodyWidth), throatLocation(throatLocation) {}

    BodyToPixelCoordinateTransformer(const PixelToBodyCoordinateTransformer&);

    /**
     * Performs the transformation
     */
    inline cv::Point2d operator()(const cv::Point2d& bodyCoordinates) const {
      return transformBodyToPixelCoordinates(bodyCoordinates, 
                                             bodyWidth, throatLocation);
    }


    BodyToPixelCoordinateTransformer(const BodyToPixelCoordinateTransformer&) = default;
    BodyToPixelCoordinateTransformer& operator=(const BodyToPixelCoordinateTransformer&) = default;

    friend class PixelToBodyCoordinateTransformer;

  private:
    double bodyWidth;
    cv::Point2d throatLocation;
  };



  /**
   * Prints the matrix into the given output stream
   */  
  template<typename tp>
  inline std::ostream& operator<<(std::ostream& os, const cv::Mat_<tp>& m) {
    for (int i = 0; i < m.rows; ++i) {
      os << "[";
      for (int j = 0; j < m.cols; ++j)
        os << " " << m(i, j);
      os << " ]" << std::endl;
    }
    return os;
  }

  template<int k>
  std::ostream& operator<<(std::ostream& os, const cv::Vec<uchar, k>& v) {
    os << "[";
    for (int i = 0; i < k; ++i) 
      os << " " << (int)v[i];

    os << " ]";
    return os;
  }

  template<typename tp, int k>
  std::ostream& operator<<(std::ostream& os, const cv::Vec<tp, k>& v) {
    os << "[";
    for (int i = 0; i < k; ++i) 
      os << " " << v[i];
    os << " ]";
    return os;
  }


  template<typename T, typename S>
  std::ostream& operator<<(std::ostream& os, const std::pair<T, S>& pair) {
    return os << "[" << pair.first << "," << pair.second << "]";
  }

  template<>
  inline std::ostream& operator<<(std::ostream& os, const cv::Mat_<uchar>& m) {
    for (int i = 0; i < m.rows; ++i) {
      os << "[";
      for (int j = 0; j < m.cols; ++j)
        os << " " << static_cast<int>(m(i, j));
      os << " ]" << std::endl;
    }
    return os;
  }

  //  std::ostream& operator<<(std::ostream& os, const cv::Mat& m);



  std::ostream& operator<<(std::ostream& os, const Pdm::PoseParameter& pose);



  /**
   * Prints a textual representation of the rectangle in the following 
   * format:
   * (x0,y0)->(x1,y1)
   *   where
   *     x0 is the x coordinate of the upper left corner
   *     y0 is the y coordinate of the upper left corner
   *     x1 is the x coordinate of the upper left corner
   *     y1 is the y coordinate of the upper left corner
   */
  std::ostream& operator<<(std::ostream& os, const cv::Rect& r);



  /**
   * Prints a textual representation of the size object in the following 
   * format:
   * (wxh)
   *   where
   *     w is the width
   *     h is the height
   */
  std::ostream& operator<<(std::ostream& os, const cv::Size& s);



  template<typename T>
  void print(const std::vector<T>& v, std::ostream& os = std::cout) {
    for (auto it = v.begin(); it != v.end(); ++it)
      os << *it << std::endl;
  }






  /**
   * Returns a random colour of given number of components, each sampled 
   * from a uniform distribution and is in [0,256)
   */
  template<int N>
  cv::Vec<uchar, N> getRandomColour() {
    cv::Vec<uchar, N> v;
    for (size_t i = 0; i < N; ++i)
      v[i] = cv::theRNG().uniform(0,256);
    return v;
  }



  /**
   * A conversion function for vector to bool conversion so that vectors can
   * also be used as bools, just like ordinary numerical data types.
   *
   * @param v Input vector
   *
   * @return False if v[0] == v[1] == ... v[N] == 0, true otherwise
   */
  template < typename T, int N >
  bool toBool(const cv::Vec<T,N>& v) {
    for (int i = 0; i < N; ++i)
      if (v[i])
        return true;
    return false;
  }

  
  



  // template<typename T>
  // inline T max(const T& a, const T& b, const T& c) {
  //   return std::max(std::max(a, b), c);
  // }

  // template<typename T>
  // inline T min(const T& a, const T& b, const T& c) {
  //   return std::min(std::min(a, b), c);
  // }


  /**
   * This is a kludge. Instead of properly passing the information, just 
   * use this C style function to get argv whenever needed.
   */
  void setArgV(int argc, char** argv);
  
  /**
   * This is a kludge. Instead of properly passing the information, just 
   * use this C style function to get argv whenever needed.
   */
  std::string getArgV();



  /**
   * Draws a polygon of one or more contours
   */
  void drawPolygon(cv::Mat& target, 
                   const std::vector < std::vector<cv::Point> >& contours,
                   const cv::Scalar colour = cv::Scalar::all(255));




  /**
   * Draws a polygon of exactly one contour
   */
  void drawPolygon(cv::Mat& target, const std::vector<cv::Point>& contour,
                   const cv::Scalar colour = cv::Scalar::all(255),
                   const cv::Point& translation = cv::Point());



  /**
   * Returns the file extension for the given file (assuming the file ends
   * in .[a-z0-9]{3,4})
   */
  std::string getExtension(const std::string& filename); 



  /**
   * Returns true if the file exists and is readable
   */
  inline bool checkIfFileExists(const std::string& filename) {
    return std::ifstream(filename).good();
  }



  /**
   * Copies all the elements from the source container into the target 
   * container, using the given functor to transform elements
   *
   * Elements are inserted, not overwritten.
   *
   * @param inputContainer Source container
   * @param outputContainer Output container (may be empty)
   * @param transformer A unary function that transforms elements to a type
   * accepted by the output container
   */
  template<typename INPUT_CONTAINER, typename OUTPUT_CONTAINER,
           typename TRANSFORMER>
  auto insert_transform (const INPUT_CONTAINER& inputContainer,
                         OUTPUT_CONTAINER& outputContainer,
                         TRANSFORMER transformer)->
    decltype(std::inserter(outputContainer, outputContainer.begin())) {
    return std::transform(inputContainer.begin(), inputContainer.end(),
                          std::inserter(outputContainer, 
                                        outputContainer.begin()),
                          transformer);
  }



  /**
   * Returns true if all elements of the matrix are exactly equal to the 
   * value given
   *
   * @param img Input image
   * @param value Value to compare each element to. Must be castable as a
   * double
   *
   * @return True iff each element in img == value
   */
  template<typename T>
  bool allEqual(const cv::Mat& img, T value) {
    return checkRange(img, true, NULL, value, value+1.);
  }



  /**
   * Expands something to a vector of floats (e.g. a vector of points or a matrix)
   */
  std::vector<float> toFloatVector(const boost::any& any);



  /**
   * Returns true if the matrices have equal dimensions and type,
   * and if their data is equal. Empty matrices are considered equal.
   */
  bool equal(const cv::Mat& a, const cv::Mat& b);



  /**
   * Returns true if the matrices have equal dimensions and type,
   * and if their data is equal. Empty matrices are considered equal.
   *
   * The difference with the operation above is that data comparison
   * is done using memcmp (instead of evaluating element-by-element).
   */
  bool memEqual(const cv::Mat& a, const cv::Mat& b);



  /**
   * Returns true if the matrices have equal dimensions and type, and if
   * their data is almost equal at the precision specified by the value of
   * epsilon, i.e. for each (i,j) in X, Y,  |X_(i,j) - Y_(i,j)| < epsilon
   */
  bool almostEqual(const cv::Mat& X, const cv::Mat& Y, 
                   double epsilon = 1.0e-5);



  /**
   * Given a point p = (x,y), and a rectangle r with upper left point at 
   * (x0,y0) and lower right point at (x1,y1), returns a point that is 
   * strictly within the rectangle as specified by the Rect class (i.e. 
   * upper and left boundaries are inclusive, right and bottom boundaries
   * exclusive). 
   *
   * @return Point (max(min(x,x1-1),x0), max(min(y,y1),y0))
   */
  template<typename T>
  cv::Point_<T> forcePointIntoRect(const cv::Point_<T>& p, 
                                   const cv::Rect& r) {
    T x = p.x;
    T y = p.y;
    x = std::min(x, static_cast<T>(r.br().x-1));
    y = std::min(y, static_cast<T>(r.br().y-1));
    x = std::max(x, static_cast<T>(r.tl().x));
    y = std::max(y, static_cast<T>(r.tl().y));
    return cv::Point_<T>(x,y);
  }



  /**
   * Reads the context of a text file where each line contains entry 
   * formatted as follows:
   *
   * filename1.ext filename2.ext <unsigned integer>
   *
   * Each entry is interpreted as follows:
   * filename1 is assumed to be a video file containing regular video
   * filename2 is assumed to be a video file containing binary ground truths
   * for the preceeding file; the groundtruth file must not exceed the first
   * video in length
   * the number is the number of annotated frames, beginning from frame 0, 
   * i.e. the number of frames to process
   *
   * The function goes through each entry, extracts each frame up to the 
   * maximum number for the entry, and stores the read matrices to the given
   * vectors (any previous content will be cleared).
   *
   * @param prefix Prefix that is used before each filename, i.e. given 
   * prefix pfx, and a filename file.ext, the actual filename that would be
   * used would become pfxfile.ext, e.g. assuming pfx = /home/joesixpack/,
   * and file video.avi, file named /home/joesixpack/video.avi would be 
   * read.
   */
  void readFramesAndGT(const std::string& videoGtFile, 
                       const std::string& prefix,
                       std::vector<cv::Mat>& outFrames,
                       std::vector<cv::Mat>& outGroundTruths);



  /**
   * Interpolates the assumedly closed point chain, such that simple
   * chain approximation (where straight line segments are not stored)
   * is converted into a point representation with no chain
   * approximation
   *
   * I.e. for a representation of points (p0,p1,...,pn) where (pj,pi)
   * need not be adjacent, and where there is an unmarked straight
   * line segment going from pj to pi, returns an explicit
   * representation (p'0,p'1,...,p'm), m >= n, such that all (p'j,p'i)
   * are adjacent and the contour -- or whatever arbitrary line --
   * shall remain unchanged.
   *
   * Note that it is assumed that there is a line going from pn to p0
   * so that the curve forms a closed loop.
   *
   * This function is intended to be used in conjunction with
   * cv::findContours.
   *
   * @param src Source chain (assumed to be CV_CHAIN_APPROX_SIMPLE coded)
   * @return The same chain but coded using CV_CHAIN_APPROX_NONE.
   */
  std::vector<cv::Point> chainApproxSimpleToChainApproxNone(const std::vector<cv::Point>& src);



  /**
   * Creates a custom colour space from a string in the following format:
   * custom:<string>,<string>,...
   */
  CustomColourSpace createCustomColourSpace(const std::string& s);



  std::vector<std::shared_ptr<PostProcessFilter>> parseSkinPostprocess(const std::string& s);



  template<typename T, typename A1, typename A2 = A1, 
           typename A3 = A2, typename A4 = A3>
  T createFromTokens(const std::string& src,
                     const std::string& separator) {
    std::vector<std::string> tokens;
    boost::split(tokens, src, boost::is_any_of(separator));
    if (tokens.size() != 4)
      throw ConfigurationFileException("Must receive four comma-separated integer arguments.");
    A1 a = boost::lexical_cast<int>(tokens[0]);
    A2 b = boost::lexical_cast<int>(tokens[1]);
    A3 c = boost::lexical_cast<int>(tokens[2]);
    A4 d = boost::lexical_cast<int>(tokens[3]);
    
    return T(a,b,c,d);
  }



#ifdef __GNUG__
  /**
   * Returns a demangled string (GCC specific!)
   *
   * The code is from here: http://stackoverflow.com/questions/281818/unmangling-the-result-of-stdtype-infoname/4541470#4541470
   */
  std::string string_demangle(const char* name);
#endif // __GNUG__



  /**
   * Dumps the internal document data in XML format into the target standard
   * C FILE* stream.
   *
   * @param file Target output FILE*
   * @param doc Document reference
   */
  void debugDump(FILE* file, const EafDocument& doc);



  /**
   * Dumps the internal document data in XML format into the target output
   * stream
   *
   * @param os Target output stream
   * @param doc Document reference
   */
  void debugDump(std::ostream& os, const EafDocument& doc);



  /**
   * Saves the internal document in XML format in the target file
   *
   * @param filename Target filename
   * @param doc Document reference
   */
  void debugSaveFile(const char* filename, const EafDocument& doc);



  /**
   * Returns the canonical path for the given relative path.
   *
   * WARNING: This code is potentially unsafe (see man 3 realpath)
   */
  std::string getCanonicalPath(const std::string& relativePath);



  std::string anyToString(const boost::any& any);



  /**
   * Returns true if the given string matches the given regex (exactly).
   */
  bool matchRegex(const std::string& regex, const std::string& string);



  /**
   * Given a depth matrix of type CV_16SC1, converts it to a CV_8UC3
   * matrix where the grey-scale values have been scaled wrt. maxima
   * and minima in the frames
   */
  cv::Mat convertToMarkusFormat(const cv::Mat& depthMat);



  /**
   * Returns the installation prefix by inspecting environment
   * variables and compile-time information
   */
  std::string getInstallPrefix();

  /**
   * Returns the installation prefix of IntraFace
   */
  std::string getIntrafacePrefix();

  /**
   * Replaces according to the given regex.
   */
  std::string replaceRegex(const std::string& regex, const std::string& replacement,
                           const std::string& string);



  /**
   * Returns true if the two vectors are identical
   */
  template<typename T, int N>
  bool operator==(const cv::Vec<T,N>& v1, const cv::Vec<T,N>& v2) {
    for (int i = 0; i < N; ++i)
      if (v1[i] != v2[i])
        return false;
    return true;
  }



  /**
   * Converts OpenCV matrix type integers to strings (e.g. CV_8UC3)
   */
  std::string cvMatTypeToString(int type);



  /**
   * Reads all the lines in a file and returns them as a vector
   */
  std::vector<std::string> readFileLines(const std::string& filename);



  /**
   * Extracts the outline of the image. Returns a binary matrix.
   *
   * @type Optional datatype (CV_8U or CV_32F)
   */
  cv::Mat extractOuterContour(const cv::Mat& m, int type = CV_8U);



  /**
   * Translates the image by tx and ty, filling missing pixels with
   * zeros, and discarding any pixels going outside the image borders.
   */
  cv::Mat translate(const cv::Mat& m, int tx, int ty);



  /**
   * Produces a new image where the two images have been joined side by side.
   */
  cv::Mat sideBySide(const cv::Mat& lh, const cv::Mat& rh);



  /**
   * pads the matrix by 50 % and aligns its dimensions to be a multiple of 16
   */
  cv::Mat padBy50PercentAndAlign16(const cv::Mat& m);



  /**
   * pads the rectangle as above
   */
  cv::Rect padBy50PercentAndAlign16(const cv::Rect& r);
}



// the cache uses ints as keys such that
// a value is stored as
// (v0 << 16) + (v1 << 8) + v2
//   where each vi is 8 bits long, the integer type needs to be at least 32
// bits long.
static_assert(sizeof(int) >= 4, "int must be at least 32 bits long");

#endif
