function main(run_index)
% optimization before sampling requires the Sundials ODE solver (http://computation.llnl.gov/projects/sundials)
% THe implementation is tested with Sundials 2.5.0

% run name
r_name = 'pMCMCresult';

% set random seed
rng(123*run_index);

% load specification for alternative models and input data
load MODELS; load DATA;

% independent chains for different models (following the run index)
n_parallel_chains    = 4;
n_alternative_models = length(MODELS);
model_index          = repmat(1:n_alternative_models,n_parallel_chains,1);
model_run_index      = repmat(1:n_parallel_chains,n_alternative_models,1)';

display(['Start sampling','_m',int2str(model_index(run_index)),'_c',int2str(model_run_index(run_index))]);

% active parameters in the selected model
theta_ind = MODELS{model_index(run_index)};

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% define log-likelihood and prior
pMCMC_params.log_likelihood = @(theta)logLH(theta,theta_ind,DATA);
pMCMC_params.log_prior      = @(theta)logPRIOR(theta);

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% prepare input for population MCMC (general parameters)
beta_N                 = 30;                              % number of temperatures
pMCMC_params.beta      = ((0:(beta_N-1))./(beta_N-1)).^5; % annealing schedule
pMCMC_params.d         = length(theta_ind);               % dimension of param. space

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% initial prop. dists
c = 0.1*linspace(0.1,0.01,beta_N);
R = zeros(pMCMC_params.d,pMCMC_params.d,length(pMCMC_params.beta));
for j = 1:length(pMCMC_params.beta)
    R(:,:,j) = c(j) * eye(pMCMC_params.d);
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% optimize the initial parameter values in each temperature
% if they don't already exist

filename = ['optimized_init_theta_model_',int2str(model_index(run_index))];

if exist([filename,'.mat'],'file')
    disp('Load initial parameters.')
    load(filename);
else
    N_optim_rounds = 100;
    
    if(model_index(run_index) == 1)
        model_index(run_index)
        model = construct_model_1;
    elseif(model_index(run_index) == 2)
        model_index(run_index)
        model = construct_model_2;
    else
        model_index(run_index)
        model = construct_model_3;
    end
    
    INIT_THETA = ms_optim(pMCMC_params.beta,model,DATA,N_optim_rounds);
    theta  = INIT_THETA(end,:);
    save(filename,'INIT_THETA');
    save(['theta_map',int2str(model_index(run_index))],'theta');
end

theta_opt = [randn(1,pMCMC_params.d); INIT_THETA];

% set random seed
rng(123*run_index);

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% RUN BURN-IN with proposal distribution adaption
pMCMC_params.N_samples    = 100000;                 % total number of samples including burn-in. Set to 100000 to reproduce results in the article
pMCMC_params.N_metr       = 1;                      % number of local moves at each metropolis round
pMCMC_params.N_subsamp    = 10;                     % interval for sub-sampling
pMCMC_params.N_burn_in    = pMCMC_params.N_samples; % number of burn-in samples
pMCMC_params.N_save       = 10000;                  % interval for saving results, 10000
pMCMC_params.N_diag       = 100001;                 % interval for showing diagnostics
pMCMC_params.N_diag_start = 1;                      % start to show diag at this sample
pMCMC_params.N_adapt      = 7500;
pMCMC_params.R            = R;
pMCMC_params.lambda       = 2/3;                    % probability of local move
pMCMC_params.cur_theta    = theta_opt;
pMCMC_params.name         = [r_name,'burnin','_m',int2str(model_index(run_index)),'_c',int2str(model_run_index(run_index))]; % name for the result file
% run
tic
result = pMCMC(pMCMC_params);
toc
close all;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% RUN ACTUAL SAMPLING with a fixed proposal distribution
pMCMC_params.N_samples    = 2500000;                  % total number of samples including burn-in. Set to 2500000 to reproduce results in the article
pMCMC_params.N_metr       = 1;                        % number of local moves at each metropolis round
pMCMC_params.N_subsamp    = 1000;                     % interval for sub-sampling
pMCMC_params.N_burn_in    = 0;                        % number of burn-in samples, no adaption
pMCMC_params.N_save       = 100000;                   % interval for saving results, 100000
pMCMC_params.N_diag       = pMCMC_params.N_samples+1; % interval for showing diagnostics
pMCMC_params.N_diag_start = 1;                        % start to show diag at this sample
pMCMC_params.N_adapt      = 3000;
pMCMC_params.R            = result.R;                 % prop. dists
pMCMC_params.lambda       = 2/3;
pMCMC_params.cur_theta    = result.final_state;       % the final state of the adaptive burn-in run
pMCMC_params.name         = [r_name,'_m',int2str(model_index(run_index)),'_c',int2str(model_run_index(run_index))]; % name for the result file
% run
pMCMC(pMCMC_params);

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

end