/*************************************************************************
 * Yeast colony measurement of gridded colonies 
 * Author: Adrian Scott
 * Date created: April 2016
 * Date modified: January 24, 2017 (Added informative comments and variable names).
 *  
 * DESCRIPTION
 * This script measures the area of yeast colonies that have
 * been arrayed on solid media in a specific pattern, imaged in a specific 
 * pattern, and organized with a specific file naming scheme.
 * 
 * REQUIREMENTS
 * This script requires nothing more than an installation of NIH's ImageJ or equivalent
 * (such as FIJI).
 * 
 * COLONY ARRAYING
 * The colonies in this experiment have been seeded onto solid agar media
 * in alternating positions of an SLAS format 96-well* plate.
 *    *Caveat: we don't actually use a 96-well plate, we use a single well
 *    OmniTray filled with solid media. A cell sorter is instructed to sort
 *    single cells into a "96-well plate", which has the effect of putting a
 *    nice grid of cells onto our agar plate.
 * 
 * The pattern of colonies is in the following diagram. An 'X' represents a colony, 
 * while a '.' is a blank space. This fills an 8x12 grid at standard 96-well spacing
 * (9mm column spacing, 9mm row spacing).
 * 
 *   _________________________
 *  | X . X . X . X . X . X . |
 *  | . X . X . X . X . X . X |
 *  | X . X . X . X . X . X . |
 *  | . X . X . X . X . X . X |
 *  | X . X . X . X . X . X . |
 *  | . X . X . X . X . X . X |
 *  | X . X . X . X . X . X . |
 *  | . X . X . X . X . X . X |
 *   _________________________
 *  
 * IMAGING REGIONS
 * Each plate is divided into sixteen imaging regions. Each region covers
 * 6 well positions (2 rows by 3 columns). Because of our checkerboard
 * pattern (detailed above), there are only 3 colonies per image.
 * For legacy reasons, these regions have unintuitive names, and are 
 * layed out like so:
 * 
 *   1A  |  2A  |  3A  |  4A
 *   ------------------------
 *   1B  |  2B  |  3B  |  4B
 *   ------------------------
 *   5A  |  6A  |  7A  |  8A
 *   ------------------------
 *   5B  |  6B  |  7B  |  8B
 *   
 *  Note that another consequence of the checkerboard pattern is that
 *  regions with an odd column number have colonies in this pattern:
 *  
 *   X . X
 *   . X .
 *   
 *   while regions with an even column number have colonies in this pattern:
 *   
 *   . X .
 *   X . X
 *   
 *   These possible colony locations within an imaging region are referred to as 'patches'.
 *   They are numbered as follows:
 * 
 *   1 . 3		(only present in regions with odd numbers)
 *   . 2 .
 *   
 *   . 5 .		(only present in regions with even number")
 *   4 . 6
 *   
 * FILE NAMING
 * The naming scheme is as follows:
 * -Each plate of solid media is given a name, and all images of that plate
 *  begin with the plate name.
 * -Next in the file name is the imaging region, detailed above  
 * -The date and time the image was taken are recorded in the file name.
 * -Each image is given a serial number, starting from 0. 
 * 
 * The full file name is of the format:
 * PLATE-REGION_YEAR-MONTH-DAY_HOUR-MINUTE-SECOND_SERIAL.JPG
 * For example:
 * YAD145B-2B_2016-04-28_16-38-10_6106.jpg
 * 
 * INPUT
 * The input for this script is a directory which contains subdirectories,
 * each of which holds the full timecourse image set for a specific region
 * on one plate. These subdirectories are typically named PLATE_REGION, but
 * that is not necessary. There is the ability to set a prefix of directories
 * to ignore - in the file below, this is demonstrated with the string "SWATCH"
 * which indicated regions that did not contain colonies, but instead contained
 * images of a calibration swatch.
 * 
 * The input is specified by changing the "topdir" variable to point to the
 * directory that contains all the imaging region subdirectories.
 * 
 * OUTPUT
 * The output of this script is a 'Measurements.csv' file in each of the imaging
 * region subdirectories. The output file has no header. In spite of the name, it's
 * actually a tab-delimited file. The format of lines in the output is as follows: 
 * Column:		Description:
 *    0			Uninformative index from ImageJ results 
 *    1			Image name that the data was extracted from, plus which patch was measured.
 *    2			Colony area
 *    3			X coordinate of colony centroid (relative to the upper left corner of the colony's 'patch' ROI)
 *    4			Y coordinate of colony centroid (as above)
 * 
 * RUNNING
 * After specifying the topdir of the project, the script can be run at any time.
 * Load the script in ImageJ or equivalent (Fiji, etc..) and press run.
 * If imaging is still in progress, the script will look for an output file in each
 * imaging region directory and make note of the last serial number in that file. It
 * will then process only images with a higher serial number. This allows the user to
 * get partial results while the experiment is still in process.
 */



/* 
 * SETUP INFO
 * This needs to be modified for new experiments. The parts that must 
 * be changed are the 'topdir' variable, and the 'crop*' variables. The crop variables
 * just specify a region of interest (ROI) in the images and crop each image to that ROI
 * before further analysis. cropX and cropY specify the top left pixel of the ROI,
 * cropW and cropH specify the width and height of the ROI, respectively. Possible
 * locations of colonies are then computed based on the size of the ROI, and the assumption
 * that the region spans a 2-row by 3-column patch of the agar plate.
 *  
 */

setBatchMode(true);  //turn off on-screen display while processing.
//Linux example:
topdir = "/mnt/data/2016-04-27-GrowthCurves/";

//Windows example:
//topdir = "D:\\Data\\2016-04-27-GrowthCurves\\";

cropX = 634;  // X coordinate of upper left pixel for region-of-interest
cropY = 100;  // Y coordinate of the same pixel
cropW = 4323; // ROI width
cropH = 3202; // ROI height

// Array to store the _center_ coordinate of six possible colony locations in the image
patchX = newArray(6);
patchY = newArray(6);
patchW = cropW/3; // Assume 3 columns of locations in the image
patchH = cropH/2; // Assume 2 rows of locations

patchX[0] = cropW/6;
patchY[0] = cropH/4;

patchX[3] = cropW/6;
patchY[3] = 3*cropH/4;

patchX[4] = cropW/2;
patchY[4] = cropH/4;

patchX[1] = cropW/2;
patchY[1] = 3*cropH/4;

patchX[2] = 5*cropW/6;
patchY[2] = cropH/4;

patchX[5] = 5*cropW/6;
patchY[5] = 3*cropH/4;

//End setup area

// Specify only area and centroid measurements to be calculated
run("Set Measurements...", "area centroid display redirect=None decimal=3");

files = getFileList(topdir);
for (file_index=0;file_index < files.length;file_index++) {
	if (endsWith(files[file_index], "/") && !startsWith(files[file_index], "SWATCH")) {
		maxImg = -1;
		// This section (if (File.exists...) statement) allows us to run this script while imaging is still taking place,
		// without overwriting the results that we have measured thus far. Each directory (that does not
		// start with "SWATCH", which is reserved for calibration images) is checked to see if it already
		// has a Measurements.csv file. If it does, then that measurement file is parsed to find the highest
		// serial number of the images that have been measured. In the next section, only images with serial
		// numbers higher than this 'maxImg' number will be measured, and appended to the existing file.
		if (File.exists(topdir+files[file_index]+"Measurements.csv")) {
			contents = File.openAsString(topdir+files[file_index]+"Measurements.csv");
			lines=split(contents,"\n");
			for (string_index = 0; string_index < lines.length; string_index++) {
				if (lengthOf(lines[string_index]) > 3) {
					temp = split(lines[string_index], "\t");
					if (temp.length > 1) {
						imagename = temp[1];
						region_date_serial_patch = split(imagename, "_");
						serial_patch = split(region_date_serial_patch[3], ".");
						serial = serial_patch[0];
						if (parseInt(serial) > maxImg) {
							maxImg = parseInt(serial);
						}
					}
				}
			}
		}

		// Once any existing results have been parsed so they can be safely ignored, we look for new
		// images.
		subdir_files = getFileList(topdir+files[file_index]);
		for (j=0; j < subdir_files.length;j++) {
			if (endsWith(subdir_files[j], ".jpg")) { // Only concerned with image files.

				// Parse image name to see if we've already done this one
				region_date_serial = split(subdir_files[j], "_");
				serial_suffix = split(region_date_serial[3], ".");
				serial = serial_suffix[0];
				imgNumber = parseInt(serial);
				// done parsing image name. Will only proceed if serial number is higher than already measured.
				if (imgNumber > maxImg) {
					// Open the image:
					open(topdir+files[file_index]+subdir_files[j]);

					// Spit out a progress update:
					print(j/subdir_files.length*100+"% subdirectory   \t" + IJ.freeMemory() + "   \t" + (file_index/files.length*100)+"% overall  " + subdir_files[j]);

					// Get the name of the image we just opened
					a=getTitle();
					// Split the image name into [Plate-region, YYYY-MM-DD, HH-mm-SS, Serial]
					fields = split(a, "_");
					// Get the region from 'Plate-region'
					more_fields = split(fields[0], "-");
					region = more_fields[1];
					// Figure out the column number for our region
					column = parseInt(substring(region, 0, 1));

					// Cut the image down to ignore the borders of the image
					makeRectangle(cropX, cropY, cropW, cropH);
					run("Crop");
					// Make the image grayscale
					run("8-bit");
					// Threshold the image based on a setting that works well at mid-to-late
					// time points. We don't want to use an adaptive threshold for all images
					// because early images may have colonies that are so small that the
					// auto-thresholding methods don't find them. Our lighting and colony colors
					// are consistent enough that we can set one threshold for the whole dataset.
					setThreshold(60, 255);
	//				setAutoThreshold("Triangle dark");  //can be used in place of fixed threshold.
					setOption("BlackBackground", false);
					// Apply the threshold to the image, making it binary
					run("Convert to Mask");

	 				// Go through our possible colony locations - note that we're not zero indexing these
	 				// names. We subtract 1 in the coordinate arrays to handle this.
					for (patch = 1; patch < 7; patch++) {
						// We only run our analysis on patches 1,2,3 if the region is in an odd
						// column, and only patches 4,5,6 get processed if the region is in an
						// even column. See IMAGING REGIONS in the preamble for details.
						if ( (column % 2 == 1 && patch < 4) || (column % 2 == 0 && patch >= 4) ){
						// Make sure we're working with the original (but cropped) image
						selectWindow(a);
						// Copy the pre-programmed patch area for this possible colony
						makeRectangle(patchX[patch-1]-patchW/2, patchY[patch-1]-patchH/2, patchW, patchH);
						run("Duplicate...", "title=" + a + "-Patch" + patch + " duplicate");
						// Measure any binary object that is larger than 1000 px^2 and looks roughly circular.
						// Measurements will be added to the ImageJ Results panel, and saved later.
						run("Analyze Particles...", "size=1000-Infinity circularity=0.50-1.00 show=Nothing display exclude stack");
						close();
						}
					}
					// Done processing patches from this image, so we close it
					selectWindow(a);
					close();
				}
			}
			// If we actually measured something, we copy everything in the Results panel, get rid of 
			// some extra newlines, and append it to the Measurements.csv file (ImageJ will create the
			// file if it does not exist.
			if (nResults > 0) {
				String.copyResults();
				File.append(replace(String.paste, "\n\n", "\n"), topdir+files[file_index]+"Measurements.csv");
			}
			// Clear the results from this  regionn so we can move on to the next one
			run("Clear Results");
		}
	}
}

setBatchMode(false);
print("Done");