function result = pMCMC(pMCMC_params)

% functions for loglikelihood and logprior
log_likelihood = pMCMC_params.log_likelihood;
log_prior      = pMCMC_params.log_prior;

% set general parameters
beta         = pMCMC_params.beta;      % schedule
d            = pMCMC_params.d;         % dimension of the targer dist.
N_samples    = pMCMC_params.N_samples; % total number of samples including burn-in
N_metr       = pMCMC_params.N_metr;    % number of local moves at each metropolis round
N_subsamp    = pMCMC_params.N_subsamp; % interval for sub-sampling
N_burn_in    = pMCMC_params.N_burn_in; % number of burn-in samples
N_save       = min(pMCMC_params.N_samples, pMCMC_params.N_save); % interval for saving intermediate results
N_diag       = pMCMC_params.N_diag;                              % interval for showing/plotting diagnostics
N_diag_start = pMCMC_params.N_diag_start;
N_adapt      = pMCMC_params.N_adapt;                             % interval for prop. dist. adaption
name         = pMCMC_params.name;                                % name for the output file
lambda       = pMCMC_params.lambda;

% covariance for proposal distributions
R         = pMCMC_params.R; % cholesky factorization of the proposal covariance in different temperatures

% initialize vectors and matrices for internal book keeping
log_lh    = zeros(1,length(beta));  % log likelihood at the current points
cur_f     = zeros(1,length(beta));  % log target at the current points i.e. log_lh(i) + log_pr(i)
cur_theta = pMCMC_params.cur_theta; % initial state

% initialize matrices to store diagnostics and samples
n_accepted_local  = zeros(1,length(beta));
n_accepted_global = zeros(1,length(beta));
n_local_updates   = zeros(1,length(beta));
n_global_updates  = zeros(1,length(beta));
LOG_LH            = zeros(floor(N_samples/N_subsamp),length(beta));
CUR_F             = zeros(floor(N_samples/N_subsamp),length(beta));
SAMPLES           = zeros(length(beta),d,floor((N_samples - N_burn_in)/N_subsamp));
memSAMPLES        = zeros(length(beta),d,N_adapt);
max_curF          = -inf;
theta_MAP         = zeros(1,d);

% initializes counters for subsampling and samples
count            = 1;
sample_count     = 1;
mem_sample_count = 1;

% compute target vals using the initial state of the chain
cur_f(1)  = log_prior(cur_theta(1,:));      % density directly from prior
log_lh(1) = log_likelihood(cur_theta(1,:));
for j = 2:length(beta)
    log_lh_temp0 = log_likelihood(cur_theta(j,:));
    log_lh(j)    = log_lh_temp0; % update the log-likelihood value
    cur_f(j)     = beta(j)*log_lh_temp0 + log_prior(cur_theta(j,:));
end


for i = 1:N_samples
    % sample between a local move and swap
    if(rand < lambda)
        % sample the temperature for local move
        j = find(rand < (0:length(beta))./(length(beta)),1)-1;
        
        % keep track of the number of local updates in different
        % temperatures
        n_local_updates(j) = n_local_updates(j) + 1;
        
        % in the case of j=1 (i.e. beta = 0), sample directly from the prior
        if(j == 1)
            cur_theta(j,:) = randn(1,d);                    % sample from standard normal prior
            %cur_theta(j,:) = log(gamrnd(a,b,[1, d]));        % sample from gamma prior
            cur_f(j)       = log_prior(cur_theta(j,:));      % update the target value
            log_lh(j)      = log_likelihood(cur_theta(j,:)); % update the log-likelihood value
        else
            % run metropolis
            for k = 1:N_metr
                % generate a sample using normal proposal
                theta_star   = cur_theta(j,:) + randn(size(cur_theta(j,:)))*R(:,:,j);
                log_lh_temp1 = log_likelihood(theta_star);
                star_f       = beta(j)*log_lh_temp1 + log_prior(theta_star);
                % check acceptance
                if ( log(rand) < (star_f - cur_f(j)) )                    
                    cur_theta(j,:)      = theta_star;   % update theta_j by the proposed value
                    cur_f(j)            = star_f;       % update the target value
                    log_lh(j)           = log_lh_temp1; % update the log-likelihood value
                    n_accepted_local(j) = n_accepted_local(j) + 1;
                end
             end
        end
        
    else
        % SAMPLING BETWEEN THE CHAINS
        j_index = find(rand < (0:length(beta)-1)./(length(beta)-1),1);
        % keep track of global updates
        n_global_updates(j_index) = n_global_updates(j_index) + 1;
        % compute acceptance ratio
        logratio = (beta(j_index-1)*log_lh(j_index) + beta(j_index)*log_lh(j_index-1)) - (beta(j_index-1)*log_lh(j_index-1) + beta(j_index)*log_lh(j_index));

        % check acceptance
        if ( log(rand) < logratio )
            cur_f(j_index)   = beta(j_index)   * log_lh(j_index-1) + log_prior(cur_theta(j_index-1,:)); % update the target value
            cur_f(j_index-1) = beta(j_index-1) * log_lh(j_index)   + log_prior(cur_theta(j_index,:));   % update the target value
            % swap parameter vectors and log-likelihood values
            cur_theta([j_index,j_index-1],:) = cur_theta([j_index-1,j_index],:);
            log_lh([j_index, j_index-1])     = log_lh([j_index-1, j_index]);            
            n_accepted_global(j_index)       = n_accepted_global(j_index) + 1;
        end
    end
    
    % keep track of the highest cur_f value for the posterior dist.
    if(cur_f(end) > max_curF)
        max_curF  = cur_f(end);
        theta_MAP = cur_theta(end,:);
    end
    
    if(mod(i,N_subsamp) == 0)
        LOG_LH(count,:) = log_lh;
        CUR_F(count,:)  = cur_f;
        count = count + 1;
        % when the burn-in is finished, collect also samples
        if(i > N_burn_in)
            SAMPLES(:,:,sample_count) = cur_theta;
            sample_count = sample_count + 1;
        end
    end
    
    % collect memory to adapt proposals
    if(i <= N_burn_in)
        memSAMPLES(:,:,mem_sample_count) = cur_theta;
        mem_sample_count = mem_sample_count + 1;
        % during the burn-in, update proposal distributions
        if((mem_sample_count-1)==N_adapt)
            for j_adapt = 1:length(pMCMC_params.beta)
                KOVARIANSSI = 2.4^2 * cov(squeeze(memSAMPLES(j_adapt,:,:))')/d;
                [R_temp,p]  = chol(KOVARIANSSI);
                if(p == 0)
                    R(:,:,j_adapt) = R_temp;
                else
                    R(:,:,j_adapt) = 0.01 * eye(pMCMC_params.d);
                end
            end
            % empty memory when when adaption is done
            mem_sample_count = 1;
            memSAMPLES       = zeros(length(beta),d,N_adapt);
        end
    end
    
    % show diagnostics
    if(mod(i,N_diag)==0 && count >= N_diag_start)
        N_sp_rows = 7;
        N_sp_cols = 5;
        
        figure(1);
        subplot(N_sp_rows,N_sp_cols,[1 2]);
        bar(100*n_accepted_local ./ n_local_updates);box off;
        
        subplot(N_sp_rows,N_sp_cols,[4 5]);
        bar(100*n_accepted_global ./ n_global_updates);box off;
        
        subplot(N_sp_rows,N_sp_cols,[3]);
        plot(cur_f)
        
        for ind_subplot = 1:length(beta)
            subplot(N_sp_rows,N_sp_cols,ind_subplot + N_sp_cols);
            plot(N_diag_start:(count-1),CUR_F(N_diag_start:(count-1),ind_subplot));
        end        
        drawnow;
    end
    
    % save results at given intervals
    if(mod(i,N_save)==0)
        result.n_accepted_local  = n_accepted_local;
        result.n_local_updates   = n_local_updates;
        result.n_accepted_global = n_accepted_global;
        result.n_global_updates  = n_global_updates;
        result.LOG_LH            = LOG_LH(1:(count-1),:);
        result.CUR_F             = CUR_F(1:(count-1),:);
        result.final_state       = cur_theta;
        result.R                 = R;
        result.max_curF          = max_curF;
        result.theta_MAP         = theta_MAP;
        if(i > N_burn_in)
            result.SAMPLES = SAMPLES(:,:,1:(sample_count-1));
        end
        % save result
        %save(name,'result','pMCMC_params');
        save(['./posterior_analysis/result_chains/',name],'result','pMCMC_params');
    end
end

end