function [fp_area, time] = colsim(sim_params,grid,theta_micenv,theta_spatial,agar,run_name,graphics)

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Simulation parameters
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

T  = sim_params.T;
dt = sim_params.dt;
subsamp_interval      = sim_params.subsamp_interval;
subsamp_area_interval = sim_params.subsamp_area_interval;
illustrate_interval   = sim_params.illustrate_interval;
save_interval         = sim_params.save_interval;

% # of rounds in the loop
N  = sim_params.T/sim_params.dt;

% create a cell array for the results
STATE = cell([round(T/subsamp_interval),1]);

% create a vector for footprint size
fp_area = zeros(1,round(T/subsamp_area_interval));

% create a vector for output time
time = zeros(1,round(T/subsamp_area_interval));

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Grid parameters
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
h          = grid.h;
max_rad    = grid.max_rad;
max_height = grid.max_height;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Spatial model parameters
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Mass movement, pressure surface
lambda_mass = theta_spatial.lambda_mass;
th_mass     = theta_spatial.th_mass;

% Nutrient transfer
lambda_colony = theta_spatial.lambda_colony; % within colony
lambda_agar   = theta_spatial.lambda_agar;   % within agar

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Initialize the grid and state
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% grid size, domain 2*max_rad mm x 2*max_rad mm x max_height mm, each
% elementary cube is of size h^3 mm.
X  = round(max_rad/h)+1; Y = round(max_rad/h)+1; Z = round(max_height/h);

state = struct( ...
    'Mg', zeros(X,Y,Z), ... % glucose state
    'Me', zeros(X,Y,Z), ... % ethanol state
    'Mq', zeros(X,Y,Z), ... % quiescent state
    'G',  zeros(X,Y,Z), ... % glucose
    'E',  zeros(X,Y,Z)  ... % ethanol
    );

% Set initial mass distribution
state.Mg(1,1, agar.h+2) = theta_spatial.mg_init;

% Set initial glucose distribution in the agar and colony

% make a quarter circle
full_circle      = numgrid('D',2*round(max_rad/h)+1)>0;
for iiii = 1:agar.h
    state.G(:,:,iiii+1) = agar.gluc_init * full_circle(round(max_rad/h)+1:end,round(max_rad/h)+1:end);% In the agar...
end
DOMAIN_agar = state.G > 0;
state.G(1, 1, agar.h+2) = agar.gluc_init; % and in the colony.

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Simulate the time-evolution of the colony
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% initialize counters
subsample_counter      = 1;
subsample_counter_area = 1;
subsample_counter_save = 1;

for i = 1:N
    % total mass
    Mtotal     = state.Mg + state.Me + state.Mq;
    DOMAIN_col = Mtotal > 0;
    
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    % compute change
    
    % nutrient transfer
    dG_ntrans = nutrient_mov(DOMAIN_agar + DOMAIN_col,state.G,lambda_colony,lambda_agar,agar,X,Y,Z);
    dE_ntrans = nutrient_mov(DOMAIN_col,state.E,lambda_colony,lambda_agar,agar,X,Y,Z);
    
    % microenvironment
    [dMg_micr, dMe_micr, dMq_micr, dG_micr, dE_micr] = ...
        microenvironment_dynamics(DOMAIN_col,state.Mg,state.Me,state.Mq,state.G,state.E,theta_micenv.theta,theta_micenv.growth_on_ethanol,X,Y,Z);
    
    % mass movement
    if(sum(sum(sum(Mtotal))) > th_mass)
        [dMg_mass,dMe_mass,dMq_mass] = mass_mov(DOMAIN_col,Mtotal,state.Mg,state.Me,state.Mq,lambda_mass,th_mass,agar.h,X,Y,Z);
    else
        dMg_mass = 0; dMe_mass = 0; dMq_mass = 0;
    end

    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    % update
    state.Mg = state.Mg + dt*(dMg_micr + dMg_mass);
    state.Me = state.Me + dt*(dMe_micr + dMe_mass);
    state.Mq = state.Mq + dt*(dMq_micr + dMq_mass);
    state.G  = state.G  + dt*(dG_micr + dG_ntrans);
    state.E  = state.E  + dt*(dE_micr + dE_ntrans);
    
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    % record the current state
    if(mod(i*dt,subsamp_interval)==0)
        STATE{subsample_counter} = state;
        subsample_counter        = subsample_counter + 1;
        display([num2str(i*dt),' hours simulated, (', num2str(100*i*dt/T), '% of the simulation accomplished).']);
    end
    
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    % record the footprint area
    if(mod(i*dt,subsamp_area_interval)==0)
        fp_area(subsample_counter_area) = (sum(sum(squeeze(state.Mg(:,:,agar.h+2))>0))) + 3*(sum(sum(squeeze(state.Mg(2:end,2:end,agar.h+2))>0))) + (sum(sum(squeeze(state.Mg(2:end,1,agar.h+2))>0))) + (sum(sum(squeeze(state.Mg(1,2:end,agar.h+2))>0)));
        time(subsample_counter_area)    = i*dt;
        subsample_counter_area          = subsample_counter_area + 1;
    end
    
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    % illustrate colony if chosen
    if(graphics && (mod(i*dt,illustrate_interval)==0))
            illustrate_colony(state,agar,subsample_counter_area,fp_area,time);
            drawnow;
    end
    
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    % save the result
    if(mod(i*dt,save_interval)==0)
        fp_area_out  = fp_area(1:subsample_counter_area-1);
        fp_area_time = time(1:subsample_counter_area-1);
        save(run_name,'STATE','sim_params','grid','theta_micenv','theta_spatial','agar','fp_area_time','fp_area_out')
        subsample_counter_save = subsample_counter_save + 1;
    end
end

end

function dX = nutrient_mov(DOMAIN,X,lambda_colony,lambda_agar,agar,I,J,K)
% M is the cell mass matrix
% X is the nutrirent distribution
% lambda is the movement rate
% dt is the time step

% Initialize
dX = zeros(I, J, K);

% Find the indices for the domain
[coorX,coorY,coorZ] = meshgrid(1:I,1:J,1:K);
temp = DOMAIN .* coorX;
temp = temp(:);
indX = temp(temp>0);
temp = DOMAIN .* coorY;
temp = temp(:);
indY= temp(temp>0);
temp = DOMAIN .* coorZ;
temp = temp(:);
indZ = temp(temp>0);

% Nutrient movement
for u = 1:length(indX)
    % Coordinates
    i = indX(u); j = indY(u); k = indZ(u);
    
    if (i-1) == 0
        mirror_x = 2;
    else
        mirror_x = (i-1);
    end

    if (j-1) == 0
        mirror_y = 2;
    else
        mirror_y = (j-1);
    end
    
    % Compute dX
    dX(i,j,k) = ( ...
            border_check(DOMAIN(mirror_x,j,k),X(mirror_x,j,k),X(i,j,k),agar,k,lambda_colony,lambda_agar,0)   ... % left
          + border_check(DOMAIN(i+1,j,k),     X(i+1,j,k),     X(i,j,k),agar,k,lambda_colony,lambda_agar,0)   ... % right
          + border_check(DOMAIN(i,mirror_y,k),X(i,mirror_y,k),X(i,j,k),agar,k,lambda_colony,lambda_agar,0)   ... % back
          + border_check(DOMAIN(i,j+1,k),     X(i,j+1,k),     X(i,j,k),agar,k,lambda_colony,lambda_agar,0)   ... % forth
          + border_check(DOMAIN(i,j,k-1),     X(i,j,k-1),     X(i,j,k),agar,k,lambda_colony,lambda_agar,-1)  ... % down
          + border_check(DOMAIN(i,j,k+1),     X(i,j,k+1),     X(i,j,k),agar,k,lambda_colony,lambda_agar,1)   ... % up
    );
end

end

% border_check is used in nutrient_mov
function dx = border_check(domain_neighbor,x_neighbor,x,agar,k,lambda_colony,lambda_agar,vertical_direction)
    if(domain_neighbor > 0)
        % lambda_colony is effective within the colony and at the border if the movement is upwards
        if(k>(agar.h+2) || (k==(agar.h+2) && vertical_direction == 1))
            dx = lambda_colony*(x_neighbor - x);
        else
            dx = lambda_agar * (x_neighbor - x);
        end
    else
        dx = 0;
    end;
end

function [dMg,dMe,dMq] = mass_mov(DOMAIN,M,Mg,Me,Mq,lambda,th,agar_h,I,J,K)
% M is the total mass distribution

% Initialize
dMg = zeros(I,J,K);
dMe = zeros(I,J,K);
dMq = zeros(I,J,K);

% Include the neigboring points next to the border
DOMAIN(1:end-1,:,:) = DOMAIN(1:end-1,:,:) + DOMAIN(2:end,:,:);
DOMAIN(2:end,:,:)   = DOMAIN(2:end,:,:) + DOMAIN(1:end-1,:,:);
DOMAIN(:,1:end-1,:) = DOMAIN(:,1:end-1,:) + DOMAIN(:,2:end,:);
DOMAIN(:,2:end,:)   = DOMAIN(:,2:end,:) + DOMAIN(:,1:end-1,:);
DOMAIN(:,:,2:end)   = DOMAIN(:,:,2:end) + DOMAIN(:,:,1:end-1);
DOMAIN = DOMAIN > 0;

% Find the indices for the domain
[coorX,coorY,coorZ] = meshgrid(1:I,1:J,1:K);
temp = DOMAIN .* coorX;
temp = temp(:);
indX = temp(temp>0);
temp = DOMAIN .* coorY;
temp = temp(:);
indY= temp(temp>0);
temp = DOMAIN .* coorZ;
temp = temp(:);
indZ = temp(temp>0);

% To prevent the mass movement to the agar
M(:,:,agar_h+1) = M(:,:,agar_h+2);

% Compute change
for u = 1:length(indX)
    i = indX(u); j = indY(u); k = indZ(u);
    
    if (i-1) == 0
        mirror_x = 2;
    else
        mirror_x = (i-1);
    end

    if (j-1) == 0
        mirror_y = 2;
    else
        mirror_y = (j-1);
    end
    
    dMg(i,j,k) = lambda * (   F(M(i,j,k),  M(mirror_x,j,k),Mg(i,j,k),  Mg(mirror_x,j,k),th) ...
                            + F(M(i,j,k),  M(i+1,j,k),     Mg(i,j,k),  Mg(i+1,j,k),     th) ...
                            + F(M(i,j,k),  M(i,mirror_y,k),Mg(i,j,k),  Mg(i,mirror_y,k),th) ...
                            + F(M(i,j,k),  M(i,j+1,k),     Mg(i,j,k),  Mg(i,j+1,k),     th) ...
                            + F(M(i,j,k),  M(i,j,k-1),     Mg(i,j,k),  Mg(i,j,k-1),     th) ...
                            + F(M(i,j,k),  M(i,j,k+1),     Mg(i,j,k),  Mg(i,j,k+1)     ,th));
    dMe(i,j,k) = lambda * (   F(M(i,j,k),  M(mirror_x,j,k),Me(i,j,k),  Me(mirror_x,j,k),th) ...
                            + F(M(i,j,k),  M(i+1,j,k),     Me(i,j,k),  Me(i+1,j,k),     th) ...
                            + F(M(i,j,k),  M(i,mirror_y,k),Me(i,j,k),  Me(i,mirror_y,k),th) ...
                            + F(M(i,j,k),  M(i,j+1,k),     Me(i,j,k),  Me(i,j+1,k),     th) ...
                            + F(M(i,j,k),  M(i,j,k-1),     Me(i,j,k),  Me(i,j,k-1),     th) ...
                            + F(M(i,j,k),  M(i,j,k+1),     Me(i,j,k),  Me(i,j,k+1),     th));    
    dMq(i,j,k) = lambda * (   F(M(i,j,k),  M(mirror_x,j,k),Mq(i,j,k),  Mq(mirror_x,j,k),th) ...
                            + F(M(i,j,k),  M(i+1,j,k),     Mq(i,j,k),  Mq(i+1,j,k),     th) ...
                            + F(M(i,j,k),  M(i,mirror_y,k),Mq(i,j,k),  Mq(i,mirror_y,k),th) ...
                            + F(M(i,j,k),  M(i,j+1,k),     Mq(i,j,k),  Mq(i,j+1,k),     th) ...
                            + F(M(i,j,k),  M(i,j,k-1),     Mq(i,j,k),  Mq(i,j,k-1),     th) ...
                            + F(M(i,j,k),  M(i,j,k+1),     Mq(i,j,k),  Mq(i,j,k+1),     th));
end

end

function value = F(m_tot,m_tot_neighbor,m_nutrient,m_nutrient_neighbor,th)
    % compute thresholded values
    m_th          = max(0,m_tot - th);
    m_th_neighbor = max(0,m_tot_neighbor - th);
    if (m_th == m_th_neighbor)
        value = 0;
    elseif(m_th > m_th_neighbor)
        value = (m_th_neighbor - m_th) * (m_nutrient / m_tot);
    elseif(m_th_neighbor > m_th)
        value = (m_th_neighbor - m_th) * (m_nutrient_neighbor / m_tot_neighbor);
    end
end


function [dMG, dME, dMQ, dG, dE] = microenvironment_dynamics(DOMAIN,MG,ME,MQ,G,E,theta,growth_on_ethanol,I, J, K)

% Initialize
dMG = zeros(I, J, K);
dME = zeros(I, J, K);
dMQ = zeros(I, J, K);
dG  = zeros(I, J, K);
dE  = zeros(I, J, K);

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Find the indices for the domain
[coorX,coorY,coorZ] = meshgrid(1:I,1:J,1:K);
temp = DOMAIN .* coorX;
temp = temp(:);
indX = temp(temp>0);
temp = DOMAIN .* coorY;
temp = temp(:);
indY= temp(temp>0);
temp = DOMAIN .* coorZ;
temp = temp(:);
indZ = temp(temp>0);

% Compute change
for u = 1:length(indX)
    i = indX(u); j = indY(u); k = indZ(u);
    dx = rhsf_micenv...
    ([MG(i,j,k);ME(i,j,k);MQ(i,j,k);G(i,j,k);E(i,j,k)],theta,growth_on_ethanol);

    dMG(i,j,k) = dx(1);
    dME(i,j,k) = dx(2);
    dMQ(i,j,k) = dx(3);
    dG(i,j,k)  = dx(4);
    dE(i,j,k)  = dx(5);
end

end

function dx = rhsf_micenv(x,theta,growth_on_ethanol)

dx = zeros(5,1);

dx(1) =   theta(1) * x(1) * x(4) ...
        - theta(2) * (1 / (x(4) + theta(3))) * x(1) ...
        - theta(4) * x(1);

dx(2) =   theta(5) * x(2) * x(5) * growth_on_ethanol ...
        + theta(2) * (1 / (x(4) + theta(3))) * x(1) ...
        - theta(6) * x(2);
    
dx(3) =   theta(4) * x(1) + theta(6) * x(2);

dx(4) = - theta(7) * x(1) * x(4);

dx(5) =   theta(8) * x(1) * x(4)  ...
        - theta(5) * x(2) * x(5) * growth_on_ethanol;
end