%% A4_likelihood_prior
% is a function to calculate likelihood and prior value, given the sampling parameters (theta and alpha).
% The system is solved and represented in RPKM values, which are then mapped into read counts for calculating likelihood.
% The prior is calculated by the log-normally distributed parameters only.
% The commented part is the qualitative information mentioned in Section S5.
% Readers can uncommented those to incorporate it into the statistical model.

function [logL,logprior] = A4_likelihood_prior(logtheta,alpha,D,M,H)
%% sample parameters

% sampling index
switch M
    case 1, logtheta_index = [1 2 3 4     7   9 10    12    14];
    case 2, logtheta_index = [1 2 3 4   6 7   9 10    12    14];
    case 3, logtheta_index = [1 2 3 4 5   7   9 10    12    14];
    case 4, logtheta_index = [1 2 3 4 5 6 7   9 10    12    14];
    case 5, logtheta_index = [1 2 3 4     7   9 10 11 12    14];
    case 6, logtheta_index = [1 2 3 4   6 7   9 10 11 12    14];
    case 7, logtheta_index = [1 2 3 4 5   7   9 10 11 12    14];
    case 8, logtheta_index = [1 2 3 4 5 6 7   9 10 11 12    14];
end

%% likelihood and prior
if min(alpha) >= 0 & max(alpha) <= 1 % uniform prior, alpha
    
    % RPKM_x
    switch H
        case {1,2} % all replicates
            % solve MDEs
            theta = exp(logtheta);
            x = A3_MDE_output(theta,D.t,H,[]);
            % RPKM_x
            mRNA_Th0  = [ x.Th0(:,2)  , x.Th0(:,4)  ];          % STAT3_m & RORC_m of Th0
            mRNA_Th17 = [ x.Th17(:,2) , x.Th17(:,4) ];          % STAT3_m & RORC_m of Th17
            x_cont    = mRNA_Th0;                               % STAT3_m & RORC_m - controlled group, Eq. (9)
            x_vari    = (1 - alpha)*mRNA_Th0 + alpha*mRNA_Th17; % STAT3_m & RORC_m - variable group,   Eq. (10) and (11)
            x_mRNA    = [ x_cont(2:end,:) ; x_vari(2:end,:) ];  % STAT3_m & RORC_m, without initial values
            RPKM_x    = repmat(x_mRNA,[1 1 3]);                 % repeat for 3 replicates
            
%             % adding qualitative info (RORC knocked-out)
%             if mod(M,2) == 0
%                 % solve MDEs, with RORC knocked-out
%                 theta_KO = theta;
%                 theta_KO(6) = 0;
%                 x_KO = A3_MDE_output(theta_KO,D.t,H,[]); % Eq. (S18)
%                 % log-likelihood, with RORC knocked-out
%                 sigma_KO = 0.25;
%                 logL_KO = -1/2*log(2*pi*sigma_KO.^2) - (x_KO.Th17(9,2) - x.Th17(9,2)).^2 ./ (2*sigma_KO.^2); % Eq. (S19) and (S20)
%                 logL_KO = logL_KO * 3; % repeat for 3 replicates
%             else
%                 logL_KO = 0;
%             end
            
        case 3 % each replicate
            RPKM_x = zeros(18,2,3);
            for k = 1:3
                % solve MDEs
                theta = exp(logtheta);
                x = A3_MDE_output(theta,D.t,H,k);
                % RPKM_x
                mRNA_Th0        = [ x.Th0(:,2)  , x.Th0(:,4)  ];                % STAT3_m & RORC_m of Th0
                mRNA_Th17       = [ x.Th17(:,2) , x.Th17(:,4) ];                % STAT3_m & RORC_m of Th17
                x_cont          = mRNA_Th0;                                     % STAT3_m & RORC_m - controlled group, Eq. (9)
                x_vari          = (1 - alpha(k))*mRNA_Th0 + alpha(k)*mRNA_Th17; % STAT3_m & RORC_m - variable group,   Eq. (12)
                x_mRNA          = [ x_cont(2:end,:) ; x_vari(2:end,:) ];        % STAT3_m & RORC_m, without initial values
                RPKM_x(:,:,k) = x_mRNA;                                         % each replicate
                
%                 % adding qualitative info (RORC knocked-out)
%                 if mod(M,2) == 0
%                     % solve MDEs, with RORC knocked-out
%                     theta_KO = theta;
%                     theta_KO(6) = 0;
%                     x_KO = A3_MDE_output(theta_KO,D.t,H,k); % Eq. (S18)
%                     % log-likelihood, with RORC knocked-out
%                     sigma_KO = 0.25;
%                     logL_KO = -1/2*log(2*pi*sigma_KO.^2) - (x_KO.Th17(9,2) - x.Th17(9,2)).^2 ./ (2*sigma_KO.^2); % Eq. (S19) and (S20)
%                 else
%                     logL_KO = 0;
%                 end
            end
%             logL_KO = sum(logL_KO); % sum up 3 replicates
    end
    
    % mapping RPKM_x into RC_x
    lib_size    = D.lib_size(2:end,:,:);                   % library size for 18 time pts & 3 replicates
    lib_size    = repmat(lib_size,[1 2 1]);                % repeat for 2 genes
    exon_length = D.exon_length(1:2);                      % exon length of STAT3 & RORC
    exon_length = repmat(exon_length,[18 1 3]);            % repeat for 18 time pts & 3 replicates
    mu          = RPKM_x .* lib_size .* exon_length / 1e9; % RC_x, mapped read counts
    
    % RC_y, read count data
    RC_y = D.RC_y(2:end,[1,2],:); % STAT3 & RORC data, without initial values
    
    % log-likelihood
    phi  = D.phi(2:end,[1,2],:); % dispersion of STAT3 & RORC, without initial values
    logL = gammaln(RC_y + 1/phi) - gammaln(1/phi) - gammaln(RC_y + 1) - log(1 + mu.*phi)./phi + RC_y .* (log(mu) - log(1/phi + mu)); % Eq. (13)
    logL = sum(logL(:)); % sum up all points, Eq. (14)
    
%     % adding qualitative info (RORC knocked-out) into log-likelihood
%     logL = logL + logL_KO; % Eq. (S21)
    
    % check log-likelihood
    if isnan(logL)
        logL = -1e7; % L = 0
    end
    
    % log-prior (logtheta~N / theta~logN)
    logtheta_mean = 0;  % prior mean
    logtheta_sigma = 1; % prior SD
    logprior = -1/2*log(2*pi*logtheta_sigma.^2) - (logtheta(logtheta_index) - logtheta_mean).^2 ./ (2*logtheta_sigma.^2);
    logprior = sum(logprior);
    
else % wrong alpha
    logL     = -1e7; % L = 0
    logprior = -1e7; % prior = 0
end

end