%% A5_popMCMC
% is a function to run a single chain population MCMC sampling for different temperatures.
% MCMC initially requires a good starting point at each temperature by a posterior threshold value.
% popMCMC consists of local and global moves randomly chosen at each iteration.
% Only during the first half period, which is assumed as burn-in, 
% there is an adaptive process to update the proposal covariance every 500th iteration for drawing better samples.
% This MCMC process outputs in total 2000 thinned samples including burn-in.

function output = A5_popMCMC(J,N_beta,D,M,H)
%% start timer
tic_MCMC = tic;

%% parameter dimension

% MDE parameters
theta = zeros(1,14);   % reserve arrays for substitution
theta([8,13]) = 1;     % fixed parameters
logtheta = log(theta); % map into log-scale

% sampling index
switch M
    case 1, logtheta_index = [1 2 3 4     7   9 10    12    14];
    case 2, logtheta_index = [1 2 3 4   6 7   9 10    12    14];
    case 3, logtheta_index = [1 2 3 4 5   7   9 10    12    14];
    case 4, logtheta_index = [1 2 3 4 5 6 7   9 10    12    14];
    case 5, logtheta_index = [1 2 3 4     7   9 10 11 12    14];
    case 6, logtheta_index = [1 2 3 4   6 7   9 10 11 12    14];
    case 7, logtheta_index = [1 2 3 4 5   7   9 10 11 12    14];
    case 8, logtheta_index = [1 2 3 4 5 6 7   9 10 11 12    14];
end

% logsample dimension
d1 = length(logtheta_index);

% alpha dimension
switch H
    case 1, d2 = 0;
    case 2, d2 = 1;
    case 3, d2 = 3;
end

% total parameter dimension
d  = d1 + d2;

%% MCMC iteration
J_s      = 500;  % sub-iteration
J_burn   = J/2;  % burn-in number
J_output = 2000; % output sample number

%% temperature schedule
beta = ( [0:(N_beta-1)] ./ (N_beta-1) ).^5;

%% initial proposal covariance, SIMGA
w_shape = 6; % weight shape parameter (can be chosen to be 3)
weight = 0.1 * ( 1 - [0:N_beta-1].^3 ./ ((N_beta/w_shape)^3 + [0:N_beta-1].^3) ); % weight of proposal covariance
SIGMA = zeros(d,d,N_beta);
for i = 1:N_beta
    SIGMA(:,:,i) = weight(i) * eye(d);
end

%%
% output samples
para_sample     = zeros(J_output,d,N_beta); % parameters
logp_sample     = zeros(J_output,N_beta);   % log-posterior
logL_sample     = zeros(J_output,N_beta);   % log-likelihood
logprior_sample = zeros(J_output,N_beta);   % log-prior

% current sample
para_cur     = zeros(d,N_beta); % parameters
logp_cur     = zeros(1,N_beta); % log-posterior
logL_cur     = zeros(1,N_beta); % log-likelihood
logprior_cur = zeros(1,N_beta); % log-prior

% latest samples for updating proposal covariance, SIGMA
para_late = zeros(J_burn,d,N_beta); % save samples during burn-in
N_late    = ones(1,N_beta);         % number of local move in a temperature

%% J == 1, initial parameters
% choose good initial parameters for each temperature
for i = 1:N_beta
    
    logp_int  = -1e7;   % p = 0
    logp_good = -10000; % acceptable logL to start MCMC
    fail_N    = -1;     % failed number to find good initial parameters
    
    % choose good initial parameters
    while logp_int < logp_good
        
        % draw initial parameters from prior
        logsample_int                = randn(1,d1);   % logN prior
        logtheta_int                 = logtheta;      % recall
        logtheta_int(logtheta_index) = logsample_int; % substitution
        switch H
            case 1,
                alpha_int = 1;             % no alpha
                para_int  = logsample_int; % initial parameters
            case {2,3},
                alpha_int = rand(1,d2);                  % uniform prior
                para_int  = [logsample_int , alpha_int]; % initial parameters
        end
        
        % likelihood & prior
        [logL_int,logprior_int] = A4_likelihood_prior(logtheta_int,alpha_int,D,M,H);
        % posterior
        logp_int = beta(i)*logL_int + logprior_int;
        
        % failed number
        fail_N = fail_N + 1;
        disp(['i = ',num2str(i),' (try',num2str(fail_N + 1),'): logp_int = ',num2str(logp_int)]) % show the tested result
        
    end
    
    % save the accepted initial parameters
    para_cur(:,i)   = para_int;
    logp_cur(i)     = logp_int;
    logL_cur(i)     = logL_int;
    logprior_cur(i) = logprior_int;
    % save the accepted initial parameters
    para_late(1,:,i) = para_cur(:,i);
    
end

%% J > 1, to start popMCMC
for j = 2:J
    %% MCMC (main process!)
    if rand < 0.5 % popMCMC - local move
        
        % choose a temperature
        i = randi([1 N_beta],1);
        
        if i == 1 % directly draw samples from the prior
            
            % draw sample
            logsample_beta0                = randn(1,d1);     % logN prior
            logtheta_beta0                 = logtheta;        % recall
            logtheta_beta0(logtheta_index) = logsample_beta0; % substitution
            switch H
                case 1,
                    alpha_beta0 = 1;               % no alpha
                    para_beta0  = logsample_beta0; % initial parameter
                case {2,3},
                    alpha_beta0 = rand(1,d2);                      % uniform prior
                    para_beta0  = [logsample_beta0 , alpha_beta0]; % initial parameter
            end
            
            % likelihood & prior
            [logL_beta0,logprior_beta0] = A4_likelihood_prior(logtheta_beta0,alpha_beta0,D,M,H);
            % posterior
            logp_beta0 = beta(i)*logL_beta0 + logprior_beta0;
            
            % always accept & replace
            para_cur(:,i)   = para_beta0;
            logp_cur(i)     = logp_beta0;
            logL_cur(i)     = logL_beta0;
            logprior_cur(i) = logprior_beta0;
            
        else % other temperatures
            
            % AM - update proposal covariance, SIGMA
            if (j <= J_burn) && (mod(N_late(i),J_s) == 0) && (N_late(i) >= 2*J_s)
                % update proposal covariance, SIGMA
                para_cov = para_late(N_late(i)-J_s+1:N_late(i),:,i); % latest samples to form covariance matrix
                epsilon = 1e-8;
                SIGMA(:,:,i) = 2.38^2/d * ( cov(para_cov) + epsilon*eye(d) ); % proposal covariance
            end
            
            % proposed parameters
            para_star                     = mvnrnd( para_cur(:,i) , SIGMA(:,:,i) ); % draw sample using covariance matrix
            logtheta_beta                 = logtheta;        % recall
            logtheta_beta(logtheta_index) = para_star(1:d1); % substitution
            switch H
                case 1,     alpha_beta = 1;                       % no alpha
                case {2,3}, alpha_beta = para_star(end-d2+1:end); % alpha
            end
            
            % likelihood & prior
            [logL_star,logprior_star] = A4_likelihood_prior(logtheta_beta,alpha_beta,D,M,H);
            % posterior
            logp_star = beta(i)*logL_star + logprior_star;
            
            % log(acceptance probability)
            logA = logp_star - logp_cur(i);
            
            % accept & replace
            if log(rand) < logA
                % replace
                para_cur(:,i)   = para_star;
                logp_cur(i)     = logp_star;
                logL_cur(i)     = logL_star;
                logprior_cur(i) = logprior_star;
            end
        end
        
        % counter, during burn-in
        if j <= J_burn
            % updated move number
            N_late(i) = N_late(i) + 1;
            % save sample, no matter accepted or rejected
            para_late(N_late(i),:,i) = para_cur(:,i);
        end
        
    else % popMCMC - global move
        
        % choose a pair of neighbour temperatures
        i = randi([1 N_beta-1],1);
        
        % log(acceptance probability)
        logA = beta(i)*logL_cur(i+1) + beta(i+1)*logL_cur(i) ...
            -  beta(i)*logL_cur(i)   - beta(i+1)*logL_cur(i+1);
        
        % accept & swap
        if log(rand) < logA
            % swap
            para_temp       = para_cur(:,i);
            logL_temp       = logL_cur(i);
            para_cur(:,i)   = para_cur(:,i+1);
            logL_cur(i)     = logL_cur(i+1);
            para_cur(:,i+1) = para_temp;
            logL_cur(i+1)   = logL_temp;
            % re-calculate power posterior after swapping
            logp_cur(i)   = beta(i)*logL_cur(i) + logprior_cur(i);
            logp_cur(i+1) = beta(i+1)*logL_cur(i+1) + logprior_cur(i+1);
        end
        
    end
    
    %% counter, timer
    if mod(j,J/100) == 0
        percent = j / (J/100);
        formatSpec = '%3d%% - j = %5d/%3d: %2d min %2d sec. \n';
        fprintf( formatSpec , [percent,j,J,floor(toc(tic_MCMC)/60),round(rem(toc(tic_MCMC),60))] );
    end
    
    %% thinning & save output samples
    if mod(j,J/J_output) == 0
        posi = j / (J/J_output);
        para_sample(posi,:,:)   = para_cur;     % parameters
        logp_sample(posi,:)     = logp_cur;     % log-posterior
        logL_sample(posi,:)     = logL_cur;     % log-likelihood
        logprior_sample(posi,:) = logprior_cur; % log-prior
    end
    
end

%% output
output.para     = para_sample;     % parameters
output.logp     = logp_sample;     % log-posterior
output.logL     = logL_sample;     % log-likelihood
output.logprior = logprior_sample; % log-prior

end