%% A6_local_run
% is a function to run a single chain MCMC sampling locally, for an alternative model, given one data set.
% This takes around 1-3 minutes to run a chain of 200,000 samples.
% Two resulting figures show parameter posterior distribution and power posterior distribution of 2000 thinned samples, including burn-in that is separated by a vertical dashed line.
% While the model matches with the corresponding correct model, horizontal red dashed lines represent the true parameters in the figure of parameter posterior distribution.
% Remind that the reliable simulation has been done for at least 3 chains with each 2,000,000 samples.
% Readers can increase the sampling size by adjusting 'J' in the following code.

function output = A6_local_run(M,H , M_simu,H_simu,repeat)
%% call the data file
if (1 <= M_simu) && (M_simu <= 8) && (1 <= H_simu) && (H_simu <= 3)
    % load simulated data
    filename = ['data_simu_M',num2str(M_simu),'H',num2str(H_simu),'_r',num2str(repeat),'.mat'];
    load(filename)
else
    % load the experimental data
    load data_expt.mat
end

%% MCMC sampling

% input, MCMC control
J      = 2e5; % iterations (= 2e6 in the actual simulation)
N_beta = 10;  % temperature number

% output, MCMC
output = A5_popMCMC(J,N_beta,D,M,H);

%% parameter dimension

% sampling index
switch M
    case 1, logtheta_index = [1 2 3 4     7   9 10    12    14];
    case 2, logtheta_index = [1 2 3 4   6 7   9 10    12    14];
    case 3, logtheta_index = [1 2 3 4 5   7   9 10    12    14];
    case 4, logtheta_index = [1 2 3 4 5 6 7   9 10    12    14];
    case 5, logtheta_index = [1 2 3 4     7   9 10 11 12    14];
    case 6, logtheta_index = [1 2 3 4   6 7   9 10 11 12    14];
    case 7, logtheta_index = [1 2 3 4 5   7   9 10 11 12    14];
    case 8, logtheta_index = [1 2 3 4 5 6 7   9 10 11 12    14];
end

% logsample dimension
d1 = length(logtheta_index);

% alpha dimension
switch H
    case 1, d2 = 0;
    case 2, d2 = 1;
    case 3, d2 = 3;
end

% total parameter dimension
d  = d1 + d2;

%% PLOT - parameter posterior distribution (beta_i = 1)
figure('Name',['parameters posterior distribution of model (',num2str(M),',',num2str(H),') given data (',num2str(M_simu),',',num2str(H_simu),')']); clf;
set(gcf,'Units', 'Normalized','Position',[0 0.5 0.5 0.5]) % left-up corner

for m = 1:d
    
    subplot(3,5,m)
    
    % plot MCMC chain
    plot(output.para(:,m,end)); hold on;
    
    % plot true parameters
    if (1 <= M_simu) && (M_simu <= 8) && (1 <= H_simu) && (H_simu <= 3) && (M == M_simu) && (H == H_simu)
        [logtheta_true , alpha_true] = AA1_para_true; % call true parameters
        if m <= d1, plot(1:2000,repmat(logtheta_true(logtheta_index(m)),[1 2000]),'r--');
        else        plot(1:2000,repmat(alpha_true(logtheta_index(m-d1)),[1 2000]),'r--');
        end
    end
    
    % burn-in period
    line([1000 1000],ylim,'Color',[0.3 0.3 0.3],'LineStyle','--');
    
    % parameter label
    if m <= d1, ylabel(['log\theta_{',num2str(logtheta_index(m)),'}']);
    else        ylabel(['\alpha_{',num2str(m-d1),'}']);
    end
    
end

%% PLOT - power posterior distribution (each temperature)
figure('Name',['power posterior distribution of model (',num2str(M),',',num2str(H),') given data (',num2str(M_simu),',',num2str(H_simu),')']); clf;
set(gcf,'Units', 'Normalized','Position',[0.5 0.5 0.5 0.5]) % right-up corner

for i = 1:N_beta
    
    subplot(2,5,i)
    
    % power posterior distribution
    plot(output.logp(:,i)); hold on;
    
    % burn-in period
    line([1000 1000],ylim,'Color',[0.3 0.3 0.3],'LineStyle','--');
    
    % temperature label
    ylabel(['log p_{\beta_{',num2str(i),'}}(\theta|D)']);
    
end

end