function [ ok ] = group_bn_comparison( data, groups, node_sizes, variable_dag, filename, toolbox_directory, result_directory, tmp_directory, gobnilp_directory, max_fan_in )
% Learns group DAGs using several different algorithms.
%
% Input:
% data			Data matrix
% groups		Grouping
% node_sizes		Arities of the variables
% variable_dag		The ground truth variable PDAG (empty if not known)
% filename		Name of the file where the results are to be stored
% toolbox_directory	Directory where matlab toolboxes are stored
% result_directory	Directory where the results are stored
% tmp_directory		Directory where temporary files are stored temporarily
% gobnilp_directory	Directory where the GOBNILP executives are
% max_fan_in		Maximum number of parents
%
% Output:
% ok			Returns 1 if the function was executed succesfully
%


% Setting paths
%if ~isempty(toolbox_directory)
%	addpath([toolbox_directory 'bnt'])
%	addpath(genpathKPM([toolbox_directory 'bnt']));
%	addpath(genpath([toolbox_directory 'BNT_StructureLearning_v1.4c']));
%	addpath([toolbox_directory 'pmtk3-master']);
%	addpath(genpathPMTK([toolbox_directory 'pmtk3-master']));
%	addpath(genpath([toolbox_directory 'BDAGL']));
%end

% Constructing filenames
output_file1 = [tmp_directory filename '_output1.dat'];
output_file2 = [tmp_directory filename '_output2.dat'];
score_file = [tmp_directory filename '.scores'];
bn_file1 = [tmp_directory filename '_dag1.bn'];
settings_file = [tmp_directory filename '_my_settings.txt'];
result_file = [result_directory filename '_results.mat'];

% Producing data for direct learning
[ new_data, new_arities ] = make_group_variables( data' - 1, groups, node_sizes );
new_data = new_data + 1;
N = length(data(:, 1));

learned_group_pdags = cell(1, 5);

tic;



% Score-based approach
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

str1 = ['gobnilp/outputfile/adjacencymatrix = "' bn_file1 '"'];
handle = fopen(settings_file, 'w');
fprintf(handle, '%s', str1);
fclose(handle);

% Learn a group-dag directly
[ ok ] = data_to_file( new_data, new_arities, output_file1, 'score' );
system([gobnilp_directory 'scoring ' output_file1 ' 1 3 > ' score_file]);
system([gobnilp_directory 'bin/gobnilp -g=' settings_file ' ' score_file]);
[ g_dag_sb_aggr ] = read_network( bn_file1 );
learned_group_pdags{3} = my_dag_to_pdag(g_dag_sb_aggr); 

t3 = toc;
toc;

% Learn a variable-dag and use it to construct a group-dag
[ ok ] = data_to_file( data', node_sizes, output_file2, 'score' );
system([gobnilp_directory 'scoring ' output_file2 ' 1 3 > ' score_file]);
system([gobnilp_directory 'bin/gobnilp -g=' settings_file  ' ' score_file]);
[ v_dag_sb ] = read_network( bn_file1 );
[pdag_learned_sb, G] = learn_struct_pdag_pc_group(v_dag_sb, groups, max_fan_in);
learned_group_pdags{4} = pdag_learned_sb;

t4 = toc;
toc;

% Learn both a variable-dag and a group-dag simultaneously
scores = read_scores(score_file, groups);
[ group_dag, variable_dag, score ] = learn_group_dag_dp( scores, groups );
learned_group_pdags{5} = my_dag_to_pdag(group_dag);

t5 = toc;
toc;

% Find the ground truth group pdag (assumes groupwise faithfulness)
pdag_opt = [];
if ~isempty(variable_dag)
	[pdag_opt, G] = learn_struct_pdag_pc_group(variable_dag, groups, N- 1);
end
toc;

% Constrained-based approach
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%



% Learn a variable-dag and use it to construct a group-dag
pdag_tmp = learn_struct_pdag_pc('cond_indep_chisquare', N, max_fan_in, data);
learned_dag = pdag_to_dag(abs(pdag_tmp));
[pdag_learned, G] = learn_struct_pdag_pc_group(learned_dag, groups, max_fan_in);
learned_group_pdags{2} = pdag_learned;

t2 = toc;
toc;

running_times = [Inf, t2 - t5, t3, t4 - t3, t5 - t4];

save(result_file, 'learned_group_pdags', 'running_times');

% Remove temporary files
delete(output_file1, output_file2, score_file, bn_file1, settings_file); 

% This continues after removal because learn_struct_pdag_pc may run out of
% memory
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Learn a group-dag directly
try
    pdag_aggr = learn_struct_pdag_pc('cond_indep_chisquare', length(groups), max_fan_in, new_data');
catch exception
   pdag_aggr = -Inf;
   exception
end
learned_group_pdags{1} = pdag_aggr;
    
t1 = toc;
toc;

running_times = [t1 - t2, t2 - t5, t3, t4 - t3, t5 - t4];


% Comparisons
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Structural Hamming distance
alg_n = length(learned_group_pdags);
shds = -Inf*ones(1, alg_n);
skeleton_distances = -Inf*ones(1, alg_n);
direction_mismatches1 = -Inf*ones(1, alg_n);
direction_mismatches2 = -Inf*ones(1, alg_n);
if ~isempty(pdag_opt)
	for i = 1:alg_n
	    if length(learned_group_pdags{1}) > 1
	        [ d, skeleton_distance, direction_mismatch1, direction_mismatch2] = pdag_hamming_distance( pdag_opt, learned_group_pdags{i});
	        shds(i) = d;
	        skeleton_distances(i) = skeleton_distance;
	        direction_mismatches1(i) = direction_mismatch1;
	        direction_mismatches2(i) = direction_mismatch2;
	    end
	end
end


%view(biograph(pdag_to_dag(pdag_aggr)));
%view(biograph(pdag_to_dag(pdag_learned)));
%view(biograph(g_dag_sb_aggr));
%view(biograph(pdag_to_dag(pdag_learned_sb)));
%view(biograph(group_dag));
%view(biograph(pdag_to_dag(pdag_opt)));


% Save results
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

save(result_file, 'learned_group_pdags', 'running_times', 'shds', 'skeleton_distances', 'direction_mismatches1', 'direction_mismatches2');



end

