function [ group_dag, variable_dag, score ] = learn_group_dag_dp( scores, groups )
% Causal learning algorithm
%
% Finds an optimal group and variable DAGs under the constraint that the variable DAG is faithful to the 
% group DAG and implies strong group causality.
%
% Input:	scores		Local scores
%		groups		Grouping
%
% Output:	group_dag	Optimal group DAG
%		variable_dag	Corresponding variable DAG
%		score		Score of the variable DAG		
%

n = length(groups);
nn = 0;
for i = 1:n
    nn = nn + length(groups{i});
end
gm = zeros(nn, 1);
for i = 1:n
    for j = 1:length(groups{i})
        gm(groups{i}(j)) = i;
    end
end

skores = cell (nn, 2^n);
parent_sets = cell(nn, 2^n);

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Compute local scores for groups										   %
% Local score for group A given that groups B_1, ..., B_k is the maximum sum of local scores of members of A when  %
% parents can be chosen from A and B_1, ..., B_k such that variables on A form a DAG.				   %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Compute maximum local scores for members of A when parents are chosen from a specified subset of A and the union of B_1, ..., B_k
for i = 1:length(skores(:, 1))
    g = gm(i);
    m = length(groups{g});
    for j = 1:length(skores(1, :))
       if bitget(j-1, g), continue; end % can't be a parent of oneself
        
       skores{i, j} = -Inf*ones(1, 2^m);
       parent_sets{i, j} = zeros(1, 2^m);
       if j == 1
            skores{i, 1}(1) = scores{i, 1}(1);
       end
       if ~isempty(scores{i, j})
        for k = 1:length(scores(i, j))
               bits = bitget(scores{i, j}(k, 2), groups{g}); 
              ind = sum(2.^(find(bits) - 1)) + 1;
              skores{i, j}(ind) = scores{i, j}(k,1);
              parent_sets{i, j}(ind) = scores{i, j}(k,2);
        end
       end
       ind3 = j + 2^(g - 1);
       if ind3 <= length(skores(1, :)) && ~isempty(scores{i, ind3})
           for k = 1:length(scores(i, ind3))
               bits = bitget(scores{i, ind3}(k, 2), groups{g}); 
              ind = sum(2.^(find(bits) - 1)) + 1;
              if scores{i, ind3}(k,1) > skores{i, j}(ind)
                skores{i, j}(ind) = scores{i, ind3}(k,1);
                parent_sets{i, j}(ind) = scores{i, ind3}(k,2);
              end
           end
       end
       bits = bitget(j-1, 1:n);
       for bi=find(bits) % loop over all parents subsets that vary by 1 bit
            ssi = bitset(j-1, bi, 0)+1;
            for k = 1:2^m
                if skores{i, ssi}(k)>skores{i, j}(k)
                    skores{i, j}(k) = skores{i,ssi}(k);
                    parent_sets{i, j}(k) = parent_sets{i,ssi}(k);
                end
            end
       end
    end
end

group_scores = -Inf*ones(n, 2^n);
parents = cell(n, 2^n);

% Find an optimal DAG on A
for i = 1:n
    for j = 1:2^n
        if bitget(j-1, i), continue; end % can't be a parent of oneself
        
        local_scores = cell2mat(skores(groups{i}, j));
        dag = computeOptimalDag(local_scores);
        s = 0;
        pp = cell(length(dag), 1);
        gs = [];
        for k = 1:length(dag(:, 1))
            p = find(dag(:, k));
            ind = sum(2.^(p - 1)) + 1;
            s = s + local_scores(k, ind);
            pp{k} = parent_sets{groups{i}(k), j}(ind);
            gs = union(gs, gm(find(bitget(parent_sets{groups{i}(k), j}(ind), 1:nn))));
        end
        if isequal(setdiff(unique(gs), i), setdiff(find(bitget(j - 1, 1:nn)), i)) ||isequal(setdiff(unique(gs), i)', setdiff(find(bitget(j - 1, 1:nn)), i))
            group_scores(i, j) = s;
            parents{i, j} = pp;
        end
    end
end

% Find an optimal group DAG
group_dag = computeOptimalDag(group_scores);

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Find the variable DAG that corresponds to the group DAG							  %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

variable_dag = zeros(nn, nn);
score = 0;
for i = 1:n
    p = find(group_dag(:, i));
    ind = sum(2.^(p - 1)) + 1;
    score = score + group_scores(i, ind);
    for j = 1:length(groups{i})
        pp = find(bitget(parents{i, ind}{j}, 1:nn));
        variable_dag(pp, groups{i}(j)) = 1;
    end
end




end

