function [pdag, G] = learn_struct_pdag_pc_group(dag, groups, k, varargin)
% LEARN_STRUCT_PDAG_PC_GROUP Learn a partially oriented DAG (pattern) for groups of variables using the PC algorithm
% 
% dag is the variable DAG
% groups is the grouping
% k is an optional upper bound on the fan-in (default: n)

% The output pdag is an adjacency matrix, in which
% pdag(i,j) = -1 if there is an i->j edge.
% pdag(i,j) = pdag(j,i) = 1 if there is an undirected edge i <-> j
%
% This file is a modified version of the learn_struct_pdag_pc.m function from the BNT toolbox
%

n = length(groups);

  
sep = cell(n,n);
ord = 0;
done = 0;
G = ones(n,n);
G=setdiag(G,0);
while ~done
  done = 1;
  [X,Y] = find(G); 
  for i=1:length(X)
    x = X(i); y = Y(i);
    %nbrs = mysetdiff(myunion(neighbors(G, x), neighbors(G,y)), [x y]);
    nbrs = unique(mysetdiff(neighbors(G, y), x));  % bug fix by Raanan Yehezkel <raanany@ee.bgu.ac.il> 6/27/04, bug fix by Pekka Parviainen 27/5/14
    
    if length(nbrs) >= ord & G(x,y) ~= 0
      done = 0;
      %SS = subsets(nbrs, ord, ord); % all subsets of size ord
      SS = subsets1(nbrs, ord);
      for si=1:length(SS)
	S = SS{si};
	if conditional_independence_DAG(dag, groups, x, y, S) % Test conditional independence based on d-separation
	  G(x,y) = 0;
	  G(y,x) = 0;
	  sep{x,y} = myunion(sep{x,y}, S);
	  sep{y,x} = myunion(sep{y,x}, S);
	  break; % no need to check any more subsets 
	end
      end
    end 
  end
  ord = ord + 1;
end


% Create the minimal pattern,
% i.e., the only directed edges are V structures.
pdag = G;
[X, Y] = find(G);
% We want to generate all unique triples x,y,z
% This code generates x,y,z and z,y,x.
for i=1:length(X)
  x = X(i);
  y = Y(i);
  Z = find(G(y,:));
  Z = mysetdiff(Z, x);
  for z=Z(:)'
    if G(x,z)==0 & ~ismember(y, sep{x,z}) & ~ismember(y, sep{z,x})
      %fprintf('%d -> %d <- %d\n', x, y, z);
      F = pdag == -1;
      F(x, y) = 1; F(y, x) = 0;
      F(z, y) = 1; F(y, z) = 0;
      if acyclic(F)
        pdag(x,y) = -1; pdag(y,x) = 0;
        pdag(z,y) = -1; pdag(y,z) = 0;
      end
    end
  end
end

% Convert the minimal pattern to a complete one,
% i.e., every directed edge in P is compelled
% (must be directed in all Markov equivalent models),
% and every undirected edge in P is reversible.
% We use the rules of Pearl (2000) p51 (derived in Meek (1995))

old_pdag = zeros(n);
iter = 0;
while ~isequal(pdag, old_pdag)
  iter = iter + 1;
  old_pdag = pdag;
  % rule 1
  [A,B] = find(pdag==-1); % a -> b
  for i=1:length(A)
    a = A(i); b = B(i);
    C = find(pdag(b,:)==1 & G(a,:)==0); % all nodes adj to b but not a
    if ~isempty(C)
      F = pdag == -1;
      F(b, C) = 1;
      F(C, b) = 0;
      if acyclic(F)
        pdag(b,C) = -1; pdag(C,b) = 0;
        %fprintf('rule 1: a=%d->b=%d and b=%d-c=%d implies %d->%d\n', a, b, b, C, b, C);
      end
    end
  end
  % rule 2
  [A,B] = find(pdag==1); % unoriented a-b edge
  for i=1:length(A)
    a = A(i); b = B(i);
    if any( (pdag(a,:)==-1) & (pdag(:,b)==-1)' );
      F = pdag == -1;
      F(a, b) = 1; F(b, a) = 0;
      if acyclic(F)
        pdag(a,b) = -1; pdag(b,a) = 0;
        %fprintf('rule 2: %d -> %d\n', a, b);
      end
    end
  end
  % rule 3
  [A,B] = find(pdag==1); % a-b
  for i=1:length(A)
    a = A(i); b = B(i);
    C = find( (pdag(a,:)==1) & (pdag(:,b)==-1)' );
    % C contains nodes c s.t. a-c->ba
    G2 = setdiag(G(C, C), 1);
    if any(G2(:)==0) % there are 2 different non adjacent elements of C
      F = pdag == -1;
      F(a, b) = 1; pdag(b, a) = 0;
      if acyclic(F)
        pdag(a,b) = -1; pdag(b,a) = 0;
        %fprintf('rule 3: %d -> %d\n', a, b);
      end
    end
  end
end



