################################################################################
########################### Plotting Functions - Tensor/GibbsGFA ###############
################################################################################

library(gplots)

#
# Draw latent component loadings
#
plot.latent.loadings <- function(mat, onlyActive=TRUE)
{
	if(onlyActive)
		mat = mat[,apply(mat,2,sd)!=0]
	cols = palette()
	pt = ncol(mat)
	if(pt > length(cols))
	{
		pt = length(cols)
		print(paste("Plotting only the first ",pt," active components"))
	}
	
	plot(mat[,1],col=cols[1],xlab="index",ylab="loading",type="l",ylim = c(min(mat),max(mat)*1.3))
	for(i in 2:pt)	
		points(mat[,i],type="l",col=cols[i])
	legend("topright",legend=1:pt,fill=cols)
}


#
# Draw Original Data latent variables
#
plot.data.latent.variables.gfa <- function(data,text)
{
	Korig <- ncol(data)
	X11(); par(mfrow=c(Korig,1), oma=c(0,0,2,0))
	for(i in 1:Korig) {
	  if(i <= 2) cols = "red"
	  if(i == 3) cols = "grey"
	  if(i == 4) cols = "green"  
	  plot(data[,i],ylim=c(-2,2),col=cols);
	}
	par(mfrow=c(1,1));
	title(main=paste("Original",text,sep=""),outer=TRUE)
}#EndFunction

color.labels = "blue"
color.dead = "black"
color.shared = "red"
color.V1 = "green"
color.V2 = "grey"
color.part = "cyan"
#
# plotZU with colors from active
# 
plotZU <- function(i,Z,U,active)
{
	  cols = getComponentColor(i,active)
	  yy = 2
	  plot(Z[,i],ylim=c(-yy,yy),col=cols,ylab="",xlab="",axes=FALSE,pch=16); box()
	  text(x=0,y=yy-0.1,labels=i,col=color.labels,cex=1.5)
	  if(i == 1)
		text(x=length(Z[,i])/2,y=yy-0.1,labels="Z",col="blue",cex=1.5)
	  mx <- 1
	  mx2 <- max(abs(U))
	  if(sum(active[,i]) != 0 & mx2>2)
	    mx <- 1.9*mx2/mx2
	  plot(U[,i]/mx,ylim=c(-yy,yy),col=cols,ylab="",xlab="",axes=FALSE,pch=16); box()
	  if(i == 1)
		text(x=(length(U[,i])+1)/2,y=yy-0.1,labels="U",col=color.labels,cex=1.5)
}

#
# Draw Original Data latent variables
#
plot.data.latent.variables.tensor <- function(Z,U,alpha,text,plot3D=FALSE)
{
	Korig <- ncol(Z)
	X11();
	op <- par(mfrow=c(Korig,2), oma=c(0,0,2,0),mar = rep(1, 4))
	for(i in 1:Korig) {
	  plotZU(i,Z,U,round(1/alpha > 1e-3))
	}
	title(main=paste("Original Tensor Z(left) U(Right)",text,sep=""),outer=TRUE)
	par(op);
	
	if(plot3D==TRUE)
	for(i in 1:Korig){
	X11();
	#heatmap.2(Z[,i]%*%t(U[,i]),col=bluered(70),Rowv=FALSE,Colv=FALSE,scale="none",trace="none")
	persp(1:nrow(Z), 1:nrow(U), Z[,i]%*%t(U[,i]),col=bluered(70),theta=30,phi=35,xlab="Z",ylab="U",zlab="Value",main=i)
	}
}#EndFunction

#
# Draw model Latent Variables for GFA, i.e. only Z
# Note that the model is invariant to the sign of the latent
# components, so they might not end up in the right direction.
#
plot.model.latent.variables.gfa <- function(model,text)
{
	K <- ncol(model$X)
	X11(); par(mfrow=c(K/2,2), oma=c(0,0,2,0))
	for(i in 1:K) {
	cols = getComponentColor(i,model$Z)
	plot(model$X[,i],ylim=c(-2,2),main=i,col=cols);
	}
	par(mfrow=c(1,1));
	title(main=paste(text,"Train LL",round(model$cost[length(model$cost)]),sep=""),outer=TRUE)
}#EndFunction

#
# Draw Tensor Latent Variables
# Note that the model is invariant to the sign of the latent
# components, so they might not end up in the right direction.
#
plot.model.latent.variables.tensor <- function(model,text,plot3D=FALSE)
{
	K <- ncol(model$X)
	X11(); op <- par(mfrow=c(K,2), oma=c(0,0,2,0),mar = rep(0.5, 4))
	for(i in 1:K) {
	plotZU(i,model$X,model$U,model$Z)
	}
	title(main=paste(text,"Train LL",round(model$cost[length(model$cost)]),sep=""),outer=TRUE)
	par(op);	
	
	if(plot3D==TRUE)	
	for(i in 1:K){
	X11();
	persp(1:nrow(model$X), 1:nrow(model$U), model$X[,i]%*%t(model$U[,i]),col=bluered(70),theta=30,phi=35,xlab="Z",ylab="U",zlab="Model's Estimated Value",main=i)
	#heatmap.2(Z[,i]%*%t(U[,i]),col=bluered(70),Rowv=FALSE,Colv=FALSE,scale="none",trace="none")
	}
}#EndFunction


getComponentColor <- function(i,active)
{
	  cols = color.part
	  if(sum(active[,i]) == 0) cols = color.dead
	  if(sum(active[,i]) == nrow(active)) cols = color.shared
	  if(sum(active[,i]) == nrow(active)/2 && sum(active[1:(nrow(active)/2),i]) == nrow(active)/2) cols = color.V1
	  if(sum(active[,i]) == nrow(active)/2 && sum(active[(nrow(active)/2+1):nrow(active),i]) == nrow(active)/2) cols = color.V2
	  	  
	  return(cols)
}
#
# Draw TEST GFA Latent Variables
# Note that the model is invariant to the sign of the latent
# components, so they might not end up in the right direction.
#
plot.model.latent.variables.gfa.test <- function(model,text,Ytest)
{
	X11(); par(mfrow=c(K/2,2), oma=c(0,0,2,0))
	Zest <- estimZgfa(Ytest,model)$X
	for(i in 1:K) {
  	    cols = getComponentColor(i,model$Z)
	    sdZ = sd(Zest[,i])
	    if(sdZ==0) sdZ = 1
	  plot(Zest[,i]/sdZ,ylim=c(-2,2),main=i,col=cols);
	}
	par(mfrow=c(1,1));
	title(main=paste(text,"Test"),outer=TRUE)
}#EndFunction

#
# Draw TEST GFA Latent Variables
# Note that the model is invariant to the sign of the latent
# components, so they might not end up in the right direction.
#
plot.model.latent.variables.tensor.test <- function(model,text,Ytest)
{
	X11(); par(mfrow=c(K/2,2), oma=c(0,0,2,0))
	Zest <- estimZtensor(Ytest,model)$X
	#TODO #estimUtensor and plot
	
	for(i in 1:K) {
	cols = getComponentColor(i,model$Z)
	sdZ = sd(Zest[,i])
	if(sdZ==0) sdZ = 1
	plot(Zest[,i]/sdZ,ylim=c(-2,2),main=i,col=cols);
	}
	par(mfrow=c(1,1));
	title(main=paste(text,"Test"),outer=TRUE)
}#EndFunction

#
# Draw Tensor Cost over Iterations
#
plot.tensor.cost <- function(model,text)
{
	x11(); par(mfrow=c(4,1), oma=c(0,0,0,0),mar=c(2,2,2,0)+0.2)
	plot(model$cost,main=paste("Best ",text," @LL=",round(model$cost[length(model$cost)]),sep=""))
	lines(x=c(500,500),y=c(-10000,10000),col="red")
	points(model$cost)
	plot(1:length(model$spes2),(model$spes1+model$spes2+model$shared),col="blue",cex=1.5,main="Zactive",ylim=c(0,K),ylab="cost",xlab="")
	points(1:length(model$shared),model$shared,col="red",pch=17,cex=1)
	points(1:length(model$spes1),model$spes1,col="green",cex=0.85)
	points(1:length(model$spes2),model$spes2,col="grey",cex=0.7)
	legend("topright",legend=c("Active in > 1","Shared in Both","Active in 1st","Active in 2nd"),fill=c("blue","red","green","grey"))
	
	if(length(model$debug$taus)>1)
	{
	plot(model$debug$taus,main="Mean Tau")
	lines(x=c(500,500),y=c(-10000,10000),col="red")
	points(model$debug$taus)
	
	plot(model$debug$btau,main="Data - Mean Squared Error")
	lines(x=c(500,500),y=c(-10000,10000),col="red")
	points(model$debug$btau)
	}
	par(mfrow=c(1,1),mar=c(5,4,2,2)+0.2);
	#title(main=paste(text,"Iterations"),outer=TRUE)
}#EndFunction


#
# Draw Tensor Cost over Iterations of Multiple Runs in a single plot
#
plot.tensor.cost.multirun <- function(res,text,mse=NULL,offset=1)
{
	X11(); par(mfrow=c(ceiling(length(res)/2),2), oma=c(0,0,2,0))
	for(i in 1:length(res)) {
	  plot(res[[i]]$cost[1:length(res[[i]]$cost)],main=get.print.string(i,res,mse),ylab="cost",xlab="")
	}
	par(mfrow=c(1,1));
	title(main=paste(text," Log Likelihood",sep=""),outer=TRUE)
}#EndFunction

################################################################################
########################### Plotting Functions - VB ############################
################################################################################

#
# Draw Tensor Cost over Iterations of Multiple Runs in a single plot
# 
plot.tensor.cost.multirun.VB <- function(res,text,mse=NULL,offset=1)
{
	X11(); par(mfrow=c(ceiling(length(res)/2),2), oma=c(0,0,2,0))
	for(i in 1:length(res)) {
	  plot(res[[i]]$cost[offset:length(res[[i]]$cost)],main=get.print.string.VB(i,res,mse),ylab="cost",xlab="")
	}
	par(mfrow=c(1,1));
	title(main=paste(text," Log Likelihood",sep=""),outer=TRUE)
}#EndFunction

#
# Draw Latent Variables for Variational Bayes GFA
# 
plot.model.latent.variables.vbGFA <- function(model,text="GFA VB")
{
	X11(); par(mfrow=c(K/2,2), oma=c(0,0,2,0))
	for(i in 1:K) {
	  active <- round(1/model$alpha > 1e-3)
	  cols = getComponentColor(i,active)  
	  plot(model$Z[,i],ylim=c(-2,2),main=i,col=cols,ylab="",xlab="");
	}
	par(mfrow=c(1,1));
	title(paste(text," Train LL",round(model$cost[length(model$cost)]),sep=""),outer=TRUE)
}#EndFunction

#
# Draw Latent Variables for Variational Bayes GFA when used with TensorData
# 
plot.model.latent.variables.vbGFA.tensorData <- function(model,text="GFA VB",data,plot3D=FALSE)
{
	if(data$L == 1)
	{
		plot.model.latent.variables.vbGFA(model,text)
		return()
	}
	
	K = ncol(model$X)
	X11(); 
	op <- par(mfrow=c(K,2), oma=c(0,0,2,0),mar = rep(0.5, 4))
	active <- round(1/model$alpha > 1e-3)
	  
	model$Z = model$X
	model$Uestim = model$U
	for(i in 1:ncol(model$Z))
	{
		if(colSums(active)[i] != 0)
		{
			model$Z[,i] = model$Z[,i]/sd(model$Z[,i])
			model$Uestim[,i] = model$Uestim[,i]/sd(model$Uestim[,i])
		}
	}

	for(i in 1:K) {
	  print("If you get error: Is U and X plotting right?")
	  plotZU(i,model$Z,model$Uestim,active)
	}
	title(paste(text," Train LL",round(model$cost[length(model$cost)]),sep=""),outer=TRUE)
	par(op);

	if(plot3D==TRUE)
	{
	for(i in 1:K){
	X11(); 
	persp(1:nrow(model$Z), 1:nrow(model$Uestim), model$Z[,i]%*%t(model$Uestim[,i]),col=bluered(70),theta=30,phi=35,xlab="Z",ylab="U",zlab="GFA's Estimated Value",main=i)
	}
	}	
}#EndFunction

#
# Draw Latent Variables for Variational Bayes Tensor GFA
# 
plot.model.latent.variables.vbTensor <- function(model,text="Tensor VB",plot3D=FALSE)
{
	
	K = ncol(model$X)
	X11();
	op <- par(mfrow=c(K,2), oma=c(0,0,2,0),mar = rep(0, 4))
	active <- round(1/model$alpha > 1e-3)
	  
	model$Z = model$X
	model$Uestim = model$U
	for(i in 1:ncol(model$Z))
	{
		if(colSums(active)[i] != 0)
		{
			model$Z[,i] = model$Z[,i]/sd(model$Z[,i])
			model$Uestim[,i] = model$Uestim[,i]/sd(model$Uestim[,i])
		}
	}

	for(i in 1:K) {
	  plotZU(i,model$Z,model$Uestim,active)
	}
	title(paste(text," Train LL",round(model$cost[length(model$cost)]),sep=""),outer=TRUE)
	par(op);
	
	if(plot3D==TRUE)	
	for(i in 1:K){
	X11(); persp(1:nrow(model$Z), 1:nrow(model$U), model$Z[,i]%*%t(model$U[,i]),col=bluered(70),theta=30,phi=35,xlab="Z",ylab="U",zlab="Model's Estimated Value",main=i) 
	}
}#EndFunction


#
# Draw Latent Variables for Variational Bayes GFA TEST
#
plot.model.latent.variables.vbGFA.test <- function(model,text="GFA VB",Ytest)
{
	X11(); par(mfrow=c(K/2,2), oma=c(0,0,2,0))
	Zest <- estimZgfa(Ytest,model)
	print("this function will not run")
	for(i in 1:K) {
	  active <- round(1/model$alpha > 1e-3)
	  pt = Zest[,i]/sd(Zest[,i])
	  if(sum(active[,i]) == 0) {cols = "black"; pt <- Zest[,i]}
	  if(sum(active[,i]) == 2) cols = "red"
	  if(sum(active[,i]) == 1 && active[1,i]) cols = "green"
	  if(sum(active[,i]) == 1 && active[2,i]) cols = "grey"  
	  plot(pt,ylim=c(-2,2),main=i,col=cols);
	}
	par(mfrow=c(1,1));
	title(paste(text," Test LL",round(model$cost[length(model$cost)]),sep=""),outer=TRUE)
}#EndFunction
################################################################################
########################### Plotting Functions - Sparse GFA ####################
################################################################################

#
# Draw Latent Variables for Sparse GFA
#
plot.model.latent.variables.sGFA <- function(model,text="Sparse GFA")
{
	X11(); par(mfrow=c(K/2,2), oma=c(0,0,2,0))
	for(i in 1:K) {
	  plot(model4$X[,i],ylim=c(-2,2),main=i);
	}
	par(mfrow=c(1,1));
	title(paste(text,"Train LL",round(model3$cost[length(model3$cost)]),sep=""),outer=TRUE)
}#EndFunction

################################################################################
########################### Plotting Functions - Tensor over L #################
################################################################################

################## Plot the full Run (activity + mse)
plotComponent.overVAR.Type <- function(modelRuns.list,data.list,Vmax,Vlab,runs,var="L",title.text,type="GB",do.x11=TRUE,ordering=FALSE)
{
if(length(modelRuns.list) == 0)
	return()
if(do.x11) X11();
op <- par(mfrow=c(2,2), oma=c(0,0,2,0),mar = c(2,4,2,0)+0.3)
activity <- plotComponentActivity.overVAR.Type(modelRuns.list,data.list,Vmax,Vlab,runs,var,do.x11=FALSE,type,ordering)
mse.W <- plotMSEW.overVAR.Type(modelRuns.list,data.list,Vmax,Vlab,runs,var,do.x11=FALSE,activity$od)
mse.U <- plotMSEU.overVAR.Type(modelRuns.list,data.list,Vmax,Vlab,runs,var,do.x11=FALSE,activity$od)
mse.Z <- plotMSEZ.overVAR.Type(modelRuns.list,data.list,Vmax,Vlab,runs,var,do.x11=FALSE,activity$od,type)
title(paste(title.text,sep=""),outer=TRUE)
par(op);
lt <- list(activity=activity,mse.W=mse.W,mse.U=mse.U,mse.Z=mse.Z)
print("-----------------------------------------------------------------------")
print(title.text)
print(lt)
print("-----------------------------------------------------------------------")
return(lt)
}

#################### Active Components
plotComponentActivity.overVAR.Type <- function(modelRuns.list,data.list,Vmax,Vlab,runs,var="L",do.x11=TRUE,type="GB",ordering=FALSE)
{
if(length(modelRuns.list) == 0)
	return()

TC <- matrix(0,Vmax,runs) #Total Active
colnames(TC) = paste("Run",1:runs,sep=".")
rownames(TC) = paste(var,Vlab,sep=".")
SC <- TC #Shared
PC <- TC #sPecific 
AC <- TC #pArtly shared
for(v in 1:Vmax)
{
	TC[v,] <- countTotalComponents.list.Type(modelRuns.list[[v]]$all,type)
	SC[v,] <- countSharedComponents.list.Type(modelRuns.list[[v]]$all,type)
	PC[v,] <- countSpecificComponents.list.Type(modelRuns.list[[v]]$all,type)
	AC[v,] <- countCombinationComponents.list.Type(modelRuns.list[[v]]$all,type)	
}

tc <- sc <- pc <- ac <- rep(0,Vmax)
for(v in 1:Vmax)
{
tc[v] <- countTotalComponents.Type(data.list[[v]],"VB") #Must be VB allways
sc[v] <- countSharedComponents.Type(data.list[[v]],"VB") #Must be VB allways
pc[v] <- countSpecificComponents.Type(data.list[[v]],"VB") #Must be VB allways
ac[v] <- countCombinationComponents.Type(data.list[[v]],"VB") #Must be VB allways
}
od <- 1:Vmax
if(ordering==TRUE) od <- order(tc,decreasing=FALSE)
tc <- tc[od]; sc <- sc[od]; pc <- pc[od]; ac <- ac[od]
TC <- diag(Vmax)[od,] %*% TC; SC <- diag(Vmax)[od,] %*% SC; PC <- diag(Vmax)[od,] %*% PC; AC <- diag(Vmax)[od,] %*% AC
rownames(TC) <- Vlab[od]; rownames(SC) <- Vlab[od]; rownames(PC) <- Vlab[od]; rownames(AC) <- Vlab[od]
##plot recovery
ylim=c(0,max(TC)+3)
cols = palette()
if(do.x11) x11();
plot(Vlab,rowMeans(TC),ylim=ylim,pch=16,col=cols[1],xlab=var,ylab="Lines(Original) Filled Points(Recovered)",main=paste("Active Components,",type))
if(sum(rowMeans(SC))!= 0) points(Vlab,rowMeans(SC),pch=15,col=cols[2],cex=1.5)
if(sum(rowMeans(PC))!= 0) points(Vlab,rowMeans(PC),pch=17,col=cols[3],cex=1)
if(sum(rowMeans(AC))!= 0) points(Vlab,rowMeans(AC),pch=14,col=cols[4],cex=0.7)
##plot original
points(Vlab,tc,type="l",lty="solid",col=cols[1])
if(sum(sc)!= 0) points(Vlab,sc,type="l",lty="longdash",col=cols[2])
if(sum(pc)!= 0) points(Vlab,pc,type="l",lty="dashed",col=cols[3])
if(sum(ac)!= 0) points(Vlab,ac,type="l",lty="dotted",col=cols[4])
#lines(x=c(0,Lmax),y=c(tc,tc),type="b",lty="dotted",col=cols[1])
#lines(x=c(0,Lmax),y=c(sc,sc),type="b",lty="dotted",col=cols[2])
#lines(x=c(0,Lmax),y=c(pc,pc),type="b",lty="dotted",col=cols[3])
legend("topleft",legend=c("Total","Shared","Specific","PartShared"),fill=cols[1:4])
return(list(tc=TC,sc=SC,pc=PC,ac=AC,od=od))
}

#################### Plot MSE W
plotMSEW.overVAR.Type <- function(modelRuns.list,data.list,Vmax,Vlab,runs,var="L",do.x11=TRUE,od)
{
	if(length(modelRuns.list) == 0)
		return()

	mse.ten.W <- matrix(0,Vmax,runs)
	colnames(mse.ten.W) = paste("Run",1:runs,sep=".")
	rownames(mse.ten.W) = paste(var,Vlab,sep=".")
	for(v in 1:Vmax)
	{
		mse.ten.W[v,] <- unlist(lapply(modelRuns.list[[v]]$all,function(x) {computeMSE.W(data.list[[v]]$W,x$W)}))
	}
	mse.ten.W <- diag(Vmax)[od,] %*% mse.ten.W; rownames(mse.ten.W) <- Vlab[od]
	if(do.x11) x11();
	ylim=c(0,max(mse.ten.W,0.2))
	plot(Vlab,rowMeans(mse.ten.W),ylim=ylim,pch=16,col="blue",xlab=var,ylab="mse",main="MSE W")

return(mse.ten.W)
}

#################### Plot MSE U
plotMSEU.overVAR.Type <- function(modelRuns.list,data.list,Vmax,Vlab,runs,var="L",do.x11=TRUE,od)
{
	if(length(modelRuns.list) == 0)
		return()

	mse.ten.U <- matrix(0,Vmax,runs)
	colnames(mse.ten.U) = paste("Run",1:runs,sep=".")
	rownames(mse.ten.U) = paste(var,Vlab,sep=".")	
	for(v in 1:Vmax)
	{
		mse.ten.U[v,] <- unlist(lapply(modelRuns.list[[v]]$all,function(x) {computeMSE.W(list(data.list[[v]]$U),list(x$U))}))
	}
	mse.ten.U <- diag(Vmax)[od,] %*% mse.ten.U; rownames(mse.ten.U) <- Vlab[od]
	if(do.x11) x11();
	ylim=c(0,max(mse.ten.U,0.2))
	plot(Vlab,rowMeans(mse.ten.U),ylim=ylim,pch=16,col="blue",xlab=var,ylab="mse",main="MSE U")

return(mse.ten.U)
}

#################### Plot MSE Z
getLatentVar <- function(model,type)
{
if(type=="GB")
	return(model$X)
return(model$Z)
}
plotMSEZ.overVAR.Type <- function(modelRuns.list,data.list,Vmax,Vlab,runs,var="L",do.x11=TRUE,od,type)
{
	if(length(modelRuns.list) == 0)
		return()

	mse.ten.Z <- matrix(0,Vmax,runs)
	colnames(mse.ten.Z) = paste("Run",1:runs,sep=".")
	rownames(mse.ten.Z) = paste(var,Vlab,sep=".")	
	for(v in 1:Vmax)
	{
		mse.ten.Z[v,] <- unlist(lapply(modelRuns.list[[v]]$all,function(x) {computeMSE.W(list(data.list[[v]]$Z),list(getLatentVar(x,type)))}))
	}
	mse.ten.Z <- diag(Vmax)[od,] %*% mse.ten.Z; rownames(mse.ten.Z) <- Vlab[od]
	if(do.x11) x11();
	ylim=c(0,max(mse.ten.Z,0.2))
	plot(Vlab,rowMeans(mse.ten.Z),ylim=ylim,pch=16,col="blue",xlab=var,ylab="mse",main="MSE Z")

return(mse.ten.Z)
}
################################################################################
################# Comparison with Other Methods Plotting Functions #############
################################################################################

#################### MSE W with GFA
compareMSE.W.overL <- function(tensorRun.VB,gfaRun.VB,tensor.data.list,gfa.data.list,text="GB",Lmax,runs)
{
if((length(tensorRun.VB) == 0) | (length(gfaRun.VB) == 0))
	return()
	
mse.ten.VB.W <- matrix(0,Lmax,runs)
mse.gfa.VB.W <- matrix(0,Lmax,runs)
for(l in 1:Lmax)
{
	mse.ten.VB.W[l,] <- unlist(lapply(tensorRun.VB[[l]]$all,function(x) {computeMSE.W(tensor.data.list[[l]]$W,x$W)}))
	mse.gfa.VB.W[l,] <- unlist(lapply(gfaRun.VB[[l]]$all,function(x) {computeMSE.W(gfa.data.list[[l]]$W,x$W)}))
}
ylim=c( min(mse.ten.VB.W,mse.gfa.VB.W) , max(mse.ten.VB.W,mse.gfa.VB.W))
x11(); plot(rowMeans(mse.ten.VB.W),ylim=ylim,pch=16,col="blue",xlab="L",ylab="mse W",main=paste("Comparison of Mean Squared Error",text))
points(rowMeans(mse.gfa.VB.W),pch=16,col="red")
legend("topleft",legend=c("Tensor GFA","GFA"),fill=c("blue","red"))
}

#################### MSE F with GFA
compareMSE.F.overL <- function(tensorRun.VB,gfaRun.VB,tensor.data.list,gfa.data.list,text="GB",Lmax,runs)
{
if((length(tensorRun.VB) == 0) | (length(gfaRun.VB) == 0))
	return()

mse.ten.VB.F <- matrix(0,Lmax,runs)
mse.gfa.VB.F <- matrix(0,Lmax,runs)
for(l in 1:Lmax)
{
	mse.ten.VB.F[l,] <- unlist(lapply(tensorRun.VB[[l]]$all,function(x) {computeMSE.W(list(1/tensor.data.list[[l]]$alpha),list(1/x$alpha))}))
	
	mt <- 1/gfa.data.list[[l]]$alpha
	mt2 <- rbind(matrix(rep(mt[1,],l),l,gfa.data.list[[l]]$K,byrow=TRUE),matrix(rep(mt[2,],l),l,gfa.data.list[[l]]$K,byrow=TRUE))
	mse.gfa.VB.F[l,] <- unlist(lapply(gfaRun.VB[[l]]$all,function(x) {computeMSE.W(list(mt2),list(1/x$alpha))}))
}
ylim=c( min(mse.ten.VB.F,mse.gfa.VB.F) , max(mse.ten.VB.F,mse.gfa.VB.F))
x11(); plot(rowMeans(mse.ten.VB.F),ylim=ylim,pch=16,col="blue",xlab="L",ylab="mse F",main=paste("Comparison of Mean Squared Error",text))
points(rowMeans(mse.gfa.VB.F),pch=16,col="red")
legend("topleft",legend=c("Tensor GFA","GFA"),fill=c("blue","red"))
}

#################################################
exploreSingleRun <- function(tensorRun,tensor.data,type,run=0)
{
if(length(tensorRun) == 0)
return()

if(length(tensor.data) > 1) #do only if toy data. real data will have only tensor.data$Y in the list.
{
#
# Plot the Data
#
plot.data.latent.variables.tensor(tensor.data$Z,tensor.data$U,tensor.data$alpha,getDataSettingString(tensor.data),plot3D = FALSE)
#
# Print & Plot a single Model Run
#
mse_ten <- unlist(lapply(tensorRun$all,function(x) {computeMSE.W(tensor.data$W,x$W)}))
#browser()
print("----------------Tensor Result----------------")
print(paste("MSE",runs,"runs:",round(mean(mse_ten),3),sep=" "))
if(type=="GB")
print.activity.multirun(tensorRun$all,mse_ten)
print("Original Tau:")
print(1/unlist(lapply(tensor.data$tau,mean)))
print("Recovered Tau:")
print(unlist(lapply(tensorRun$best$tau,function(x) { mean(1/x) })))	
#plot
tString <- paste("Tensor",getDataSettingString(tensor.data),sep=" ")
plot.model.latent.variables.tensor(tensorRun$best,tString,plot3D=FALSE)
plot.tensor.cost.multirun(tensorRun$all,tString,mse_ten)
plot.tensor.cost(tensorRun$best,tString)
}

#run = 2
if(run!=0)
{
	model <- tensorRun$all[[run]]
	plot.model.latent.variables.tensor(model,tString,plot3D=FALSE)
	plot.tensor.cost(model,tString)
}
else
	model <- tensorRun$best

M <- length(model$W)
K <- ncol(model$Z)
print("-------------------------RANGE OF DATA -------------------------")
for(m in 1:M) #print range of data
{
for(k in 1:K) 
print(paste("M",m,"K",k," ",paste(round(range((remake.tensor(model,k)[[m]])),2),collapse=","),sep=""))
}

print("-------------------------TOTAL VAIRANCE of remaked tensor-------------------------")
for(m in 1:M) #print tot var of data
{
for(k in 1:K) 
print(paste("M",m,"K",k," ",round(get.total.var.tensor(remake.tensor(model,k)[[m]]),5),sep=""))
}

badk = 0
print("-------------------------Does a Component Matter? High=Yes-------------------------")
for(m in 1:M)
for(k in 1:K) 
{
diff <- sum((tensor.data$Y[[m]] - remake.tensor(model)[[m]])^2) - 
sum((tensor.data$Y[[m]] - remake.tensor.withoutK(model,k)[[m]])^2)
diff2 <- sum((remake.tensor(model)[[m]]- remake.tensor.withoutK(model,k)[[m]])^2)
print(paste("M",m,"K",k,"    Diff Org: ",format(x=round(diff,0),width=7),"    Diff Proj: ",format(x=round(diff2,0),width=7),sep=""))
#To judge a component's influence one should compare the sum-of-squares of the original data with the sum-of-squares of the data subtracted the specific component. http://www.models.life.ku.dk/~rasmus/presentations/parafac_tutorial/paraf.htm
if(abs(diff) < 1000 && abs(diff) != 0)
	badk = k
}

apply(model$U,2,var)
apply(model$X,2,var)
for(m in 1:M)
apply(model$W[[m]],2,var)

print(badk)
print(cor(model$U[,badk],tensor.data$U))
print(cor(model$X[,badk],tensor.data$Z))
for(m in 1:M)
if(sum(model$W[[m]][,badk])!=0)
print(cor(model$W[[m]][,badk],tensor.data$W[[m]]))

return(model)
}


###########################################################################
#norm for plots
normU.max1 <- function(model)
{
	ll <- normK.max1(model$U,model$W)
	model$U <- ll$U
	model$W <- ll$W	
	return(model)
}
normZ.max1 <- function(model)
{
	ll <- normK.max1(model$Z,model$W)
	model$Z <- ll$U
	model$W <- ll$W	
	return(model)
}
normX.max1 <- function(model)
{
	ll <- normK.max1(model$X,model$W)
	model$X <- ll$U
	model$W <- ll$W	
	return(model)
}

normK.max1 <- function(U,W)
{
	L <- nrow(U)
	M <- length(W)
	D <- unlist(lapply(W,nrow))
	if(L == 1)
	{
		Ru <- as.vector(U)
		U <- U*outer(rep(1,L),1/Ru)
		U[is.na(U)] = 0
		for(m in 1:M)
			W[[m]] <- W[[m]]*outer(rep(1,D[m]),Ru)
	}
	else
	{
		Ru.x <- apply(U,2,max)
		Ru.n <- apply(U,2,min)
		Ru <- Ru.x
		inds <- abs(Ru.n)  > Ru.x
		Ru[inds] <- Ru.n[inds]
		U <- U*outer(rep(1,L),1/Ru)
		U[is.na(U)] = 0
		for(m in 1:M)
			W[[m]] <- W[[m]]*outer(rep(1,D[m]),Ru)
	}
	return(list(U=U,W=W))
}

