#
# Tools for generating test data for Tensor GFA and corresponding GFA variants
#
# For questions and bug reports please contact
# suleiman.khan@aalto.fi
#
# Copyright 2013 Suleiman Ali Khan. All rights reserved.
# The software is licensed under the FreeBSD license; see LICENSE
# included in the package.


###########################################
#
# Functions for Components which are visually recognizable
# Generated as in JMLR2013 Klami et al paper
getVisualX.1.Paper <- function(Ntrain,Ntest)
{
	K <- 4		
	if(Ntrain == 1) return(matrix(1,Ntrain+Ntest,K))
	
	N = Ntrain + Ntest
	X <- matrix(0,N,K)       # Latent components
	X[,1] <- sin((1:N)/(N/(pi*4))) #(pi*11)
	X[,2] <- cos((1:N)/(N/(pi*4))) #cosine
	#X[,3] <- c((1:Ntrain-Ntrain/2)^2,(1:Ntest-Ntest/2)^2); X[,3] <- (X[,3] - mean(X[,3]))/sd(X[,3]) #quadratic
	X[,3] <- rnorm(N) #rnorm
	X[,4] <- as.matrix(c(2*((1:Ntrain)/Ntrain-0.5),2*((1:Ntest)/Ntest-0.5)),N,1) #linear
	
	Xtrain <- X[1:Ntrain,]
	Xtest <- X[(1:Ntest)+Ntrain,]
#	Xtrain <- norm.ztransform(Xtrain)
#	Xtest <- norm.ztransform(Xtest)
	Xt <- norm.ztransform.Rec(Xtrain)
	Xtrain <- Xt$mat
	Xt2 <- norm.ztransform.Rec(mat=Xtest,cm=Xt$cm,cv=Xt$cv)
	Xtest <- Xt2$mat

#Xtrain <- Xtrain + abs(min(Xtrain)) ###non-neg check
	
	return(list(Xtrain=Xtrain,Xtest=Xtest))
}
getVisualU.1.Paper <- function(L)
{
	K = 4
	if(L == 1) return(matrix(1,L,K))
		
	Lorig = L
	if(Lorig < 10) Lorig = 10
	U <- matrix(0,Lorig,K)
	U[,1] <- sin((1:Lorig)/(Lorig/(pi*2))) #(pi*5.5)
	U[,2] <- cos((1:Lorig)/(Lorig/(pi*2))) #cosine
	#U[,3] <- c((1:Lorig-Lorig/2)^2); U[,3] <- (U[,3] - mean(U[,3]))/sd(U[,3]) #quadratic
	U[,3] <- rnorm(Lorig) #rnorm
	U[,4] <- as.matrix(2*((1:Lorig)/Lorig-0.5),Lorig,1) #linear
	U <- matrix(U[1:L,],nrow=L,ncol=K)
	U <- norm.ztransform(U)
	#U <- U + abs(min(U))	###non-neg check
	return(U)
}
getVisualAlpha.1.Paper <- function(M)
{
	M1 = ceiling(M/2)
	M2 = floor(M/2)
	alpha <- matrix(0,M,4)    # Component precisions for the two data sets
	for(m in 1:M1)
	alpha[m,] <- c(1,1,1e6,1) # 1   = active (the value determines the data scale)
	if(M2 > 0)
	for(m in 1:M2)	
	alpha[m+M1,] <- c(1,1,1,1e6) # 1e6 = inactive
	return(alpha)
}

getRandomW.1.Paper <- function(D,alpha)
{
	M <- nrow(alpha)
	K <- ncol(alpha)
	W <- vector("list",length=M)
	# Create some random projection vectors
	for(view in 1:M) {
	  W[[view]] <- matrix(0,D[view],K)
	  for(k in 1:K) {
	  if(1/alpha[view,k] < 1e-3)
	    W[[view]][,k] <- 0
	  else
	  {
	    #W[[view]][,k] <- rnorm(D[view],0,1/sqrt(alpha[view,k]))
		if(k==1)
		W[[view]][,k] <- sin((1:D[view])/(D[view]/(pi*2))) #(pi*11)
		if(k==2)
		W[[view]][,k] <- cos((1:D[view])/(D[view]/(pi*2))) #cosine
		if(k==3)
		W[[view]][,k] <- rnorm(D[view])
		if(k==4)
		W[[view]][,k] <- as.matrix(2*((1:D[view])/D[view]-0.5),D[view],1) #linear
		
		W[[view]][,k] <- (W[[view]][,k]-mean(W[[view]][,k]))/sd(W[[view]][,k])
	  }
	  }

	}
	return(W)
}

getVisualComponents.1.Paper <- function(Ntrain,Ntest,L,D,M,sparse=FALSE)
{
	X <- getVisualX.1.Paper(Ntrain,Ntest)
	U <- getVisualU.1.Paper(L)
	alphaD <- NULL
	
	alpha <- getVisualAlpha.1.Paper(M)
	W <- getRandomW.1.Paper(D,alpha)

	return(list(Xtrain=X$Xtrain,Xtest=X$Xtest,U=U,W=W,alpha=alpha,K=4,L=L,D=D,M=M,Ntrain=Ntrain,Ntest=Ntest,alphaD=alphaD))
}
####################################################################################
###########################################
#
# Functions for Components which are visually recognizable
# Generated as in JMLR2013 Klami et al paper
getVisualX.3K.Paper <- function(Ntrain,Ntest)
{
	K <- 3		
	if(Ntrain == 1) return(matrix(1,Ntrain+Ntest,K))
	
	N = Ntrain + Ntest
	X <- matrix(0,N,K)       # Latent components
	X[,1] <- sin((1:N)/(N/(pi*4))) #(pi*11)
	X[,2] <- cos((1:N)/(N/(pi*4))) #cosine
	X[,3] <- as.matrix(c(2*((1:Ntrain)/Ntrain-0.5),2*((1:Ntest)/Ntest-0.5)),N,1) #linear
	
	Xtrain <- X[1:Ntrain,]
	Xtest <- X[(1:Ntest)+Ntrain,]
#	Xtrain <- norm.ztransform(Xtrain)
#	Xtest <- norm.ztransform(Xtest)
	Xt <- norm.ztransform.Rec(Xtrain)
	Xtrain <- Xt$mat
	Xt2 <- norm.ztransform.Rec(mat=Xtest,cm=Xt$cm,cv=Xt$cv)
	Xtest <- Xt2$mat

#Xtrain <- Xtrain + abs(min(Xtrain)) ###non-neg check
	
	return(list(Xtrain=Xtrain,Xtest=Xtest))
}
getVisualU.3K.Paper <- function(L)
{
	K = 3
	if(L == 1) return(matrix(1,L,K))
		
	Lorig = L
	if(Lorig < 10) Lorig = 10
	U <- matrix(0,Lorig,K)
	U[,1] <- sin((1:Lorig)/(Lorig/(pi*2))) #(pi*5.5)
	U[,2] <- cos((1:Lorig)/(Lorig/(pi*2))) #cosine
	U[,3] <- as.matrix(2*((1:Lorig)/Lorig-0.5),Lorig,1) #linear
	U <- matrix(U[1:L,],nrow=L,ncol=K)
	U <- norm.ztransform(U)
	return(U)
}
getVisualAlpha.3K.Paper <- function(M)
{
	K = 3
	M1 = ceiling(M/2)
	M2 = floor(M/2)
	alpha <- matrix(0,M,K)    # Component precisions for the two data sets
	for(m in 1:M1)
	alpha[m,] <- c(1,1e6,1) # 1   = active (the value determines the data scale)
	if(M2 > 0)
	for(m in 1:M2)	
	alpha[m+M1,] <- c(1,1,1e6) # 1e6 = inactive
	return(alpha)
}

getRandomW.3K.Paper <- function(D,alpha)
{
	K = 3
	M <- nrow(alpha)
	K <- ncol(alpha)
	W <- vector("list",length=M)
	# Create some random projection vectors
	for(view in 1:M) {
	  W[[view]] <- matrix(0,D[view],K)
	  for(k in 1:K) {
	  if(1/alpha[view,k] < 1e-3)
	    W[[view]][,k] <- 0
	  else
	  {
	    #W[[view]][,k] <- rnorm(D[view],0,1/sqrt(alpha[view,k]))
		if(k==1)
		W[[view]][,k] <- sin((1:D[view])/(D[view]/(pi*2))) #(pi*11)
		if(k==2)
		W[[view]][,k] <- cos((1:D[view])/(D[view]/(pi*2))) #cosine
		if(k==3)
		W[[view]][,k] <- as.matrix(2*((1:D[view])/D[view]-0.5),D[view],1) #linear
		
		W[[view]][,k] <- (W[[view]][,k]-mean(W[[view]][,k]))/sd(W[[view]][,k])
	  }
	  }

	}
	return(W)
}

getVisualComponents.3K.Paper <- function(Ntrain,Ntest,L,D,M,sparse=FALSE)
{
	X <- getVisualX.3K.Paper(Ntrain,Ntest)
	U <- getVisualU.3K.Paper(L)
	alphaD <- NULL
	
	alpha <- getVisualAlpha.3K.Paper(M)
	W <- getRandomW.3K.Paper(D,alpha)

	return(list(Xtrain=X$Xtrain,Xtest=X$Xtest,U=U,W=W,alpha=alpha,K=3,L=L,D=D,M=M,Ntrain=Ntrain,Ntest=Ntest,alphaD=alphaD))
}
####################################################################################
###########################################
#
# Functions for Components which are visually recognizable
# Generated as in JMLR2013 Klami et al paper
getVisualX.1 <- function(Ntrain,Ntest)
{
	K <- 4		
	if(Ntrain == 1) return(matrix(1,Ntrain+Ntest,K))
	
	N = Ntrain + Ntest
	X <- matrix(0,N,K)       # Latent components
	X[,1] <- sin((1:N)/(N/40)) #(pi*11)
	X[,2] <- cos((1:N)/(N/40)) #cosine
	X[,3] <- c((1:Ntrain-Ntrain/2)^2,(1:Ntest-Ntest/2)^2); X[,3] <- (X[,3] - mean(X[,3]))/sd(X[,3]) #quadratic
	X[,4] <- as.matrix(c(2*((1:Ntrain)/Ntrain-0.5),2*((1:Ntest)/Ntest-0.5)),N,1) #linear
	
	Xtrain <- X[1:Ntrain,]
	Xtest <- X[(1:Ntest)+Ntrain,]
#	Xtrain <- norm.ztransform(Xtrain)
#	Xtest <- norm.ztransform(Xtest)
	Xt <- norm.ztransform.Rec(Xtrain)
	Xtrain <- Xt$mat
	Xt2 <- norm.ztransform.Rec(mat=Xtest,cm=Xt$cm,cv=Xt$cv)
	Xtest <- Xt2$mat

#Xtrain <- Xtrain + abs(min(Xtrain)) ###non-neg check
	
	return(list(Xtrain=Xtrain,Xtest=Xtest))
}
getVisualU.1 <- function(L)
{
	K = 4
	if(L == 1) return(matrix(1,L,K))
		
	Lorig = L
	if(Lorig < 10) Lorig = 10
	U <- matrix(0,Lorig,K)
	U[,1] <- sin((1:Lorig)/(Lorig/(pi*6))) #(pi*5.5)
	U[,2] <- cos((1:Lorig)/(Lorig/(pi*6))) #cosine
	U[,3] <- c((1:Lorig-Lorig/2)^2); U[,3] <- (U[,3] - mean(U[,3]))/sd(U[,3]) #quadratic
	U[,4] <- as.matrix(2*((1:Lorig)/Lorig-0.5),Lorig,1) #linear
	U <- matrix(U[1:L,],nrow=L,ncol=K)
	U <- norm.ztransform(U)
#U <- U + abs(min(U))	###non-neg check
	
	#print(cor(U))
	#x11(); plot(U[,1])
	#x11(); plot(U[,2])
	#x11(); plot(U[,3])
	#x11(); plot(U[,4])	
	return(U)
}
getVisualAlpha.1 <- function(M)
{
	M1 = ceiling(M/2)
	M2 = floor(M/2)
	alpha <- matrix(0,M,4)    # Component precisions for the two data sets
	for(m in 1:M1)
	alpha[m,] <- c(1,1,1e6,1) # 1   = active (the value determines the data scale)
	if(M2 > 0)
	for(m in 1:M2)	
	alpha[m+M1,] <- c(1,1,1,1e6) # 1e6 = inactive
	return(alpha)
}
getVisualAlpha.1.Sparse <- function(M,D,N)
{
	alpha <- getVisualAlpha.1(M)
	K <- ncol(alpha)
	alphaD <- list()
	v <- N
	for(m in 1:M)
	{
		if(D[m] < N)
			v <- D[m]
		alphaD[[m]] <- matrix(1e6,D[m],K)	#everything off	
		for(k in 1:K)
		{
			if(alpha[m,k] < 1e3) #view active
				alphaD[[m]][sample(D[m],v),k] <- 1 #sample at max v values to be 1
		    #N = 100; D = 100; mean(1/rgamma(10000,N/D,1) <= N)*D - gives stochastically ~cN values less than N, irrespective of D
		}
	}
	return(list(alpha=alpha,alphaD=alphaD))
}
getVisualComponents.1 <- function(Ntrain,Ntest,L,D,M,sparse=FALSE)
{
	# Do Not normalize sparse=TRUE components, as dimensions with zero variance get scaled up badly.
	X <- getVisualX.1(Ntrain,Ntest)
	U <- getVisualU.1(L)
	alphaD <- NULL
	if(sparse)
	{
		a <- getVisualAlpha.1.Sparse(M,D,Ntrain)
		alpha <- a$alpha; alphaD <- a$alphaD;
		W <- getRandomW.Sparse(alpha,alphaD)
	}
	else
	{
		alpha <- getVisualAlpha.1(M)
		W <- getRandomW(D,alpha)
	}
	return(list(Xtrain=X$Xtrain,Xtest=X$Xtest,U=U,W=W,alpha=alpha,K=4,L=L,D=D,M=M,Ntrain=Ntrain,Ntest=Ntest,alphaD=alphaD))
}

### ICA Component Generation
getRandomU.ICA <- function(L,K)
{
	if(L == 1) return(matrix(1,L,K))
	U <- matrix(0,L,K)
	for(k in 1:K)
		U[,k] <- rnorm(L) #TODO: should be drawn from alphaU
	U <- norm.ztransform(U)		
	return(U)
}
getRandomX.ICA <- function(Ntrain,Ntest,K)
{
	if(Ntrain == 1) return(list(Xtrain=matrix(1,Ntrain,K),Xtest=matrix(1,Ntest,K)))
	
	X <- matrix(0,Ntrain+Ntest,K)
	for(k in 1:K)
		X[,k] <- rnorm(Ntrain+Ntest)
	Xtrain <- norm.ztransform(X[1:Ntrain,])
	Xtest <- norm.ztransform(X[(1:Ntest)+Ntrain,])
	return(list(Xtrain=Xtrain,Xtest=Xtest))
}
getRandomComponents.ICA <- function(Ntrain,Ntest,L,D,M,K)
{
	X <- getRandomX.ICA(Ntrain,Ntest,K)
	U <- getRandomU.ICA(L,K)
	alpha <- getRandomAlpha.UniformOverCardinality(M,K)
	W <- getRandomW(D,alpha)
	return(list(Xtrain=X$Xtrain,Xtest=X$Xtest,U=U,W=W,alpha=alpha,K=K,L=L,D=D,M=M,Ntrain=Ntrain,Ntest=Ntest))
}
###
getRandomU <- function(L,K)
{
	if(L == 1) return(matrix(1,L,K))
	U <- matrix(0,L,K)
	for(l in 1:L)
		U[l,] <- rnorm(K) #TODO: should be drawn from alphaU
	U <- norm.ztransform(U)
	return(U)
}

norm.ztransform.Rec <- function(mat,cm=NULL,cv=NULL)
{
  cmat = ncol(mat)
  rmat = nrow(mat)
  if(length(cm)==0)
	cm = apply(mat,2,"mean")
  mat = mat - matrix(cm,rmat,cmat,byrow=TRUE)
  if(length(cv)==0)
	cv = apply(mat,2,"sd")
  mat = mat/matrix(cv,rmat,cmat,byrow=TRUE)
  #cv1 = mean(apply(mat,2,"var"))
  #print(cv1)
  return(list(mat=mat,cm=cm,cv=cv))
}

getRandomX <- function(Ntrain,Ntest,K)
{
	if(Ntrain == 1) return(list(Xtrain=matrix(1,Ntrain,K),Xtest=matrix(1,Ntest,K)))
	
	N = Ntrain+Ntest
	X <- matrix(0,N,K)
	for(n in 1:N)
		X[n,] <- rnorm(K)#(K,0,abs(rnorm(1)))

	Xtrain <- X[1:Ntrain,]
	Xtest <- X[(1:Ntest)+Ntrain,]

	Xt <- norm.ztransform.Rec(Xtrain)
	Xtrain <- Xt$mat
	Xt2 <- norm.ztransform.Rec(mat=Xtest,cm=Xt$cm,cv=Xt$cv)
	Xtest <- Xt2$mat
	
	return(list(Xtrain=Xtrain,Xtest=Xtest))
}
getRandomW <- function(D,alpha)
{
	M <- nrow(alpha)
	K <- ncol(alpha)
	W <- vector("list",length=M)
	# Create some random projection vectors
	for(view in 1:M) {
	  W[[view]] <- matrix(0,D[view],K)
	  for(k in 1:K) {
	  if(1/alpha[view,k] < 1e-3)
	    W[[view]][,k] <- 0
	  else
	  {
	    W[[view]][,k] <- rnorm(D[view],0,1/sqrt(alpha[view,k]))
	    #W[[view]][,k] <- (W[[view]][,k] - mean(W[[view]][,k]))/sd(W[[view]][,k])
	    #W[[view]][,k] <- W[[view]][,k] + abs(min(W[[view]][,k]))		  ###for non-neg component check
	  }
	  }

	}
	return(W)
}
getRandomW.Sparse <- function(alpha,alphaD)
{
	M <- length(alphaD)
	W <- vector("list",length=M)
	K <- ncol(alphaD[[1]])
	# Create some random projection vectors
	for(view in 1:M) {
	  D <- nrow(alphaD[[view]])
	  W[[view]] <- matrix(0,D,K)
	  for(d in 1:D) {
	  for(k in 1:K) {
	  if(1/alpha[view,k] < 1e-3)
	    W[[view]][d,k] <- 0
	  else
	    W[[view]][d,k] <- rnorm(1,0,1/sqrt(alphaD[[view]][d,k]))
	  }
	  }
	}
	return(W)
}
getRandomComponents <- function(Ntrain,Ntest,L,D,M,K,card="Power")
{
	X <- getRandomX(Ntrain,Ntest,K)
	U <- getRandomU(L,K)
	if(card=="Uniform")
		alpha <- getRandomAlpha.UniformOverCardinality(M,K)
	if(card=="Power")
		alpha <- getRandomAlpha.PowerlawOverCardinality(M,K)
	if(card=="Custom1") #1 specific per component, 1 shared with another, 1 with all
		alpha <- getAlpha.Custom1(M,K)
	if(card=="Custom1p2") #above + 2 sp to 1st K.
		alpha <- getAlpha.Custom1p2(M,K)
	if(card=="Custom2") #pattern, all possible components with power of 2
		alpha <- getAlpha.Custom2(M,K)		
	if(card=="ONN") #all onn
		alpha <- getAlpha.ONN(M,K)
	W <- getRandomW(D,alpha)
	return(list(Xtrain=X$Xtrain,Xtest=X$Xtest,U=U,W=W,alpha=alpha,K=K,L=L,D=D,M=M,Ntrain=Ntrain,Ntest=Ntest))
}

getAlpha.ONN <- function(M,K)
{
	alpha <- matrix(1,M,K) #all switched onn
	return(alpha)
}

getAlpha.Custom2 <- function(M,K)
{
	#pattern, all possible components with power of 2
	if(K != (2*M-1))
	{
		print("ERROR cant apply Custom2 component setting. K != (2*M-1)")
		return(NULL)
	}
	alpha <- matrix(1e6,M,K) #all switched off
	
	for(m in 1:M)
		alpha[m,m] = 1 # 1/16

	if(M>=2)
	for(m in 1:(M/2))
		alpha[ (m*2-1):(m*2),M+m] = 1 #1/8

	if(M>=4)
	for(m in 1:(M/4))
		alpha[ (m*4-3):(m*4),M+M/2+m] = 1 #1/4

	if(M>=8)
	for(m in 1:(M/8))
		alpha[ (m*8-7):(m*8),M+M/2+M/4+m] = 1 #1/2

	if(M>=16)
	for(m in 1:(M/16))
		alpha[ (m*16-15):(m*16),M+M/2+M/4+M/8+m] = 1 #1/1

	return(alpha)
}

getAlpha.Custom1 <- function(M,K)
{
	if(K != 2*M)
	{
		print("ERROR cant apply Custom1 component setting. K != 2*M")
		return(NULL)
	}
	alpha <- matrix(1e6,M,K) #all switched off
	for(m in 1:M)
		alpha[m,m] = 1 # one component specific to each view

	for(m in 1:(M-1))
		alpha[m:(m+1),M+m] = 1 #shared with next view
		
	alpha[,2*M] = 1 #shared in all views

	return(alpha)
}

getAlpha.Custom1p2 <- function(M,K)
{
	if(K != (2*M+2))
	{
		print("ERROR cant apply Custom1 component setting. K != 2*M+2")
		return(NULL)
	}
	alpha <- matrix(1e6,M,K) #all switched off
	for(m in 1:M)
		alpha[m,m] = 1 # one component specific to each view

	for(m in 1:(M-1))
		alpha[m:(m+1),M+m] = 1 #shared with next view
		
	alpha[,2*M] = 1 #shared in all views
	alpha[1,(2*M+1):(2*M+1)] = 1 # 2 more components specific to first view added
	
	return(alpha)
}


getRandomAlpha.UniformOverCombinations <- function(M,K)
{
	#alpha <- matrix(0,M,K)
	#runif(M*K)
}

getRandomAlpha.PowerlawOverCardinality <- function(M,K)
{
	alpha <- matrix(1e6,M,K) #all switched off	
	#ceil(40*(.9^(c(1:40))))
	
	y = runif(K)
	x0 = 1
	x1 = M
	n = -1.05
	x = ((x1^(n+1) - x0^(n+1))*y + x0^(n+1))^(1/(n+1))-1
	x = ceiling(x)
	x[x == 0] = 1
	x[1] = M
	#hist(ceil(x))
	for(k in 1:K)
	{
		a_count = x[k]
		inds = sample(1:M,a_count,replace=FALSE)
		alpha[inds,k] = 1 #switch on
	}
	print(x)
	print(rowSums(alpha==1)>0)
	return(alpha)
}
getRandomAlpha.UniformOverCardinality <- function(M,K)
{
	alpha <- matrix(1e6,M,K) #all switched off
	for(k in 1:K)
	{
		a_count = sample(1:M,1)
		inds = sample(1:M,a_count,replace=FALSE)
		alpha[inds,k] = 1 #switch on
	}
	return(alpha)
}

getNoiseTensor <- function(data,tau,dim.noise=FALSE)
{
  if(dim.noise)
	return(getNoiseTensor.Dimensional(data,tau))
	
  ts <- sqrt(get.total.var.tensor(data))
  #print(ts)
  noise <- array(rnorm( det(diag(dim(data))),mean=0,sd=(1/sqrt(tau)) ),dim=dim(data))*ts
  return(noise)
}
getNoiseTensor.Dimensional <- function(data,tau)
{
  N <- dim(data)[1]
  D <- dim(data)[2]
  L <- dim(data)[3]
  if(length(tau) != D)
  	print("Length of Tau does not matches")

  noise <- array(0,dim=dim(data))
  for(d in 1:D)
  	noise[,d,] <- rnorm(N*L,0,1/sqrt(tau[d])) #sd(as.vector(data[,d,]))
  return(noise)
}

#
# 2 components shared in view, 2 specific
# 100% Collinear U is the Worst Case. See SBMSS Paper.
#
#
# 
#
data.train.test.1 <- function(CP,tau=c(10,5),dim.noise=FALSE,addNoise=TRUE,normMethod = NULL,normTotalVar=TRUE)
{
K <- CP$K; D <- CP$D; M <- CP$M; L <- CP$L; Ntrain <- CP$Ntrain; Ntest <- CP$Ntest
U <- CP$U; W <- CP$W; alpha <- CP$alpha
Xtrain <- CP$Xtrain; Xtest <- CP$Xtest #X <- CP$X;

# Sample data from the model.
CP$X <- Xtrain
Ytrain <- make.tensor(CP,kk="all")
CP$X <- Xtest
Ytest <- make.tensor(CP,kk="all")
CP$X <- NULL

if(normTotalVar)
for(view in 1:M){
  tmp <- norm.total.var(Ytrain[[view]])
  Ytrain[[view]] <- tmp$data
  #print(tmp$pre)
  tmp  <- norm.total.var(Ytest[[view]])
  Ytest[[view]] <- tmp$data
}

OrigTensor <- list()
OrigTensor$Ytrain <- Ytrain
OrigTensor$Ytest <- Ytest

#should not normalize the toy data here.. because then comparison with orig tensor becomes tricky
norm.method <- "norm.none" #"norm.slabscaling" #"norm.tensor.K2009" #"norm.slab" #"norm.doublecentring" #"norm.none"
if(length(normMethod)>0) #overwrite default choice
	norm.method <- normMethod
normalize <- get(norm.method)
Pre <- list(); Pre$Ytrain <- list(); Pre$Ytest <- list()
for(view in 1:M) {
  if(norm.method == "norm.slabscaling")
	W[[view]] <- adjustWscaling(Ytrain[[view]],W[[view]])

  ##Notes on Normalization
  #1) Normalize data with total.var -> introduces low variance structural replica's, with correlated 2 loading views.
  #2) Normalize data with K2009 -> no extra components

  noise <- 0
  if(addNoise)
  	noise <- getNoiseTensor(Ytrain[[view]],tau[[view]],dim.noise)
  tmp <- normalize(Ytrain[[view]]+ noise) #norm.total.var generates mse of zero
  Ytrain[[view]] <- tmp$data
  Pre$Ytrain[[view]] <- tmp$pre
#  if(norm.method == "norm.total.rms") #comment this from here - and do in proper way if needed
#	OrigTensor$Ytrain[[view]] <- OrigTensor$Ytrain[[view]]/tmp$pre
  
  if(addNoise)
  	noise <- getNoiseTensor(Ytest[[view]],tau[[view]],dim.noise) 
  tmp <- normalize(Ytest[[view]]+ noise) #TODO: test should be normalized with training pre
  Ytest[[view]] <- tmp$data
  Pre$Ytest[[view]] <- tmp$pre  
}
return(list(Y=Ytrain,W=W,Ytest=Ytest,tau=tau,Z=Xtrain,Ztest=Xtest,D=D,L=L,M=M,N=Ntrain,K=K,U=U,alpha=alpha,norm.method=norm.method,dim.noise=dim.noise,Pre=Pre,OrigTensor=OrigTensor))
}

concatenateTensorViews <- function(dat)
{
	dd <- unlist(lapply(dat,function(x) { dim(x)[2]}))
	#ten <- array(0,dim=c(dim(tensor.data$Y[[1]])[1],sum(dd),dim(tensor.data$Y[[1]])[3]))
	ten <- array(0,dim=c(dim(dat[[1]])[1],sum(dd),dim(dat[[1]])[3]))
	M <- length(dat)
	
	st = 1
	en = dd[1]
	ten[,st:en,] = dat[[1]]
	if(M>1)
	for(m in 2:M)
	{
		st = sum(dd[1:(m-1)])+1
		en = sum(dd[1:m])
		ten[,st:en,] = dat[[m]]
	}
	return(ten)
}
concatenateMatrix <- function(lm)
{
	M <- length(lm)
	mat <- lm[[1]]
	if(M > 1)
	for(m in 2:M)
		mat <- rbind(mat,lm[[m]])
	return(mat)
}

adjustWscalingPostRun <- function(ten.data)
{
	CP <- list()
	CP$W <- ten.data$W
	CP$X <- ten.data$Z
	CP$U <- ten.data$U	
	Ytrain <- make.tensor(CP,kk="all")
	W <- ten.data$W
	M <- length(W)
	for(m in 1:M)
		W[[m]] <- adjustWscaling(Ytrain[[m]],W[[m]])
		
	ten.data$W	<- W
	return(ten.data)
}
adjustWscaling <- function(ten,W)
{
	sc <- getWrescales(ten)
	W <- W/matrix(rep(sc,ncol(W)),nrow(W),ncol(W),byrow=FALSE)
	#print("adjustWscaling"); browser()
	return(W)
}

##Component found by TGFA but missed by GFA
data.train.test.X <- function(D=c(15*1,7*1),L=2,M=2,N=100,SNR=c(10,5))
{
#for(view in 1:2) {
#  W[[view]] <- matrix(0,D[view],K)
#  for(k in 1:K) {
#    W[[view]][,k] <- rnorm(D[view],0,1/sqrt(alpha[view,k]))
#  }
#  tmp <- Z[,c(2:4)] %*% t(W[[view]][,c(2:4)])
#  Y[[view]] <- array(0,dim=c(nrow(tmp),ncol(tmp),2))
#  if(view == 1)
#  	tmp <- Z[,c(1:4)] %*% t(W[[view]][,c(1:4)])
#  Y[[view]][,,1] <- tmp + matrix(rnorm(N*D[view],0,1/sqrt(tau[view])),N,D[view])

#  tmp <- Z[,c(2:4)] %*% t(W[[view]][,c(2:4)])
#  if(view == 2)
#  	tmp <- Z[,c(1:4)] %*% t(W[[view]][,c(1:4)])
#  Y[[view]][,,2] <- tmp + matrix(rnorm(N*D[view],0,1/sqrt(tau[view])),N,D[view])
#  
#  #Ygfa[[view]] <- cbind(tmp,tmp*0.8 + matrix(rnorm(nrow(tmp)*ncol(tmp),0,.8),nrow(tmp),ncol(tmp)))
#}
}


#
# Create a GFA data set for a tensor dataset
#
data.tensor.to.gfa <- function(data)
{
Y <- data$Y
Ytest <- data$Ytest
W <- data$W
U <- data$U
ot <- data$OrigTensor[[1]]

M = length(Y)
L = dim(Y[[1]])[3]
Ygfa <- vector("list",length=M*L)
Ygfatest <- vector("list",length=M*L)
Wgfa <- vector("list",length=M*L)
otgfa <- vector("list",length=M*L)
SNR <- 0
tau <- 0
if(data$dim.noise)
	tau = list()
kk = 1
  for(m in 1:M)
  for(l in 1:L)
  {
   Ygfatest[[kk]] <- Ytest[[m]][,,l]
   Ygfa[[kk]] <- Y[[m]][,,l]
   otgfa[[kk]] <- ot[[m]][,,l]
   Wgfa[[kk]] <- W[[m]]*matrix(rep(U[l,],nrow(W[[m]])), nrow(W[[m]]), ncol(W[[m]]),byrow=TRUE)
   #SNR[kk] <- data$SNR[m]
   if(data$dim.noise) tau[[kk]] <- data$tau[[m]]
   else   tau[kk] <- data$tau[m]
   kk = kk + 1
  }
  
return(list(Y=Ygfa,W=Wgfa,Ytest=Ygfatest,tau=tau,Z=data$Z,Ztest=data$Ztest,D=data$D,L=data$L,M=data$M,N=data$N,K=data$K,U=data$U,alpha=data$alpha,OrigTensor=otgfa))  
}

#
# Create a POOLED GFA data set for a tensor dataset
#
pool.tensor <- function(Y)
{
	Ygfa <- list(); L = dim(Y[[1]])[3]
	for(m in 1:M)
	{
	  l = 1
	  Ygfa[[m]] <- Y[[m]][,,l]
	  for(l in 2:L)
	   Ygfa[[m]] <- rbind(Ygfa[[m]],Y[[m]][,,l])
	Y[[m]] = array(Ygfa[[m]],dim=c(nrow(Ygfa[[m]]),ncol(Ygfa[[m]]),1))   
	print(dim(Y[[m]]))
	}
	return(Y)
}

data.tensor.to.pooled.tensor <- function(data)
{
	data = data.tensor.to.pooled.gfa(data)
	M = length(data$Y)
	for(m in 1:M)
	{
		data$Y[[m]] = array(data$Y[[m]],dim=c(nrow(data$Y[[m]]),ncol(data$Y[[m]]),1))
		data$Ytest[[m]] = array(data$Ytest[[m]],dim=c(nrow(data$Ytest[[m]]),ncol(data$Ytest[[m]]),1))
		data$OrigTensor[[m]] = array(data$OrigTensor[[m]],dim=c(nrow(data$OrigTensor[[m]]),ncol(data$OrigTensor[[m]]),1))
	}
	return(data)
}

data.tensor.to.pooled.gfa <- function(data)
{
Y <- data$Y
Ytest <- data$Ytest
W <- data$W
U <- data$U
ot <- data$OrigTensor[[1]]

M = length(Y)
L = dim(Y[[1]])[3]
Ygfa <- vector("list",length=M)
Ygfatest <- vector("list",length=M)
Wgfa <- vector("list",length=M)
otgfa <- vector("list",length=M)
SNR <- 0
tau <- 0
if(data$dim.noise)
	tau = list()
for(m in 1:M)
{
  l = 1
  Ygfatest[[m]] <- Ytest[[m]][,,l]
  Ygfa[[m]] <- Y[[m]][,,l]
  otgfa[[m]] <- ot[[m]][,,l]
  Wgfa[[m]] <- W[[m]]*matrix(rep(U[l,],nrow(W[[m]])), nrow(W[[m]]), ncol(W[[m]]),byrow=TRUE)
  if(L>=2)
  for(l in 2:L)
  {
   Ygfatest[[m]] <- rbind(Ygfatest[[m]],Ytest[[m]][,,l])
   Ygfa[[m]] <- rbind(Ygfa[[m]],Y[[m]][,,l])
   otgfa[[m]] <- rbind(otgfa[[m]],ot[[m]][,,l])
   Wgfa[[m]] <- rbind(Wgfa[[m]],W[[m]]*matrix(rep(U[l,],nrow(W[[m]])), nrow(W[[m]]), ncol(W[[m]]),byrow=TRUE))
  }
  if(data$dim.noise) tau[[m]] <- data$tau[[m]]
  else   tau[m] <- data$tau[m]
}
  
return(list(Y=Ygfa,W=Wgfa,Ytest=Ygfatest,tau=tau,Z=data$Z,Ztest=data$Ztest,D=data$D,L=1,M=data$M,N=data$N*data$L,K=data$K,U=rep(1,data$K),alpha=data$alpha,OrigTensor=otgfa))  
}

getDataSettingString <- function(data)
{
	str <- paste("M=",data$M,", L=",data$L,", D=",paste(data$D,collapse="-"),", N=",data$N,", SNR=",paste(data$SNR,collapse="-"),sep="")
	return(str)
}
