# Copyright (C) 2010 Tommi Suvitaival and Ilkka Huopaniemi
#
# This file is part of multiWayCCA.
#
# multiWayCCA is free software: you can redistribute it and/or modify
# it under the terms of the Lesser GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# multiWayCCA is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with multiWayCCA.  If not, see <http://www.gnu.org/licenses/>.


## MANOVA functions

## Multi-level 3-way version
# 2.11.10 - bug fixed: matrix 'eff' transposed
# 15.10.10 - Argument 'z' added.
# 23.9.10

# Call: effects[sampleLevels,] = sampleEffMat(design=design, effects=effects, sampleLevels=levels$A, multiWay=TRUE)
sampleEffMat = function(z, design, effects, sampleLevels, multiWay=TRUE) {

# 	eff = array(data=NA, dim=c(ncol(effects),length(sampleLevels)))
	eff = array(data=NA, dim=c(length(sampleLevels),ncol(effects)))
	eff0 = rep(x=NA, times=nrow(z))
	sizePopu = NA

	for (i in 1:length(sampleLevels)) { # Go through all levels of the covariate.
		popu = which(design[,sampleLevels[i]]==1)
		sizePopu = length(popu)
		if (sizePopu>0) {
# 			if (ncol(design)>length(sampleLevels)) {
			if (multiWay) {
				eff0 = rowSums(as.matrix(z[,popu]-t(design[popu,-sampleLevels]%*%effects[-sampleLevels,]), nrow=K))
			} else {
				eff0 = rowSums(as.matrix(z[,popu],nrow=K))
			}
			eff0 = eff0/(sizePopu+1)
# 			eff[,i] = rnorm(n=length(eff0), mean=eff0, sd=sqrt(1/(sizePopu+1))) # sampling
			eff[i,] = rnorm(n=length(eff0), mean=eff0, sd=sqrt(1/(sizePopu+1))) # sampling
		} else {
# 			eff[,i] = rep(x=0, times=K)
			eff[i,] = rep(x=0, times=K)
		}
	}
	
	return(eff)
	
}

# sampleEff - a list of boolean values containing entries 'effB', 'effC', 'effAB', 'effAC', 'effBC', 'effABC'
sampleEffA = function(z, covA, covB, covC, effB, effC, effAB, effAC, effBC, effABC, sampleEff) {

	K = nrow(z) # number of clusters
	A = ncol(effAB)
	
	effA = array(0, dim=c(K,A))
	
	if (A>1) {
		B = ncol(effB)
		C = ncol(effC)
		effA0 = rep(x=NA, times=K)
		sizePopu = NA
		sizeSubPopu = NA
		for (ai in 2:A) { # Go through all levels of covariate 'a'.
			popu = which(covA==ai)
			sizePopu = length(popu)
			if (sizePopu>0) {
				effA0 = rowSums(as.matrix(z[,popu],nrow=K))
				if (B>1 & (sampleEff$B|sampleEff$AB)) {
					for (bi in 2:B) { # Go through all levels of covariate 'b' to remove the b-effects.
						sizeSubPopu = length(which(covA==ai&covB==bi))
						if (sizeSubPopu>0) {
							effA0 = effA0 - sizeSubPopu*(effB[,bi]+effAB[,ai,bi])
						}
					}
				}
				if (C>1 & (sampleEff$C|sampleEff$AC|sampleEff$BC|sampleEff$ABC)) {
					for (ci in 2:C) { # Go through all levels of covariate 'c' to remove the c-effects.
						sizeSubPopu = length(which(covA==ai&covC==ci))
						if (sizeSubPopu>0) {
							effA0 = effA0 - sizeSubPopu*(effC[,ci]+effAC[,ai,ci])
							if (B>1 & (sampleEff$BC|sampleEff$ABC)) {
								for (bi in 2:B) { # Go through all levels of covariate 'b' for interaction with 'c'.
									sizeSubPopu = length(which(covA==ai&covB==bi&covC==ci))
									if (sizeSubPopu>0) {
										effA0 = effA0 - sizeSubPopu*(effBC[,bi,ci]+effABC[,ai,bi,ci])
									}
								}
							}
						}
					}
				}
				effA0 = effA0/(sizePopu+1)
				effA[,ai] = rnorm(mean=effA0, sd=sqrt(1/(sizePopu+1))) # sampling
			}
		}
	}

	return(effA)

}

# sampleEff - a list of boolean values containing entries 'effA', 'effB', 'effC', 'effAC', 'effBC', 'effABC'
sampleEffAB = function(z, covA, covB, covC, effA, effB, effC, effAC, effBC, effABC, sampleEff) {

	K = nrow(z) # number of clusters
	A = ncol(effA)
	B = ncol(effB)
	
	effAB = array(0, dim=c(K,A,B))
	
	if (A>1 & B>1) {
		C = ncol(effC)
		effAB0 = rep(x=NA, times=K)
		sizePopu = NA
		sizeSubPopu = NA
		for (ai in 2:A) { # Go through all levels of covariate 'a'.
			for (bi in 2:B) { # Go through all levels of covariate 'b'.
				popu = which(covA==ai&covB==bi)
				sizePopu = length(popu)
				if (sizePopu>0) {
					effAB0 = rowSums(as.matrix(z[,popu],nrow=K)) - sizePopu*(effA[,ai]+effB[,bi])
					for (ci in 2:C) { # Go through all levels of covariate 'c' to remove the c-effects.
						sizeSubPopu = length(which(covA==ai&covB==bi&covC==ci))
						if (sizeSubPopu>0) {
							effAB0 = effAB0 - sizeSubPopu*(effC[,ci]+effAC[,ai,ci]+effBC[,bi,ci]+effABC[,ai,bi,ci])
						}
					}
					effAB0 = effAB0/(sizePopu+1)
					effAB[,ai,bi] = rnorm(mean=effAB0, sd=sqrt(1/(sizePopu+1))) # sampling
				}
			}
		}
	}

	return(effAB)

}

# sampleEff - a list of boolean values containing entries 'effA', 'effB', 'effC', 'effAB', 'effAC', 'effBC'
sampleEffABC = function(z, covA, covB, covC, effA, effB, effC, effAB, effAC, effBC, sampleEff) {

	K = nrow(z) # number of clusters
	A = ncol(effA)
	B = ncol(effB)
	C = ncol(effC)
	
	effABC = array(0, dim=c(K,A,B,C))

	if (A>1 & B>1 & C>1) {
		effABC0 = rep(x=NA, times=K)
		sizePopu = NA
		for (ai in 2:A) { # Go through all levels of covariate 'a'.
			for (bi in 2:B) { # Go through all levels of covariate 'b'.
				for (ci in 2:C) { # Go through all levels of covariate 'c'.
					popu = which(covA==ai&covB==bi&covC==ci)
					sizePopu = length(popu)
					if (sizePopu>0) {
						effABC0 = rowSums(as.matrix(z[,popu],nrow=K)) - sizePopu*(effA[,ai]+effB[,bi]+effC[,ci]+effAB[,ai,bi]+effAC[,ai,ci]+effBC[,bi,ci])
					}
					effABC0 = effABC0/(sizePopu+1)
					effABC[,ai,bi,ci] = rnorm(mean=effABC0, sd=sqrt(1/(sizePopu+1))) # sampling
				}
			}
		}
	}
	
	return(effABC)

}

## end of multi-level 3-way version


# 4.8.09 - also effects mu_c, mu_g and mu_cg sampled from 1 to S to cope with 2-level 3-way data

sampleMu_c = function(z,case,gender,state,mu_g,mu_s,mu_cg,mu_sc,mu_sg,mu_scg) {

	K = nrow(z) # klusterien lkm.
	#S = dim(mu_s)[2] # HMM-tilojen lkm.
	S = max(state)
	
	# ajasta riippumattomien parametrien vaikutus
	mu_c0 = rowSums(matrix(z[,(case==1)],nrow=K)) - sum(case==1&gender==1)*(mu_g+mu_cg)
	# Poistetaan tilariippuvien parametrien vaikutus.
	if (S>0) {
		for (s in 1:S) { # 's' goes from 2 to 'S' because time-point 1 does not have time-specific effect. - 3.4.09
			mu_c0 = mu_c0 - ( sum(state==s&case==1)*mu_s[,s] + sum(state==s&case==1)*mu_sc[,s] + sum(state==s&case==1&gender==1)*mu_sg[,s] + sum(state==s&case==1&gender==1)*mu_scg[,s] )
		}
	}
	Ns = sum(case==1)
	mu_c0 = mu_c0 / (Ns+1)
	mu_c = mvrnorm(mu=mu_c0,Sigma=diag(1/(Ns+1),nrow=K))
	
	return(mu_c)

}

sampleMu_g = function(z,case,gender,state,mu_c,mu_s,mu_cg,mu_sc,mu_sg,mu_scg) {

	K = nrow(z) # klusterien lkm.
	#S = dim(mu_s)[2] # HMM-tilojen lkm.
	S = max(state)

	# ajasta riippumattomien parametrien vaikutus
	mu_g0 = rowSums(matrix(z[,(gender==1)],nrow=K)) - sum(case==1&gender==1)*(mu_c+mu_cg)
	# Poistetaan tilariippuvien parametrien vaikutus.
	if (S>0) {
		for (s in 1:S) { # 's' goes from 2 to 'S' because time-point 1 does not have time-specific effect. - 3.4.09
			mu_g0 = mu_g0 - ( sum(state==s&gender==1)*mu_s[,s] + sum(state==s&case==1&gender==1)*mu_sc[,s] + sum(state==s&gender==1)*mu_sg[,s] + sum(state==s&case==1&gender==1)*mu_scg[,s] )
		}
	}
	Ns = sum(gender==1)
	mu_g0 = mu_g0 / (Ns+1)
	mu_g = mvrnorm(mu=mu_g0,Sigma=diag(1/(Ns+1),nrow=K))

	return(mu_g)

}

sampleMu_cg = function(z,case,gender,state,mu_c,mu_g,mu_s,mu_sc,mu_sg,mu_scg) {

	K = nrow(z) # klusterien lkm.
	#S = dim(mu_s)[2] # HMM-tilojen lkm.
	S = max(state)

	# ajasta riippumattomien parametrien vaikutus
	mu_cg0 = rowSums(matrix(z[,(case==1&gender==1)],nrow=K)) - sum(case==1&gender==1)*(mu_c+mu_g)
	# Poistetaan tilariippuvien parametrien vaikutus.
	if (S>0) {
		for (s in 1:S) { # 's' goes from 2 to 'S' because time-point 1 does not have time-specific effect. - 3.4.09
			mu_cg0 = mu_cg0 - sum(state==s&case==1&gender==1)*( mu_s[,s]+mu_sc[,s]+mu_sg[,s]+mu_scg[,s] )
		}
	}
	Ns = sum(case==1&gender==1)
	mu_cg0 = mu_cg0 / (Ns+1)
	mu_cg = mvrnorm(mu=mu_cg0,Sigma=diag(1/(Ns+1),nrow=K))

	return(mu_cg)

}

sampleMu_s = function(z,case,gender,state,mu_c,mu_g,mu_cg,mu_sc,mu_sg,mu_scg) {

	K = nrow(z) # klusterien lkm.
	#S = dim(mu_sc)[2] # HMM-tilojen lkm.
	S = max(state)

	if (S>0) {
		mu_s = matrix(nrow=K,ncol=S)
		mu_s[,1] = 0 # No time point 1-specific effect. -3.4.09
		for (s in 1:S) { # 's' goes from 2 to 'S' because time-point 1 does not have time-specific effect. - 3.4.09
			Ns = sum(state==s) # havaintojen lkm. HMM-tilassa 's'
			mu_s0 = ( rowSums(matrix(z[,(state==s)],nrow=K)) - ( sum(state==s&case==1)*mu_c + sum(state==s&gender==1)*mu_g + sum(state==s&case==1&gender==1)*mu_cg + sum(state==s&case==1)*mu_sc[,s] + sum(state==s&gender==1)*mu_sg[,s] + sum(state==s&case==1&gender==1)*mu_scg[,s] ) ) / (Ns+1)
			# Oletetaan kullekin 'mu_s':n klusterille sama varianssi.
			Sgm = diag(1/(Ns+1),nrow=K)
			mu_s[,s] = mvrnorm(mu=mu_s0,Sigma=Sgm)
		}
	} else {
		mu_s = matrix(0,nrow=K,ncol=1)
	}

	return(mu_s)

}

sampleMu_sc = function(z,case,gender,state,mu_c,mu_g,mu_s,mu_cg,mu_sg,mu_scg) {

	K = nrow(z) # klusterien lkm.
	#S = dim(mu_s)[2] # HMM-tilojen lkm.
	S = max(state)

	if (S>0) {
		mu_sc = matrix(nrow=K,ncol=S)
# 		mu_sc[,1] = 0 # No time point 1-specific effect. -3.4.09
# 		for (s in 2:S) { # No time point 1-specific effect. -3.4.09
		for (s in 1:S) { # Experiment 3.4.09: Sample mu_s for 1:S, mu_sc for 2:S, and no mu_c.
			Ns = sum(state==s&case==1) # havaintojen lkm. HMM-tilassa 's'
			mu_sc0 = ( rowSums(matrix(z[,(state==s&case==1)],nrow=K)) - (sum(state==s&case==1)*mu_c + sum(state==s&case==1&gender==1)*mu_g + sum(state==s&case==1)*mu_s[,s] + sum(state==s&case==1&gender==1)*mu_cg + sum(state==s&case==1&gender==1)*mu_sg[,s] + sum(state==s&case==1&gender==1)*mu_scg[,s] ) ) / (Ns+1)
			# Oletetaan kullekin 'mu_s':n klusterille sama varianssi.
			Sgm = diag(1/(Ns+1),nrow=K)
			mu_sc[,s] = mvrnorm(mu=mu_sc0,Sigma=Sgm)
		}
	} else {
		mu_sc = matrix(0,nrow=K,ncol=1)
	}

	return(mu_sc)

}

sampleMu_sg = function(z,case,gender,state,mu_c,mu_g,mu_s,mu_cg,mu_sc,mu_scg) {

	K = nrow(z) # klusterien lkm.
	#S = dim(mu_s)[2] # HMM-tilojen lkm.
	S = max(state)

	if (S>0) {
		mu_sg = matrix(nrow=K,ncol=S)
# 		mu_sg[,1] = 0 # No time point 1-specific effect. -3.4.09
# 		for (s in 2:S) { # No time point 1-specific effect. -3.4.09
		for (s in 1:S) { # Experiment 3.4.09: Sample mu_s for 1:S, mu_sc for 2:S, and no mu_c.
			Ns = sum(state==s&gender==1) # havaintojen lkm. HMM-tilassa 's'
			mu_sg0 = ( rowSums(matrix(z[,(state==s&gender==1)],nrow=K)) - ( 
							sum(state==s&case==1&gender==1)*mu_c + sum(state==s&gender==1)*mu_g + sum(state==s&gender==1)*mu_s[,s] + sum(state==s&case==1&gender==1)*mu_cg + sum(state==s&case==1&gender==1)*mu_sc[,s] + sum(state==s&case==1&gender==1)*mu_scg[,s] ) ) / (Ns+1)
			# Oletetaan kullekin 'mu_s':n klusterille sama varianssi.
			Sgm = diag(1/(Ns+1),nrow=K)
			mu_sg[,s] = mvrnorm(mu=mu_sg0,Sigma=Sgm)
		}
	} else {
		mu_sg = matrix(0,nrow=K,ncol=1)
	}

	return(mu_sg)

}

sampleMu_scg = function(z,case,gender,state,mu_c,mu_g,mu_s,mu_cg,mu_sc,mu_sg) {

	K = nrow(z) # klusterien lkm.
	#S = dim(mu_s)[2] # HMM-tilojen lkm.
	S = max(state)

	if (S>0) {
		mu_scg = matrix(nrow=K,ncol=S)
# 		mu_scg[,1] = 0 # No time point 1-specific effect. -3.4.09
# 		for (s in 2:S) { # No time point 1-specific effect. -3.4.09
		for (s in 1:S) { # Experiment 3.4.09: Sample mu_s for 1:S, mu_sc for 2:S, and no mu_c.
			Ns = sum(state==s&case==1&gender==1) # havaintojen lkm. HMM-tilassa 's'
			mu_scg0 = ( rowSums(matrix(z[,(state==s&case==1&gender==1)],nrow=K)) - sum(state==s&case==1&gender==1) * (mu_c+mu_g+mu_s[,s]+mu_cg+mu_sc[,s]+mu_sg[,s]) ) / (Ns+1)
			# Oletetaan kullekin 'mu_s':n klusterille sama varianssi.
			Sgm = diag(1/(Ns+1),nrow=K)
			mu_scg[,s] = mvrnorm(mu=mu_scg0,Sigma=Sgm)
		}
	} else {
		mu_scg = matrix(0,nrow=K,ncol=1)
	}

	return(mu_scg)

}
