# Copyright (C) 2010 Tommi Suvitaival and Ilkka Huopaniemi
#
# This file is part of multiWayCCA.
#
# multiWayCCA is free software: you can redistribute it and/or modify
# it under the terms of the Lesser GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# multiWayCCA is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Lesser General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with multiWayCCA.  If not, see <http://www.gnu.org/licenses/>.


## Multi-way, multi-view analysis

multiWayCCA = function(data, nXlat, nYlat, sampleEff=NULL, takeLog, maxBurnIn, Niterfinal, path, runId) {

	time.start = Sys.time()

	## Create directories

	pathRun = paste(path,"results/",runId,"/", sep="")
	dir.create(pathRun) # Create a directory for plottings.
	if (file.exists(paste(path,"scripts/",runId,".R", sep="")))
		file.copy(paste(path,"scripts/",runId,".R",sep=""),paste(pathRun,runId,".R",sep="")) # Save this script.
	
	## Separate the data that is used by the sampler from possible data generation parameters.

	dataIn = list()
	dataIn$X = as.matrix(data$X)
	dataIn$Y = as.matrix(data$Y)
	
	# Restructure the covariates.
	data$covariatesList = list()
	data$covariatesList$a = data$covariates[1,]
	if (nrow(data$covariates)>1) {
		data$covariatesList$b = data$covariates[2,]
	} else {
		data$covariatesList$b = rep(x=1, times=length(data$covariatesList$a))
	}
	if (nrow(data$covariates)>2) {
		data$covariatesList$c = data$covariates[3,]
	} else {
		data$covariatesList$c = rep(x=1, times=length(data$covariatesList$a))
	}
	if (nrow(data$covariates)>3) {
		data$covariatesList$d = data$covariates[4,]
	} else {
		data$covariatesList$d = rep(x=1, times=length(data$covariatesList$a))
	}

	## More parameters. Do not change these unless you know what you are doing.

	Niter <- 5 # number of Gibbs samples to draw between printing
	unitScale = TRUE # TRUE/FALSE: do/do not scale variables to be of unit scale

	# Number of features in each view: they do not have to be equal.
	nX = nrow(dataIn$X) # the number of features in view X
	nY = nrow(dataIn$Y) # the number of features in view Y
	
	N = ncol(dataIn$X) # number of samples
	
	if (!is.null(takeLog)) { # Optionally, take logarithm of the data to make it more normally distributed. Watch out for zero values when taking the logarithm!
		if (takeLog$X)
			dataIn$X = log(dataIn$X+0.01)
		if (takeLog$Y)
			dataIn$Y = log(dataIn$Y+0.01)
	}

	seed = as.numeric(Sys.time())
	set.seed(seed)

	## Define sampling parameters

	param = list()
	
	param$zDim = 3 # dimensionality of the common latent variable in learning. Should be 3: one shared latent variable plus one data set-specific variable for each data set.

	param$arvonta = TRUE # FALSE/TRUE: do/do not allow empty clusters in 'xLat' and 'yLat'
	param$jama = FALSE # TRUE/FALSE: do/do not use left-over cluster in 'xLat' and 'yLat'
	param$sampling = list()
	param$sampling$MuX = FALSE # TRUE/FALSE: do/do not estimate the feature-specific mean parameter
	param$sampling$Mu = FALSE
	if (unitScale) { # If the data is scaled, do not estimate scale parameter.
		param$sampling$scale = FALSE
	} else {
		param$sampling$scale = TRUE
	}
	param$clustering = rep(x=TRUE, times=2) # vector of TRUE/FALSE values: cluster variables in the sources
	if (is.na(nXlat)) {
		param$clustering[1] = FALSE
		nXlat = nX
		colnames(dataIn$X) = NULL
	}
	if (is.na(nYlat)) {
		param$clustering[2] = FALSE
		nYlat = nY
		colnames(dataIn$Y) = NULL
	}
	
	## Define prior parameters
	setParameters <- function() {
		betaFirst <<- 1e-1
		betaPrior <<- 1e-1
		sigFirst <<- 1e-1
		sigPrior <<- 1e-1
		psi_prior_type <<- 1
		psi_prior <<- 1
		mu_vec <<- NULL # 'mu' no longer is a parameter of data but of latent variables.
		mu_prior <<- 1

		## Zero values of projection matrix 'W'
		# Limit some values of projection matrix 'W' to zero, leading to data set-specific components of latent variable 'z'.
		setWtoZero <<- array(0,dim=c(2,param$zDim))
		setWtoZero[1,3] <<- 1 # Third z does not interact with xlat.
		setWtoZero[2,2] <<- 1 # Second z does not interact with ylat.
	}
	setParameters()
	
	## END OF PARAMETERS

	## Covariate effects to be estimated

	if (is.null(sampleEff)) {
		sampleEff = list()
		sampleEff$A = TRUE
		sampleEff$B = TRUE
		sampleEff$C = TRUE
		sampleEff$D = TRUE
		sampleEff$AB = TRUE
		sampleEff$AC = TRUE
		sampleEff$AD = TRUE
		sampleEff$BC = TRUE
		sampleEff$BD = TRUE
		sampleEff$CD = TRUE
		sampleEff$ABC = TRUE
		sampleEff$ABD = TRUE
		sampleEff$ACD = TRUE
		sampleEff$BCD = TRUE
		sampleEff$ABCD = TRUE
	}

	param$sampleEffects = list()
	if (!is.null(sampleEff$A)) {
		param$sampleEffects$A = (length(which(data$covariatesList$a>1))>0 & sampleEff$A==TRUE)
	} else {
		param$sampleEffects$A = FALSE
	}
	if (!is.null(sampleEff$B)) {
		param$sampleEffects$B = (length(which(data$covariatesList$b>1))>0 & sampleEff$B==TRUE)
	} else {
		sampleEffects$B = FALSE
	}
	if (!is.null(sampleEff$C)) {
		param$sampleEffects$C = (length(which(data$covariatesList$c>1))>0 & sampleEff$C==TRUE)
	} else {
		sampleEffects$C = FALSE
	}
	if (!is.null(sampleEff$D)) {
		param$sampleEffects$D = (length(which(data$covariatesList$d>1))>0 & sampleEff$D==TRUE)
	} else {
		sampleEffects$D = FALSE
	}
	if (!is.null(sampleEff$AB)) {
		param$sampleEffects$AB = (length(which(data$covariatesList$a>1&data$covariatesList$b>1))>0 & sampleEff$AB==TRUE)
	} else {
		sampleEffects$AB = FALSE
	}
	if (!is.null(sampleEff$AC)) {
		param$sampleEffects$AC = (length(which(data$covariatesList$a>1&data$covariatesList$c>1))>0 & sampleEff$AC==TRUE)
	} else {
		sampleEffects$AC = FALSE
	}
	if (!is.null(sampleEff$AD)) {
		param$sampleEffects$AD = (length(which(data$covariatesList$a>1&data$covariatesList$d>1))>0 & sampleEff$AD==TRUE)
	} else {
		sampleEffects$AD = FALSE
	}
	if (!is.null(sampleEff$BC)) {
		param$sampleEffects$BC = (length(which(data$covariatesList$b>1&data$covariatesList$c>1))>0 & sampleEff$BC==TRUE)
	} else {
		sampleEffects$BC = FALSE
	}
	if (!is.null(sampleEff$BD)) {
		param$sampleEffects$BD = (length(which(data$covariatesList$b>1&data$covariatesList$d>1))>0 & sampleEff$BD==TRUE)
	} else {
		sampleEffects$BD = FALSE
	}
	if (!is.null(sampleEff$CD)) {
		param$sampleEffects$CD = (length(which(data$covariatesList$c>1&data$covariatesList$d>1))>0 & sampleEff$CD==TRUE)
	} else {
		sampleEffects$CD = FALSE
	}
	if (!is.null(sampleEff$ABC)) {
		param$sampleEffects$ABC = (length(which(data$covariatesList$a>1&data$covariatesList$b>1&data$covariatesList$c>1))>0 & sampleEff$ABC==TRUE)
	} else {
		sampleEffects$ABC = FALSE
	}
	if (!is.null(sampleEff$ABD)) {
		param$sampleEffects$ABD = (length(which(data$covariatesList$a>1&data$covariatesList$b>1&data$covariatesList$d>1))>0 & sampleEff$ABD==TRUE)
	} else {
		sampleEffects$ABD = FALSE
	}
	if (!is.null(sampleEff$ACD)) {
		param$sampleEffects$ACD = (length(which(data$covariatesList$a>1&data$covariatesList$c>1&data$covariatesList$d>1))>0 & sampleEff$ACD==TRUE)
	} else {
		sampleEffects$ACD = FALSE
	}
	if (!is.null(sampleEff$BCD)) {
		param$sampleEffects$BCD = (length(which(data$covariatesList$b>1&data$covariatesList$c>1&data$covariatesList$d>1))>0 & sampleEff$BCD==TRUE)
	} else {
		sampleEffects$BCD = FALSE
	}
	if (!is.null(sampleEff$ABCD)) {
		param$sampleEffects$ABCD = (length(which(data$covariatesList$b>1&data$covariatesList$c>1&data$covariatesList$d>1))>0 & sampleEff$ABCD==TRUE)
	} else {
		sampleEffects$ABCD = FALSE
	}

	## Convert covariates to be of the same form as in the implementation.

	# design - matrix, N x (A+B+C+D+A*B+A*C+A*D+B*C+B*D+C*D+A*B*C+A*B*D+A*C*D+B*C*D+A*B*C*D)
	dataIn$design = create4wayDesignMatrixWithEffects(covA=data$covariatesList$a, covB=data$covariatesList$b, covC=data$covariatesList$c, covD=data$covariatesList$d)

	## The data pre-processing

	# Transform the features to be zero-meaned.
# 	dataIn$X = dataIn$X-rowMeans(dataIn$X[,(data$covariates$a==1 & data$covariates$b==1 & data$covariates$c==1 & data$covariates$d==1)])
# 	dataIn$Y = dataIn$Y-rowMeans(dataIn$Y[,(data$covariates$a==1 & data$covariates$b==1 & data$covariates$c==1 & data$covariates$d==1)])
	dataIn$X = dataIn$X-rowMeans(dataIn$X[,(data$covariatesList$a==1 & data$covariatesList$b==1 & data$covariatesList$c==1 & data$covariatesList$d==1)])
	dataIn$Y = dataIn$Y-rowMeans(dataIn$Y[,(data$covariatesList$a==1 & data$covariatesList$b==1 & data$covariatesList$c==1 & data$covariatesList$d==1)])
	# Transform the features to be unit-scale.
	if (unitScale) {
# 		dataIn$X = dataIn$X/apply(dataIn$X[,(data$covariates$a==1 & data$covariates$b==1 & data$covariates$c==1 & data$covariates$d==1)],1,sd)
# 		dataIn$Y = dataIn$Y/apply(dataIn$Y[,(data$covariates$a==1 & data$covariates$b==1 & data$covariates$c==1 & data$covariates$d==1)],1,sd)
		dataIn$X = dataIn$X/apply(dataIn$X[,(data$covariatesList$a==1 & data$covariatesList$b==1 & data$covariatesList$c==1 & data$covariatesList$d==1)],1,sd)
		dataIn$Y = dataIn$Y/apply(dataIn$Y[,(data$covariatesList$a==1 & data$covariatesList$b==1 & data$covariatesList$c==1 & data$covariatesList$d==1)],1,sd)
	}

	## Compute empirical priors
	priors = list()
# 	priors$muX0 = rowMeans(dataIn$X[,(data$covariates$a==1 & data$covariates$b==1 & data$covariates$c==1 & data$covariates$d==1)])
# 	priors$muY0 = rowMeans(dataIn$Y[,(data$covariates$a==1 & data$covariates$b==1 & data$covariates$c==1 & data$covariates$d==1)])
	priors$muX0 = rowMeans(dataIn$X[,(data$covariatesList$a==1 & data$covariatesList$b==1 & data$covariatesList$c==1 & data$covariatesList$d==1)])
	priors$muY0 = rowMeans(dataIn$Y[,(data$covariatesList$a==1 & data$covariatesList$b==1 & data$covariatesList$c==1 & data$covariatesList$d==1)])
# 	priors$varsX0 = apply(dataIn$X[,(data$covariates$a==1 & data$covariates$b==1 & data$covariates$c==1 & data$covariates$d==1)],1,sd)
# 	priors$varsY0 = apply(dataIn$Y[,(data$covariates$a==1 & data$covariates$b==1 & data$covariates$c==1 & data$covariates$d==1)],1,sd)
	priors$varsX0 = apply(dataIn$X[,(data$covariatesList$a==1 & data$covariatesList$b==1 & data$covariatesList$c==1 & data$covariatesList$d==1)],1,sd)
	priors$varsY0 = apply(dataIn$Y[,(data$covariatesList$a==1 & data$covariatesList$b==1 & data$covariatesList$c==1 & data$covariatesList$d==1)],1,sd)
	priors$N0 = N # weight of the prior
	#priors$N0 = length(which(data$covariates$a==1 & data$covariates$b==1 & data$covariates$c==1)) # weight of the prior

	## SAMPLING

	## Initialize sampling

	print("Sampling:")
	print(paste(N," samples and ", nX, "+", nY, " variables", sep=""))
	print(paste("modeled by ", nXlat,"+", nYlat, " components", sep=""))
	print(paste("Covariate levels: A=", max(data$covariatesList$a), ", B=", max(data$covariatesList$b), ", C=", max(data$covariatesList$c), ", D=", max(data$covariatesList$d), sep=""))
	print("Number of samples per population:")
	for (ai in 1:max(data$covariatesList$a)) {
		for (bi in 1:max(data$covariatesList$b)) {
			for (ci in 1:max(data$covariatesList$c)) {
				for (di in 1:max(data$covariatesList$d)) {
					print(paste("a", ai, "-b", bi, "-c", ci, "-d", di, ": ", length(which(data$covariatesList$a==ai&data$covariatesList$b==bi&data$covariatesList$c==ci&data$covariatesList$d==di)), sep=""))
				}
			}
		}
	}
	tmp = "Effects estimated: {"
	if(param$sampleEffects$A)
		tmp = paste(tmp,"A")
	if(param$sampleEffects$B)
		tmp = paste(tmp,"B")
	if(param$sampleEffects$C)
		tmp = paste(tmp,"C")
	if(param$sampleEffects$D)
		tmp = paste(tmp,"D")
	if(param$sampleEffects$AB)
		tmp = paste(tmp,"AB")
	if(param$sampleEffects$AC)
		tmp = paste(tmp,"AC")
	if(param$sampleEffects$AD)
		tmp = paste(tmp,"AD")
	if(param$sampleEffects$BC)
		tmp = paste(tmp,"BC")
	if(param$sampleEffects$BD)
		tmp = paste(tmp,"BD")
	if(param$sampleEffects$CD)
		tmp = paste(tmp,"CD")
	if(param$sampleEffects$ABC)
		tmp = paste(tmp,"ABC")
	if(param$sampleEffects$ABD)
		tmp = paste(tmp,"ABD")
	if(param$sampleEffects$ACD)
		tmp = paste(tmp,"ACD")
	if(param$sampleEffects$BCD)
		tmp = paste(tmp,"BCD")
	if(param$sampleEffects$ABCD)
		tmp = paste(tmp,"ABCD")
	tmp = paste(tmp,"}")
	print(tmp)
	
	temp <- sampleMPCCA(data=dataIn, priors=priors, param=param, nIter=Niter, xlatDim=nXlat, ylatDim=nYlat, zDim=param$zDim)

	round <- 0
	while (round*Niter<maxBurnIn) { ## Burn-in
		round <- round + 1
		print(runId)
		print(Sys.time())
		print(paste("Drawing burn-in samples from",(round-1)*Niter+1,"to",round*Niter))
		print("===")
		temp <- sampleMPCCAcont(data=dataIn, model=temp$model, priors=priors, param=param, nIter=Niter)
	} ## end of burn-in

	modelBi = temp$model

	## Sampling after burn-in
	print(runId)
	print(Sys.time())
	print(paste("Drawing posterior samples from",round*Niter+1,"to",round*Niter+Niterfinal))
	print("===")
	samples <- sampleMPCCAcont(data=dataIn, model=modelBi, priors=priors, param=param, nIter=Niterfinal, printing=TRUE)
	## End of sampling after burn-in

	## END OF SAMPLING


	## Save the Gibbs samples into file
	save(list=ls(),file=paste(pathRun,"results.RData",sep="")) # Save the results.

	## Collect posterior samples

	# The lists are decomposed into matrices corresponding to each variable.
	maxSaved = 2000
# 	Nps = length(samples$storage) # number of posterior samples
	if (length(samples$storage)>maxSaved) {
		mod = floor(length(samples$storage)/maxSaved)
		Nps = maxSaved
	} else {
		mod = 1
		Nps = length(samples$storage)
	}
	posterior = list() # a list for the posterior samples

	# The bCCA part
	posterior$z = array(dim=c(param$zDim,N,Nps)) # latent variable of Xlat and Ylat
	posterior$mu = array(dim=c(nXlat+nYlat,Nps)) # latent variable-specific mean
	posterior$PsiX = array(dim=c(nXlat,nXlat,Nps))
	posterior$PsiY = array(dim=c(nYlat,nYlat,Nps))
	posterior$Ux = array(dim=c(nXlat,param$zDim,Nps))
	posterior$Uy = array(dim=c(nYlat,param$zDim,Nps))
	posterior$Wx = array(dim=c(nXlat,param$zDim,Nps))
	posterior$Wy = array(dim=c(nYlat,param$zDim,Nps))
	posterior$WxRSS = array(dim=c(nXlat,Nps))
	posterior$WyRSS = array(dim=c(nYlat,Nps))
	posterior$beta = array(dim=c(param$zDim,Nps))
	posterior$like = list()
	posterior$like$z = rep(NA,Nps)
	posterior$like$xLat = rep(NA,Nps) # If no dimensionality reduction, this is the data likelihood -3.11.10
	posterior$like$yLat = rep(NA,Nps)
	# The dimensionality reduction part
	if (param$clustering[1]) {
		posterior$Vx = array(dim=c(nX,nXlat,Nps)) # clusterings of the data X
		posterior$xlat = array(dim=c(nXlat,N,Nps)) # 'xlats'
		posterior$SigmaX = array(dim=c(nX,Nps))
		posterior$varsX = array(dim=c(nX,Nps)) # variable-specific variance for data X
		posterior$muX = array(dim=c(nX,Nps)) # variable-specific mean of data X
		posterior$like$X = rep(NA,Nps)
	}
	if (param$clustering[2]) {
		posterior$Vy = array(dim=c(nY,nYlat,Nps)) # clusterings of the data Y
		posterior$ylat = array(dim=c(nYlat,N,Nps)) # 'ylats'
		posterior$SigmaY = array(dim=c(nY,Nps))
		posterior$varsY = array(dim=c(nY,Nps)) # variable-specific variance for data Y
		posterior$muY = array(dim=c(nY,Nps)) # variable-specific mean of data Y
		posterior$like$Y = rep(NA,Nps)
	}

	# The MANOVA part
	A = max(data$covariatesList$a)
	B = max(data$covariatesList$b)
	C = max(data$covariatesList$c)
	D = max(data$covariatesList$d)
	posterior$eff = list()
	if (param$sampleEff$A)
		posterior$eff$A = array(dim=c(Nps,param$zDim,A))
	if (param$sampleEff$B)
		posterior$eff$B = array(dim=c(Nps,param$zDim,B))
	if (param$sampleEff$C)
		posterior$eff$C = array(dim=c(Nps,param$zDim,C))
	if (param$sampleEff$D)
		posterior$eff$D = array(dim=c(Nps,param$zDim,D))
	if (param$sampleEff$AB)
		posterior$eff$AB = array(dim=c(Nps,param$zDim,A,B))
	if (param$sampleEff$AC)
		posterior$eff$AC = array(dim=c(Nps,param$zDim,A,C))
	if (param$sampleEff$AD)
		posterior$eff$AD = array(dim=c(Nps,param$zDim,A,D))
	if (param$sampleEff$BC)
		posterior$eff$BC = array(dim=c(Nps,param$zDim,B,C))
	if (param$sampleEff$BD)
		posterior$eff$BD = array(dim=c(Nps,param$zDim,B,D))
	if (param$sampleEff$CD)
		posterior$eff$CD = array(dim=c(Nps,param$zDim,C,D))
	if (param$sampleEff$ABC)
		posterior$eff$ABC = array(dim=c(Nps,param$zDim,A,B,C))
	if (param$sampleEff$ABD)
		posterior$eff$ABD = array(dim=c(Nps,param$zDim,A,B,D))
	if (param$sampleEff$ACD)
		posterior$eff$ACD = array(dim=c(Nps,param$zDim,A,C,D))
	if (param$sampleEff$BCD)
		posterior$eff$BCD = array(dim=c(Nps,param$zDim,B,C,D))
	if (param$sampleEff$ABCD)
		posterior$eff$ABCD = array(dim=c(Nps,param$zDim,A,B,C,D))

	# Canonical correlations
	corBCCA = array(dim=c(param$zDim,Nps))
	corCCA = array(dim=c(min(nXlat,nYlat),Nps))
	tmp = list()
	# Also effects need to be saved, later.

	# Take posterior samples from the first chain only.
	for (i in 1:Nps) {
		j = mod*(i-1)+1
		# The bCCA part
		posterior$z[,,i] = samples$storage[[j]]$z
		posterior$mu[,i] = samples$storage[[j]]$mu
		posterior$PsiX[,,i] = samples$storage[[j]]$PsiX
		posterior$PsiY[,,i] = samples$storage[[j]]$PsiY
		posterior$Wx[,,i] = samples$storage[[j]]$W[1:nXlat,]
		posterior$Wy[,,i] = samples$storage[[j]]$W[(nXlat+(1:nYlat)),]
		posterior$WxRSS[,i] = rowSums((posterior$Wx[,,i,drop=F])^2)
		posterior$WyRSS[,i] = rowSums((posterior$Wy[,,i,drop=F])^2)
		posterior$beta[,i] = samples$storage[[j]]$beta
		#Feed function 'findRotation' with posterior samples 'j'.
# 		tmp = findRotation(posterior$xlat[,,i,drop=F],posterior$ylat[,,i,drop=F],samples$storage[[j]])
# 		corBCCA[,i] = tmp$cor
# 		posterior$Ux[,,i] = tmp$Ux
# 		posterior$Uy[,,i] = tmp$Uy
		
		# The dimensionality reduction part
		if (param$clustering[1]) {
			posterior$Vx[,,i] = samples$storage[[j]]$Vx
			posterior$xlat[,,i] = samples$storage[[j]]$xlat
			posterior$SigmaX[,i] = samples$storage[[j]]$SigmaX
			posterior$varsX[,i] = samples$storage[[j]]$varsX
			posterior$muX[,i] = samples$storage[[j]]$muX
			posterior$like$X[i] = samples$storage[[j]]$like$X
		}
		if (param$clustering[2]) {
			posterior$Vy[,,i] = samples$storage[[j]]$Vy
			posterior$ylat[,,i] = samples$storage[[j]]$ylat
			posterior$SigmaY[,i] = samples$storage[[j]]$SigmaY
			posterior$varsY[,i] = samples$storage[[j]]$varsY
			posterior$muY[,i] = samples$storage[[j]]$muY
			posterior$like$Y[i] = samples$storage[[j]]$like$Y
		}
		
		# The ANOVA part
		if (param$sampleEff$A) { # 1
			posterior$eff$A[i,,] = t(samples$storage[[j]]$effects[dataIn$design$ind$tab$A,])
		}
		if (param$sampleEff$B) { # 2
			posterior$eff$B[i,,] = t(samples$storage[[j]]$effects[dataIn$design$ind$tab$B,])
		}
		if (param$sampleEff$C) # 3
			posterior$eff$C[i,,] = t(samples$storage[[j]]$effects[dataIn$design$ind$tab$C,])
		if (param$sampleEff$D) # 4
			posterior$eff$D[i,,] = t(samples$storage[[j]]$effects[dataIn$design$ind$tab$D,])
		for (ai in 1:A) {
			if (param$sampleEff$AB) # 5
				posterior$eff$AB[i,,ai,] = t(samples$storage[[j]]$effects[dataIn$design$ind$tab$AB[ai,],])
			if (param$sampleEff$AC) # 6
				posterior$eff$AC[i,,ai,] = t(samples$storage[[j]]$effects[dataIn$design$ind$tab$AC[ai,],])
			if (param$sampleEff$AD) # 7
				posterior$eff$AD[i,,ai,] = t(samples$storage[[j]]$effects[dataIn$design$ind$tab$AD[ai,],])
			for (bi in 1:B) {
				if (param$sampleEff$ABC) # 11
					posterior$eff$ABC[i,,ai,bi,] = t(samples$storage[[j]]$effects[dataIn$design$ind$tab$ABC[ai,bi,],])
				if (param$sampleEff$ABD) # 12
					posterior$eff$ABD[i,,ai,bi,] = t(samples$storage[[j]]$effects[dataIn$design$ind$tab$ABD[ai,bi,],])
				if (ai==1) {
					if (param$sampleEff$BC) # 8
						posterior$eff$BC[i,,bi,] = t(samples$storage[[j]]$effects[dataIn$design$ind$tab$BC[bi,],])
					if (param$sampleEff$BD) # 9
						posterior$eff$BD[i,,bi,] = t(samples$storage[[j]]$effects[dataIn$design$ind$tab$BD[bi,],])
				}
				for (ci in 1:C) {
					if (param$sampleEff$ABCD) # 15
						posterior$eff$ABCD[i,,ai,bi,ci,] = t(samples$storage[[j]]$effects[dataIn$design$ind$tab$ABCD[ai,bi,ci,],])
					if (ai==1) {
						if (param$sampleEff$BCD) # 14
							posterior$eff$BCD[i,,bi,ci,] = t(samples$storage[[j]]$effects[dataIn$design$ind$tab$BCD[bi,ci,],])
					}
					if (bi==1) {
						if (param$sampleEff$ACD) # 13
							posterior$eff$ACD[i,,ai,ci,] = t(samples$storage[[j]]$effects[dataIn$design$ind$tab$ACD[ai,ci,],])
						if (ai==1) {
							if (param$sampleEff$CD) # 10
								posterior$eff$CD[i,,ci,] = t(samples$storage[[j]]$effects[dataIn$design$ind$tab$CD[ci,],])
						}
					}
				}
			}
		}
			
# 		if (param$zDim>1) {
# 			corCCA[,i] = cancor(t(posterior$xlat[,,j]),t(posterior$ylat[,,j]))$cor
# 		} else if (nXlat==1 & nYlat==1) {
# 			corCCA[,i] = cancor(as.matrix(posterior$xlat[,,j]),as.matrix(posterior$ylat[,,j]))$cor
# 		}
		# likelihoods
		posterior$like$z[i] = samples$storage[[j]]$like$z
		posterior$like$xLat[i] = samples$storage[[j]]$like$xlat
		posterior$like$yLat[i] = samples$storage[[j]]$like$ylat
	} ## End of collect posterior samples

	## For generated data, arrange the components of 'ylat' and 'xlat' into order of generation.

	if (param$clustering[1]) {
		Vxsum = apply(posterior$Vx,c(1,2),sum)
		Vx.avg = Vxsum/max(Vxsum) # set values to be from zero to one
		Vx.mode = (Vx.avg==apply(Vx.avg,1,max))*1
	}
	if (param$clustering[2]) {
		Vysum = apply(posterior$Vy,c(1,2),sum)
		Vy.avg = Vysum/max(Vysum) # set values to be from zero to one
		Vy.mode = (Vy.avg==apply(Vy.avg,1,max))*1
	}
	if (!is.null(data$setX$genPar$V)) { # For generated data the correct ordering of clusters is known.
		Vx.ind = compare_clusters(data$setX$genPar$V,Vx.avg)
	} else { # real data - no correct ordering
		Vx.ind = 1:nXlat
	}
	if (!is.null(data$setY$genPar$V)) { # For generated data the correct ordering of clusters is known.
		Vy.ind = compare_clusters(data$setY$genPar$V,Vy.avg)
	} else { # real data - no correct ordering
		Vy.ind = 1:nYlat
	}

	## Printing and plotting
	if (doPlotting) {
		print("Plotting")
		plotSeries(x=cbind(posterior$like$z,posterior$like$xLat,posterior$like$yLat,posterior$like$X,posterior$like$Y), fname=paste(pathRun,"like.png",sep=""), ylab=c("z", "xLat", "yLat", "X", "Y")) # likelihoods
		if (param$clustering[1]) {
			if (!is.null(data$setX$genPar$V)) {
				plotSurface(Vx.avg[,Vx.ind],file=paste(pathRun,"Vx",sep=""),Xtrue=data$setX$genPar$V) # Vx
			} else { # If real data, write metabolite names under each cluster.
				write_clust(NULL,rownames(dataIn$X),Vx.mode,Vx.avg,paste(pathRun,"Vx.txt",sep=""))
				plotSurface(Vx.avg[,Vx.ind],file=paste(pathRun,"Vx",sep="")) # Vx
			}
			plotMatrixDistribution(posterior$xlat,file=paste(pathRun,"xlat",sep="")) # xlat
			plot_var(data=dataIn$X, mu0=priors$muX0, var0=priors$varsX0, vars=posterior$varsX, sigmas=posterior$SigmaX, mus=posterior$muX, path=paste(pathRun,"varsX",sep=""), sorted=TRUE, varsGen=NULL)
			plot_cor(x=dataIn$X, covA=data$covariatesList$a, covB=data$covariatesList$b, covC=data$covariatesList$c, VV=posterior$Vx, Vmode=Vx.mode, path=paste(pathRun,"X-",sep="")) # Correlation series
		} else {
			write(x=rownames(dataIn$X), file=paste(pathRun,"Vx.txt",sep=""))
		}
		if (param$clustering[2]) {
			if (!is.null(data$setY$genPar$V)) {
				plotSurface(Vy.avg[,Vy.ind],file=paste(pathRun,"Vy",sep=""),Xtrue=data$setY$genPar$V) # Vy
			} else { # If real data, write metabolite names under each cluster.
				write_clust(NULL,rownames(dataIn$Y),Vy.mode,Vy.avg,paste(pathRun,"Vy.txt",sep=""))
				plotSurface(Vy.avg[,Vy.ind],file=paste(pathRun,"Vy",sep="")) # Vy
			}
			plotMatrixDistribution(posterior$ylat,file=paste(pathRun,"ylat",sep="")) # ylat
			plot_var(data=dataIn$Y, mu0=priors$muY0, var0=priors$varsY0, vars=posterior$varsY, sigmas=posterior$SigmaY, mus=posterior$muY, path=paste(pathRun,"varsY",sep=""), sorted=TRUE, varsGen=NULL)
			plot_cor(x=dataIn$Y, covA=data$covariatesList$a, covB=data$covariatesList$b, covC=data$covariatesList$c, VV=posterior$Vy, Vmode=Vy.mode, path=paste(pathRun,"Y-",sep="")) # Correlation series
		} else {
			write(x=rownames(dataIn$Y), file=paste(pathRun,"Vy.txt",sep=""))
		}
		textFile = paste(pathRun,"cors.txt",sep="")
		write(paste("\n",N," common samples.\nDataset X: ",nX," variables\nDataset Y: ",nY," variables",sep=""),file=textFile,append=T)
	# 		write(paste("\nSampler converged in ",rnd*Niter," iterations.",sep=""),file=textFile,append=T)
		write(paste("\nNumber of posterior samples: ",Nps,sep=""),file=textFile,append=T)
		# Plot the posterior distributions of variables
		plotVectorDistributionWithLabels(posterior$mu,file=paste(pathRun,"mu",sep="")) # Wx
		plotMatrixDistribution(X=posterior$PsiX[Vx.ind,Vx.ind,], file=paste(pathRun,"PsiX",sep="")) # PsiX
		plotMatrixDistribution(X=posterior$PsiY[Vy.ind,Vy.ind,], file=paste(pathRun,"PsiY",sep="")) # PsiY
	# 		plotMatrixDistribution(posterior$Wx[Vx.ind,,,drop=F],file=paste(pathRun,"Wx",sep="")) # Wx
	# 		plotMatrixDistribution(posterior$Wy[Vy.ind,,,drop=F],file=paste(pathRun,"Wy",sep="")) # Wy
		plotMatrixDistribution(X=posterior$Wx[Vx.ind,,,drop=F], file=paste(pathRun,"Wx",sep=""), labels=TRUE, Xtrue=data$setX$genPar$W) # Wx
		plotMatrixDistribution(X=posterior$Wy[Vy.ind,,,drop=F], file=paste(pathRun,"Wy",sep=""), labels=TRUE, Xtrue=data$setY$genPar$W) # Wy
		plotVectorDistributionWithLabels(posterior$WxRSS[Vx.ind,],file=paste(pathRun,"WxRSS",sep=""),box=FALSE) # Wx row square sums
		plotVectorDistributionWithLabels(posterior$WyRSS[Vy.ind,],file=paste(pathRun,"WyRSS",sep=""),box=FALSE) # Wy row square sums
		if (sum(is.na(posterior$Ux))==0 & sum(is.nan(posterior$Ux))==0)
			plotMatrixDistribution(posterior$Ux[Vx.ind,,,drop=F],file=paste(pathRun,"Ux",sep="")) # Ux
		if (sum(is.na(posterior$Uy))==0 & sum(is.nan(posterior$Uy))==0)
			plotMatrixDistribution(posterior$Uy[Vy.ind,,,drop=F],file=paste(pathRun,"Uy",sep="")) # Uy
		if (!is.null(data$z)) {
			plotMatrixDistribution(X=posterior$z, file=paste(pathRun,"z",sep=""), Xtrue=t(data$z)) # z
		} else {
			plotMatrixDistribution(X=posterior$z, file=paste(pathRun,"z",sep="")) # z
		}
		plotVectorDistributionWithLabels(posterior$beta,file=paste(pathRun,"beta",sep=""),box=FALSE) # beta
		# MANOVA part
		if (param$sampleEff$A)
			timedevelop(mu=posterior$eff$A, path=pathRun, fname="eff-A", effLab="a", effLev=NULL)
		if (param$sampleEff$B)
			timedevelop(mu=posterior$eff$B, path=pathRun, fname="eff-B", effLab="b", effLev=NULL)
		if (param$sampleEff$C)
			timedevelop(mu=posterior$eff$C, path=pathRun, fname="eff-C", effLab="c", effLev=NULL)
		if (param$sampleEff$D)
			timedevelop(mu=posterior$eff$D, path=pathRun, fname="eff-D", effLab="d", effLev=NULL)
		if (param$sampleEff$AB) {
			for (ai in 1:A) {
				timedevelop(mu=posterior$eff$AB[,,ai,], path=pathRun, fname=paste("eff-AB-a",ai,sep=""), effLab=c("a","b"), effLev=ai)
			}
		}
		if (param$sampleEff$AC) {
			for (ai in 1:A) {
				timedevelop(mu=posterior$eff$AC[,,ai,], path=pathRun, fname=paste("eff-AC-a",ai,sep=""), effLab=c("a","c"), effLev=ai)
			}
		}
		if (param$sampleEff$AD) {
			for (ai in 1:A) {
				timedevelop(mu=posterior$eff$AD[,,ai,], path=pathRun, fname=paste("eff-AD-a",ai,sep=""), effLab=c("a","d"), effLev=ai)
			}
		}
		if (param$sampleEff$BC) {
			for (bi in 1:B) {
				timedevelop(mu=posterior$eff$BC[,,bi,], path=pathRun, fname=paste("eff-BC-b",bi,sep=""), effLab=c("b","c"), effLev=bi)
			}
		}
		if (param$sampleEff$BD) {
			for (bi in 1:B) {
				timedevelop(mu=posterior$eff$BD[,,bi,], path=pathRun, fname=paste("eff-BD-b",bi,sep=""), effLab=c("b","d"), effLev=bi)
			}
		}
		if (param$sampleEff$CD) {
			for (ci in 1:C) {
				timedevelop(mu=posterior$eff$BD[,,ci,], path=pathRun, fname=paste("eff-CD-c",ci,sep=""), effLab=c("c","d"), effLev=ci)
			}
		}
		if (param$sampleEff$ABC) {
			for (ai in 1:A) {
				for (bi in 1:B) {
					timedevelop(mu=posterior$eff$ABC[,,ai,bi,], path=pathRun, fname=paste("eff-ABC-a",ai,"b",bi,sep=""), effLab=c("a","b","c"), effLev=c(ai,bi))
				}
			}
		}
		if (param$sampleEff$ABD) {
			for (ai in 1:A) {
				for (bi in 1:B) {
					timedevelop(mu=posterior$eff$ABD[,,ai,bi,], path=pathRun, fname=paste("eff-ABD-a",ai,"b",bi,sep=""), effLab=c("a","b","d"), effLev=c(ai,bi))
				}
			}
		}
		if (param$sampleEff$ACD) {
			for (ai in 1:A) {
				for (ci in 1:C) {
					timedevelop(mu=posterior$eff$ACD[,,ai,ci,], path=pathRun, fname=paste("eff-ACD-a",ai,"c",ci,sep=""), effLab=c("a","c","d"), effLev=c(ai,ci))
				}
			}
		}
		if (param$sampleEff$BCD) {
			for (bi in 1:B) {
				for (ci in 1:C) {
					timedevelop(mu=posterior$eff$BCD[,,bi,ci,], path=pathRun, fname=paste("eff-BCD-b",bi,"c",ci,sep=""), effLab=c("b","c","d"), effLev=c(bi,ci))
				}
			}
		}
		if (param$sampleEff$ABCD) {
			for (ai in 1:A) {
				for (bi in 1:B) {
					for (ci in 1:C) {
						timedevelop(mu=posterior$eff$ABCD[,,ai,bi,ci,], path=pathRun, fname=paste("eff-ABCD-a",ai,"b",bi,"c",ci,sep=""), effLab=c("a","b","c","d"), effLev=c(ai,bi,ci)) # Bug fixed 31.1.11.
					}
				}
			}
		}
# 		pathSeries = paste(pathRun,"series/", sep="")
# 		dir.create(pathSeries) # Create a directory for series plottings.
# 		plotSeries(x=posterior$like$z, fname=paste(pathSeries,"likeZ.png",sep=""))
# 		plotSeries(x=posterior$like$X, fname=paste(pathSeries,"likeX.png",sep=""))
# 		plotSeries(x=posterior$like$Y, fname=paste(pathSeries,"likeY.png",sep=""))
# 		plotSeries(x=posterior$like$xLat, fname=paste(pathSeries,"likeXlat.png",sep=""))
# 		plotSeries(x=posterior$like$yLat, fname=paste(pathSeries,"likeYlat.png",sep=""))
# 		plotSeries(x=posterior$like$z+posterior$like$X+posterior$like$Y+posterior$like$xLat+posterior$like$yLat, fname=paste(pathSeries,"likeTot.png",sep=""))
	} else {
		print("No plotting performed!")
	}


	## End the script
	print(runId)
	print(paste("Started",time.start))
	print(paste("Complete at",Sys.time()))

# 	stopp

	return(posterior)

}
