package jmprojection;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

/**
 * Used to represent list of adjacent FibNodes.
 * It is an array of ArrayLists and each ArrayList contains NodeKeyPairs
 * representing FibNodes that are adjacent to each FibNode.
 * <br><br>
 * This class should be modified to use any <i>Node</i> fulfilling requirements
 * of some not-yet-defined <i>Node interface</i>. NodeKeyPair shouldn't be used
 * directly but instead we should use some wrapper which can use any <i>Node</i>,
 * not just FibNode.
 * <br><br>
 * For minimum spannig tree -algorithm add(AdjacencyList, AdjacencyList) should
 * be finished.
 * @author Jarkko Miettinen
 */
public class AdjacencyList
{

	public static final int KEEP_CLONES = 1;
	public static final int KEEP_SMALLER_KEY = 2;
	public static final int KEEP_LARGER_KEY = 4;

	private List<List<NodeKeyPair>> innerList;

	private Map<FibNode, List<NodeKeyPair>> nodeMapping;

	private FibNode[] nodes;

	public AdjacencyList(FibNode[] nodes)
	{
		int len = nodes.length;
		this.innerList = new ArrayList<List<NodeKeyPair>>(len);
		this.nodes = new FibNode[len];
		nodeMapping = new HashMap<FibNode, List<NodeKeyPair>>();
		for (int i = 0; i < len; i++)
		{
			innerList.add(new ArrayList<NodeKeyPair>());
			nodeMapping.put(nodes[i], innerList.get(i));
			this.nodes[i] = nodes[i];
		}
	}

	public int size()
	{
		return this.nodes.length;
	}

	/**
	 * Returns FibNode-array containing all the nodes in this adjacencylist
	 * @return FibNode-array
	 */
	public FibNode[] getNodes()
	{
		return nodes;
	}

	/**
	 * Adds connection from given node to another node with given weight
	 * @param from <code>FibNode</code> where the edge starts
	 * @param to <code>FibNode</code> where the edge end
	 * @param weight weight of the edge
	 */
	public void addConnection(FibNode from, FibNode to, double weight)
	{
		addConnection(from, new NodeKeyPair(to, weight));
	}

	public void addBiConnection(FibNode oneEnd, FibNode anotherEnd, double weight)
	{
		addConnection(oneEnd, anotherEnd, weight);
		addConnection(anotherEnd, oneEnd, weight);
	}

	/**
	 * Adds predefined connection to given FibNode
	 * @param from
	 * @param connection
	 */
	public void addConnection(FibNode from, NodeKeyPair connection)
	{
		List<NodeKeyPair> ar = nodeMapping.get(from);
		ar.add(connection);
	}

	/**
	 * Removes all connections between two nodes.
	 * @param from
	 * @param to
	 * @return
	 */
	public boolean removeConnections(FibNode from, FibNode to)
	{
		ArrayList ar = (ArrayList) nodeMapping.get(from);
		boolean found = false;
		for (int i = 0, size = ar.size(); i < size; i++)
		{
			NodeKeyPair nkp = (NodeKeyPair) ar.get(i);
			if (nkp.getNode().equals(to))
			{
				found = true;
				ar.remove(i);
			}
		}
		return found;
	}

	/**
	 * Removes a specific connection
	 * @param from
	 * @param connection
	 * @return
	 */
	public boolean removeConnection(FibNode from, NodeKeyPair connection)
	{
		ArrayList ar = (ArrayList) nodeMapping.get(from);
		return ar.remove(connection);
	}

	/**
	 * Returns iterator edges from given node.
	 * @param node
	 * @return Iterator iterating over all NodeKeyPairs, which would be edges,
	 * starting from given node.
	 */
	public Iterator<NodeKeyPair> getIterator(FibNode node)
	{
		List<NodeKeyPair> ar = nodeMapping.get(node);
		return ar.iterator();
	}

	public String toString()
	{
		/* We shouldn't use AdjacencyList to represent dense graphs so log(V)*V
		 * is good estimate for the number of nodes and every node takes 2 +
		 * log[10](node.getId()) characters.
		 */
		int len = nodes.length;
		int size = (int) Math.ceil(Math.log(len)/Math.log(10))*len + 1;
		StringBuffer sb = new StringBuffer(size);
		for (int i = 0, n = nodes.length; i < n; i++)
		{
			FibNode currentNode = nodes[i];
			sb.append("[" + currentNode.getId() + "]: ");
			sb.append(innerList.get(i));
			sb.append('\n');
		}
		return sb.toString();
	}
}
