package jmprojection;

/**
 * Innerclass FibNode implements a node in Fibonacci-Heap.
 */
public class FibNode{
	private FibNode left;
	private FibNode child;
	private FibNode parent;
	private FibNode right;
	private Object objectKey;
	private boolean mark;
	private double key;
	private int id;
	private int degree;

	/**
	 * Creates a new FibNode with key of 0.0 and given id
	 * @param id identifier of this FibNode, used by ShortestPathDijkstra
	 */
	public FibNode(int id){
		this(id, 0.0);
	}

	public FibNode(int id, double key){
		this(id, key, null);
	}

	/**
	 * Creates a new FibNode with given id and given key if the key is >= 0.
	 * @param id id identifier of this FibNode, used by ShortestPathDijkstra
	 * @param key the key associated with this FibNode
	 * @param objectKey some object possibly associated with this FibNode
	 */
	public FibNode(int id, double key, Object objectKey){
		this.degree = 0;
		this.parent = null;
		this.child = null;
		this.left = this;
		this.right = this;
		if (key >= 0.0){
			this.key = key;
		}
		else{
			key = 0.0;
		}
		this.id = id;
		this.objectKey = objectKey;
	}

	/**
	 * Used to retrieve the Object assosiated with this FibNode
	 * @return
	 */
	public Object getObjectKey(){
		return this.objectKey;
	}

	public void setObjectKey(Object o){
		this.objectKey = o;
	}

	/**
	 * Sets this FibNodes parent and child references to null and degree to
	 * zero. Left and right FibNode are set to reference this.
	 */
	public void reset(){
		this.degree = 0;
		this.parent = this.child = null;
		this.left = this.right = this;
		this.key = 0.0;
		this.objectKey = null;
	}

	/**
	 * Returns parents.
	 * @return parent <code>FibNode</code> of this node.
	 */
	public FibNode getParent(){
		return this.parent;
	}

	/**
	 * Sets new parent.
	 * @param newParent <code>FibNode</code> set to be this
	 * <code>FibNode's</code> new parent.
	 */
	public void setParent(FibNode newParent){
		this.parent = newParent;
	}

	/**
	 * Returns <code>FibNode</code> that is this <code>FibNode's</code>
	 * child. Returned <code>FibNode</code> is some <code>FibNode</code>
	 * on the doubly-linked list of childs.
	 * @return child of this FibNode
	 */
	public FibNode getChild(){
		return this.child;
	}

	/**
	 *  Sets <code>FibNode</code> pointing to this <code>FibNode's</code>
	 *  child to newChild but doesn't modify the doubly-linked child-list
	 *  in any way. This modification is done in FibHeap.
	 * @param newChild
	 */
	public void setChild(FibNode newChild){
		this.child = newChild;
	}

	/**
	 * Returns FibNode left from this node.
	 * @return <code>FibNode</code> on the left side of this
	 * <code>FibNode</code> on the doubly-linked childList / rootList.
	 */
	public FibNode getLeft(){
		return this.left;
	}

	/**
	 * Sets this <code>FibNode's</code> left <code>FibNode</code>  to
	 * newLeft but doesn't alter the doubly-linked list in any way. This
	 * alteration is done in FibHeap.
	 * @param newLeft
	 */
	public void setLeft(FibNode newLeft){
		this.left = newLeft;
	}

	/**
	 * Returns FibNode right from this node.
	 * @return <code>FibNode</code> on the right side of this
	 * <code>FibNode</code> on the doubly-linked childList / rootList.
	 */
	public FibNode getRight(){
		return this.right;
	}

	/**
	 * Sets this <code>FibNode's</code> right <code>FibNode</code>  to
	 * newRight but doesn't alter the doubly-linked list in any way. This
	 * alteration is done in FibHeap.
	 * @param newLeft
	 */
	public void setRight(FibNode newRight){
		this.right = newRight;
	}

	/**
	 * Returns boolean whether or not this <code>FibNode</code> has lost a
	 * child since the last time this was made the child of another node.
	 * Used in FibHeap's cascadingCut() method.
	 * @return has this FibNode lost a child since last time it was made the
	 * child of another node.
	 */
	public boolean isMarked(){
		return this.mark;
	}

	/**
	 * Used to set whether or not this <code>FibNode</code> has lost a child
	 * since the last time this was made the child of another node. Used in
	 * FibHeap's link, cut and cascadingCut -methods.
	 * @param mark sets mark to given value.
	 */
	public void setMark(boolean mark){
		this.mark = mark;
	}

	/**
	 * Returns number of children on this <code>FibNode's</code> childList
	 * this only exist as a doubly-linked list of references among the
	 * children.
	 * @return number of children.
	 */
	public int getDegree(){
		return this.degree;
	}

	/**
	 * Increases this <code>FibNode's</code> number of children by one.
	 */
	public void increaseDegree(){
		this.degree++;
	}

	/**
	 * Decreases this <code>FibNode's</code> number of children by one.
	 */
	public void decreaseDegree(){
		if (--this.degree < 0) this.degree = 0;
	}

	/**
	 * Returns a <code>double</code> key associated with this
	 * <code>FibNode</code>. More modular approach would be to associate a
	 * Object as the key but performance would suffer in Dijkstra because of
	 * castings of Object to Double and retrieving it's double value I
	 * think.
	 * @return this FibNode's key.
	 */
	public double getKey(){
		return this.key;
	}

	/**
	 * Sets new <code>double</code> key.
	 * @param newKey
	 */
	public void setKey(double newKey){
		this.key = newKey;
	}

	/**
	 * Used by ShortestPathDijkstra to associate this <code>FibNode</code> with certain
	 * position in array. A more modular approach would be to use hashmap /
	 * -table but this is chosen because of the performance. Although I
	 * think it doesn't make any noticeable difference.
	 * @return the id.
	 */
	public int getId(){
		return this.id;
	}

	public String toString(){
		StringBuffer sb = new StringBuffer("id: " + this.id);
		if (this.parent != null){
			sb = sb.append(" parent: " + parent.getId());
		}
		if (this.child != null){
			sb = sb.append(" child: " + child.getId());
		}
		sb = sb.append(" left: " + this.left.getId() + " right: " +
				this.right.getId());
		return sb.toString();
	}
}