package jmprojection;

import java.util.Random;

/**
 * Maps a matrix full of data vectors to two dimension by Sammon's Non-linear
 * mapping.<br>
 * Created on 1.7.2005
 * @author Jarkko Miettinen
 *
 */
public class Sammons extends AbstractMapping implements Curvilinear{

	private double[][] lSpaceMatrix;
	private double[][] lSpaceDistances;
	private double[][] pSpaceMatrix;
	private double[][] pSpaceDistances;
	private int pDim;
	//Magic factor used in Sammons mapping, empirically
	//determined to be ~ 0.3 or 0.4
	private static final double MF = 0.4;
	//Constant used in calculating the error
	private double c;
	public final static String NAME_OF_MAPPING = "Sammons Mapping";
	public final static int DEFAULT_PROJECTION_DIMENSION = 2;

	public Sammons(double[][] data, int projectionDimensions)
	{
		super(data);
		if (projectionDimensions <= 1){ pDim = DEFAULT_PROJECTION_DIMENSION;}
		else pDim = projectionDimensions;
		//We presume n x k matrix
		lSpaceMatrix = new double[data.length][data[0].length];
		java.util.Random r = new java.util.Random(System.currentTimeMillis());
		pSpaceMatrix = new double[data.length][pDim];
		lSpaceDistances = new double[data.length][data.length];
		//Copy state-matrix to our own LSpaceMatrix
		//In addition, initialize twoSpaceMatrix with random numbers
		//TODO change this to a method mentioned in Sammons articles
		//paragraph on first page.

		double maxDistance = 0.0;
		for (int i = 0; i < data.length;i++){
			double[] v1 = data[i];
			for (int j = i+1; j < data.length;j++){
				double[] v2 = data[j];
				double dist = dist(v1,v2);
				if (dist > maxDistance){ maxDistance = dist;}
			}
		}

		for (int i = 0; i < pSpaceMatrix.length; i++)
		{
			for (int j = 0; j < pDim; j++)
			{
				pSpaceMatrix[i][j] = maxDistance * r.nextDouble();
			}
		}

		for (int i = 0; i < data.length; i++){
			System.arraycopy(data[i], 0, lSpaceMatrix[i], 0, data[i].length);
		}

		//Calculate c for error term
		double d;
		for (int i = 0; i < lSpaceMatrix.length - 1; i++){
			for (int j = i+1; j < lSpaceMatrix.length; j++){
				d = dist(lSpaceMatrix[i], lSpaceMatrix[j]);
				lSpaceDistances[i][j] = lSpaceDistances[j][i] = d;
				c += d;
			}
		}
		pSpaceDistances = new double[lSpaceMatrix.length][lSpaceMatrix.length];
	}

	/**
	 * The constructor first creates a new matrix where it copies the data
	 * given in constructors parameters. Next the places of the vectors
	 * in 2D-space are initialized by generating some random numbers for each
	 * (x,y)-pair in It then initializes the lSpaceDistances matrix filled with
	 * distances (lSpaceDistances[i][j] means distance from i to j or j to i.
	 * This could be implemented in a linear list, but i am not sure whether one
	 * should save some memory or computation speed that might be lost when
	 * searching for the right index in linear distance-array. The constant c
	 * used in error-calculations is calculated at the same time.
	 */
	public Sammons(double[][] data) {
		this(data, DEFAULT_PROJECTION_DIMENSION);
	}

	private double getLSpaceDistance(int firstPoint, int secondPoint){
		return lSpaceDistances[firstPoint][secondPoint];
	}

	/**
	 * If this is too slow with large amount of points, consider switching from
	 * precalculating distances to in situ distance calculation for client jvm
	 * may be a bit slow with array access.
	 * @param from point where we calculate distance from
	 * @param to point to which we calculate distance to
	 * @return Euclidean distance between two points in two dimensions
	 * @throws ArrayIndexOutOfBoundsException if points are out of matrix
	 * indices. Should never happen, of course, but indices are never checked.
	 */
	private double getPSpaceDistance(int from, int to){
		return pSpaceDistances[from][to];
	}

	private void calculatePSpaceDistances(){
		for (int i = 0; i < pSpaceMatrix.length;i++){
			pSpaceDistances[i][i] = 0.0;
			for (int j = i+1; j < pSpaceMatrix.length;j++){
				double diff = 0.0;
				for (int k = 0; k < pDim; k++)
				{
					double temp = pSpaceMatrix[i][k] - pSpaceMatrix[j][k];
					diff += temp*temp;
				}
				diff = Math.sqrt(diff);
				pSpaceDistances[i][j] = pSpaceDistances[j][i] = diff;
			}
		}
	}

	/**
	 * When we calculate first and second partial derivative at the same time,
	 * we can use the common part of (d*[p,j] - d[p,j])/(d[p,j]*d*[p,j])
	 * In addition, 1 + (d*[p,j] - d[p,j])/d[p,j] simplifies to d*[p,j]/d[p,j]
	 * @param p
	 * @param q
	 */
	private double gradient(int p, int q){
		//distance in original space
		double dOrig;
		//distance in mapped space
		double dMap;
		//y[p,q] - y[j,q]
		double diff;
		//First partial derivative
		double pder = 0.0;
		//Secord partial derivative
		double pder2 = 0.0;
		double temp;

		for (int j = 0; j < lSpaceMatrix.length;j++){
			if (j != p){
				dOrig = getLSpaceDistance(p, j);
				dMap = getPSpaceDistance(p, j);
				diff = pSpaceMatrix[p][q] - pSpaceMatrix[j][q];
				if (dOrig == 0.0) dOrig = 1.0;
				if (dMap == 0.0) dMap = 1.0;
				temp = (dOrig - dMap)/(dOrig*dMap);
				pder += temp*diff;
				pder2 += temp - diff*diff/(dMap*dMap*dMap);
			}
		}
		return ((-pder)/Math.abs(pder2));
	}

	/* (non-Javadoc)
	 * @see megNet.DistanceMapping#getState()
	 */
	public double[][] getState() {
		int n = pSpaceMatrix.length;
		double returnMatrix[][] = new double[pDim][n];
		for (int i = 0; i < n; i++){
			for (int j = 0; j < pDim;j++){
				returnMatrix[j][i] = pSpaceMatrix[i][j];
			}
		}
		return returnMatrix;
	}

	/**
	 * @see DistanceMapping#mappingError()
	 * mapping error = (1/c) * sum((dOrig[i][j]-dMap[i][j])/dOrig[i][j])
	 * through all i<j to N
	 */
	public double mappingError() {
		double error = 0;
		double temp;
		double dOrig;

		for (int i = 0; i < lSpaceMatrix.length;i++){
			for (int j = i + 1; j < lSpaceMatrix.length; j++){
				dOrig = getLSpaceDistance(i,j);
				temp = dOrig - getPSpaceDistance(i,j);
				temp *= temp;
				temp /= dOrig;
				error += temp;
			}
		}
		return error/c;
	}

	/**
	 * @see DistanceMapping#iterate()
	 * Calculates
	 */
	public void iterate() {
		double newPSpaceMatrix[][] = new double[pSpaceMatrix.length][pSpaceMatrix[0].length];
		calculatePSpaceDistances();
		int permArray[] = genRandPermutation(pSpaceMatrix.length);
		for (int p = 0; p < pSpaceMatrix.length;p++){
			int i = permArray[p];
			for (int q = 0; q < pSpaceMatrix[i].length;q++){
				newPSpaceMatrix[i][q] = pSpaceMatrix[i][q] -
				(MF * gradient(i, q));
			}
		}
		pSpaceMatrix = newPSpaceMatrix;
		iterations++;
	}

	/**
	 * @see DistanceMapping#getName()
	 */
	public String getName() {
		return Sammons.NAME_OF_MAPPING;
	}

	public double[][] getDyDxError() {
		int n = this.lSpaceMatrix.length;
		int arrayLength = n*(n+1)/2;
		double[][] dxdyArray = new double[arrayLength][2];
		n = 0;
		for (int i = 0; i < this.lSpaceMatrix.length; i++){
			for (int j = i + 1; j < this.lSpaceMatrix.length;j++){
				double dx = getLSpaceDistance(i,j);
				double dy = getPSpaceDistance(i,j);
				dxdyArray[n][1] = dx;
				dxdyArray[n][0] = dy;
				n++;
			}
		}
		return dxdyArray;
	}

	/**
	 * @return Sammon's stress which happens to be the same as this mapping's
	 * error.
	 */
	public double sammonsStress() {
		return mappingError();
	}

	/* (non-Javadoc)
	 * @see AbstractMapping#getParameters()
	 */
	public String getParameters()
	{
		return "Iterations " + iterations();
	}
}
