package jmprojection;

import java.util.ArrayList;

/**
 *
 * Henceforth:<br>
 * N-space means the original dimension of the data<br>
 * P-space or projection space means the space where data is projected by this
 * algorithm, usually two dimensions.<br><br>
 *
 * Curvilinear Distance Analysis algorithm.<br>
 * Algorithm by John Aldo Lee, Amaury Lendasse, Nicolas Donckers,
 * Michel Verleysen<br>
 * in a article named "A Robust Nonlinear Projection Method"<br>
 * Published in ESANN'2000 proceedings, ISBN 2-930307-00-5, pp. 13-20<br><br>
 *
 * In the algorithm we first quantizes the data-vectors in to centroids.
 * Centroids represent space-averaged vectors in N-space.<br><br>
 *
 * We then calculate  distances in N-space between said centroid so that
 * distance from centroid A  to centroid B goes graph made out of centroids
 * connecting to centroids near them.<br>
 * The distance between two centroids in N-space is therefore usually
 * longer than just basic <I>Euclidean distance</I>. This distance is called
 * <I>curvilinear distance</I>. <br><br>
 *
 * We then project those centroids to P-space and try to keep Euclidean
 * distances in P-space near curvilinear distances in N-space. This distance
 * preservation is biased towards small distances by monotonically decreasing
 * neighbourhood factor so that small distances come more and more significant
 * and longer distances become less and less significant.<br>
 * This step is iterated for some more or less arbitrary times.<br><br>
 *
 * After centroids have been projected in to P-space, we project the actual
 * data-vectors and try to preserve their distances to centroids in the same
 * way we tried to preserve inter-centroid distances in last step.<br>
 * After we've iterated this step for some more or less arbitrary times we're
 * done.<br><br>
 *
 * @author Jarkko Miettinen<br>
 */
public class CDA extends AbstractMapping implements Curvilinear
{
	public static final double DEFAULT_LAMBDA = 0.95;
	public static final double DEFAULT_ALPHA = 0.5;
	public static final double DEFAULT_MAX_LOSS = 0.10;

	// default number of iterations in every iterative process:
	// neural water, centroid and data-vector projection.
	public static final int DEFAULT_TRAINING_LENGTH = 100;

	// default size of k in k-nearest neighbourhood
	public static final int DEFAULT_NEIGHBOURHOOD_SIZE = 5;
	public static final int DEFAULT_NUMBER_OF_PROJECTION_DIMENSIONS = 2;
	public final static String NAME_OF_MAPPING = "Curvilinear Distance Analysis";

	private double curvidistances[][];
	private double preDistances[][];
	private double nSpaceVect[][];
	private double nSpaceCent[][];
	private double pSpaceVect[][];
	private double pSpaceCent[][];

	private double maxLoss;
	private double maxMutualDist;
	private double startAlpha;
	private double startLambda;
	private double minLambda;
	private double minAlpha = 0.02;
	private double magicCoeff = 3.5;

	private int tlen;
	private int oDim;
	private int pDim;
	private int nHoodSize;
	private int iterations;

	private AdjacencyList neighList;
	private boolean initialized = false;

	/**
	 * @param data
	 */
	public CDA(double[][] data)
	{
		this(data, DEFAULT_ALPHA, DEFAULT_LAMBDA, DEFAULT_MAX_LOSS,
				DEFAULT_TRAINING_LENGTH, DEFAULT_NEIGHBOURHOOD_SIZE,
				DEFAULT_NUMBER_OF_PROJECTION_DIMENSIONS);
	}

	/**
	 * @param data N x D matrix, where N is number of points and D the
	 * dimensionality of those points
	 * @param startingAlpha starting value of learning factor [0,1]
	 * @param startingLambda starting value of neighbourhoodfactor [0,1]
	 * @param maximumLoss maximum allowed loss of accuracy in vector quantization
	 * @param trainingLength number of iteration-steps this algorithm is going
	 * to take
	 * @param neighbourhoodSize number of neighbours to create for each centroid
	 * @param projDim number of dimensions where to project the data
	 */
	public CDA(double[][] data, double alpha, double lambda,
			double maximumLoss, int trainingLength, int neighourhoodSize,
			int projDim)
	{
		super(data);
		if (alpha <= 0 || alpha > 1.0)
		{
			this.startAlpha = DEFAULT_ALPHA;
		}
		else
		{
			this.startAlpha = alpha;
		}
		if (lambda <= 0 || lambda > 1.0)
		{
			this.startLambda = DEFAULT_LAMBDA;
		}
		else
		{
			this.startLambda = lambda;
		}
		if (maximumLoss < 0 || maximumLoss > 1.0)
		{
			this.maxLoss = DEFAULT_MAX_LOSS;
		}
		else
		{
			this.maxLoss = maximumLoss;
		}
		if (trainingLength <= 0)
		{
			this.tlen = DEFAULT_TRAINING_LENGTH;
		}
		else
		{
			this.tlen = trainingLength;
		}
		if (neighourhoodSize >= data.length || neighourhoodSize <= 0)
		{
			this.nHoodSize = Math.min(DEFAULT_NEIGHBOURHOOD_SIZE,
					data.length - 1);
		}
		else
		{
			this.nHoodSize = neighourhoodSize;
		}
		oDim = data[0].length;
		if (projDim >= oDim)
		{
			pDim = Math.min(DEFAULT_NEIGHBOURHOOD_SIZE, oDim);
		}
		else
		{
			pDim = projDim;
		}
		nSpaceVect = new double[data.length][oDim];
		for (int i = 0; i < data.length; i++)
		{
			System.arraycopy(data[i], 0, nSpaceVect[i], 0, data[i].length);
		}
	}

	/*
	 * (non-Javadoc)
	 *
	 * @see AbstractMapping#getState()
	 */
	@Override
	public double[][] getState()
	{
		double state[][];
		state = new double[pDim][pSpaceVect.length];
		for (int i = 0; i < pDim; i++)
		{
			for (int j = 0; j < pSpaceVect.length; j++)
			{
				state[i][j] = pSpaceVect[j][i];
			}
		}
		return state;
	}

	/**
	 * Calculates and returns mapping error after current number of iterations.
	 * <pre>
	 * Error = 0.5 * Sum((D(i,j) - d(i,j))2 * F(d(i,j)), i,j = 1..N, i != j)
	 * where
	 * D(i,j) = curvilinear distance between points i and j in N-space
	 * d(i,j) = Euclidean distance between points i and j in projection space
	 * F(x)   = Neighbourhood function, in this case unitstep function
	 * </pre>
	 * @return mapping error, not the most informative error in the world
	 */
	@Override
	public double mappingError()
	{
		double nhRadius = magicCoeff * maxMutualDist * calcLambda(iterations, tlen);
		double error = 0.0;
		for (int i = 0; i < pSpaceVect.length; i++){
			for (int j = 0; j < pSpaceCent.length; j++){
				if (i != j){
					double dOrig = calcCurviDistance(i,j,nhRadius);
					double dMap = dist(pSpaceVect[i], pSpaceCent[j]);
					double temp =  dOrig - dMap;
					temp *= temp * nhFunction(dMap, nhRadius);
				}
			}
		}
		return 0.5*error;
	}

	/*
	 * (non-Javadoc)
	 *
	 * @see AbstractMapping#iterate()
	 */
	@Override
	public void iterate()
	{
		if (!initialized)
		{
			neuralWater((int) (30 + Math.ceil(Math.log(nSpaceVect.length))));
			preCalcDistances();
			if (projectionStatus.isCanceled()) return;
			curvidistances = createDistanceGraph(nHoodSize);
			if (projectionStatus.isCanceled()) return;
			minLambda = calcMinLambda();
			if (projectionStatus.isCanceled()) return;
			pSpaceCent = initializeToPSpace(nSpaceCent);
			if (projectionStatus.isCanceled()) return;
			for (int i = 0; i < tlen; i++)
			{
				if (projectionStatus.isCanceled()) return;
				projectCentroids(i);
				
			}
			if (nSpaceCent.length == nSpaceVect.length)
			{
				pSpaceVect = pSpaceCent; //No need to project points twice
				iterations = tlen;
			}
			else
			{
				if (projectionStatus.isCanceled()) return;
				pSpaceVect = initializeToPSpace(nSpaceVect);
			}
			initialized = true;
		}
		if (iterations < tlen)
		{
			projectVectors(iterations++);
		}
	}

	/**
	 * Calculates Sammon's stress by following formula:<br><pre>
	 * (1 / c) * Sum((d*(i,j) - d(i,j))^2/d*(i,j), i = 1..n-1, j = i + 1..n)
	 *
	 * where c = Sum(d*(i,j), i = 1..n-1, j = i + 1..n),
	 * d*(i,j) = Euclidean distance between points i and j in original space
	 * d(i,j)  = Euclidean distance between points i and j in projection space
	 * </pre>
	 * @return said Sammon's Stress
	 * @see AbstractMapping#sammonsStress()
	 */
	@Override
	public double sammonsStress()
	{
		double error = 0;
		double temp;
		double dOrig;
		double c = 0.0;

		for (int i = 0; i < nSpaceVect.length; i++)
		{
			for (int j = i + 1; j < nSpaceVect.length; j++)
			{
				dOrig = dist(nSpaceVect[i], nSpaceVect[j]);
				c += dOrig;
				temp = dist(pSpaceVect[i], pSpaceVect[j]);
				temp *= temp;
				temp /= dOrig;
				error += temp;
			}
		}
		return error / c;
	}

	/**
	 * @return String representation of given parameters.
	 */
	@Override
	public String getParameters()
	{
		StringBuilder sb = new StringBuilder("alpha_min: ");
		sb.append(minAlpha).append(" alpha_max: ").append(startAlpha);
		sb.append(" lambda_min: ").append(minLambda).append(" lambda_max: ");
		sb.append(startLambda).append(" training length: ").append(tlen);
		sb.append(" projection dimension: ").append(pDim);
		sb.append("neighbourhood size: ").append(nHoodSize);
		return sb.toString();
	}

	/**
	 * Return name of this mapping
	 */
	public String getName()
	{
		return NAME_OF_MAPPING;
	}

	public double[][] getCentroids()
	{
		double state[][];
		state = new double[pDim][pSpaceCent.length];
		for (int i = 0; i < pDim; i++)
		{
			for (int j = 0; j < pSpaceCent.length; j++)
			{
				state[i][j] = pSpaceCent[j][i];
			}
		}
		return state;
	}

	/**
	 * calculates distances between every centroid pair (c[i],c[j]) in original
	 * space and projection space and returns K x 2 array, where array[m][1] is
	 * the distance in original space ("dx") and array[m][0] the distance in
	 * projection space ("dy") for every m=0..K-1. K is n*(n+1)/2 where n is
	 * number of centroids.
	 * @return pair-wise distance-matrix
	 */
	public double[][] getDyDxError()
	{
		int centCount;
		int arrayLen;
		int counter;
		double dxdyError[][];

		centCount = nSpaceCent.length;
		arrayLen = centCount*(centCount + 1) / 2;
		dxdyError = new double[arrayLen][2];
		counter = 0;
		for (int i = 0; i < centCount; i++)
		{
			for (int j = i + 1; j < centCount; j++)
			{
				double dx;
				double dy;

				dx = curvidistances[i][j];
				dy = dist(pSpaceCent[i], pSpaceCent[j]);
				dxdyError[counter][0] = dx;
				dxdyError[counter][1] = dy;
				counter++;
			}
		}
		return dxdyError;
	}

	/**
	 * Calculates minimum lambda-value based on curvilinear distances in
	 * n-space. Minimu value is minDist / maxDist as suggested by Lee et al.
	 * @return minimum value for lambda
	 */
	private double calcMinLambda()
	{
		double min;
		double max;
		double temp;

		min = Double.MAX_VALUE;
		max = 0.0;

		for (int i = 0; i < curvidistances.length; i++)
		{
			for (int j = 0; j < curvidistances[i].length; j++)
			{
				if (i == j)
				{
					continue;
				}
				temp = curvidistances[i][j];
				if (temp < min)
				{
					min = temp;
				}
				if (temp > max && temp < Double.MAX_VALUE)
				{
					max = temp;
				}
			}
		}
		return min / max;
	}

	private double calcAlpha(final int epoch, final int tlen)
	{
		return startAlpha
		* Math.pow((minAlpha / startAlpha), epoch / (tlen - 1.0));
	}

	/**
	 * Calculates current relative radius of neighbourhood. The neighbourhood function is
	 * f(epoch, training length) = * lambda_max *
	 * (lambda_min / lambda_max)^(epoch/(training length - 1))
	 * where max(d[i,j]) is the maximum distance between two point in n-space.
	 * This is then used with maxMutualDist and magicCoeff private fields to
	 * calculate current size of neighbourhood.
	 * @param epoch current epoch
	 * @param tlen length of training
	 * @return current relative radius of neighbourhood
	 */
	private double calcLambda(final int epoch, final int tlen)
	{
		double lambda = startLambda *
		Math.pow((minLambda / startLambda), epoch / (tlen - 1.0));
		return lambda;
	}

	private void preCalcDistances()
	{
		int centCount;
		int vectCount;

		centCount = nSpaceCent.length;
		vectCount = nSpaceVect.length;
		preDistances = new double[vectCount][centCount];
		for (int i = 0; i < vectCount; i++)
		{
			for (int j = 0; j < centCount; j++)
			{
				preDistances[i][j] = dist(nSpaceVect[i], nSpaceCent[j]);
			}
		}
	}

	private void neuralWater(final int maxEpochs)
	{
		int permArray[];
		double maxDist;
		double curAlpha;
		double curDataPoint[];
		double nearestCentroid[];
		ArrayList<double[]> centroids;

		maxDist = 0.0;
		for (int i = 0; i < nSpaceVect.length; i++)
		{

			for (int j = i + 1; j < nSpaceVect.length; j++)
			{
				double tempDist = dist(nSpaceVect[i], nSpaceVect[j]);
				if (tempDist > maxDist)
				{
					maxDist = tempDist;
				}
			}
		}

		maxMutualDist = maxDist;
		maxDist *= maxLoss;

		centroids = new ArrayList<double[]>();
		for (int epoch = 0; epoch < maxEpochs; epoch++)
		{
			permArray = genRandPermutation(nSpaceVect.length);
			curAlpha = calcAlpha(epoch, maxEpochs);
			for (int i = 0; i < nSpaceVect.length; i++)
			{
				double nearestDist;

				curDataPoint = nSpaceVect[permArray[i]];
				nearestCentroid = null;
				nearestDist = Double.MAX_VALUE;
				for (int j = 0, centCount = centroids.size(); j < centCount; j++)
				{
					double tempDist;
					double tempCentroid[];

					tempCentroid = centroids.get(j);
					tempDist = dist(tempCentroid, curDataPoint);
					if (tempDist < maxDist &&
							(nearestCentroid == null || tempDist < nearestDist))
					{
						nearestDist = tempDist;
						nearestCentroid = tempCentroid;
					}
				}
				if (nearestCentroid == null)
				{
					double newCentroid[];
					newCentroid = new double[oDim];
					System.arraycopy(curDataPoint, 0, newCentroid, 0, oDim);
					centroids.add(newCentroid);
				}
				else
				{
					moveTowards(curAlpha, nearestCentroid, curDataPoint);
				}
			}
		}
		nSpaceCent = new double[centroids.size()][];
		centroids.toArray(nSpaceCent);
	}

	/**
	 * Initializes a give set of data-points to two dimensions at random points.
	 * This is done by getting for each vector and each coordinate a random
	 * number between [0,1] and multiplying that by the maximum distance between
	 * points in original dimension.
	 * @param vecs vectors to project to projection dimension
	 * @return matrix of essentially random points
	 */
	private double[][] initializeToPSpace(double[][] vecs)
	{
		double pDimVecs[][];
		double curVec[];

		pDimVecs = new double[vecs.length][pDim];
		for (int i = 0; i < vecs.length; i++)
		{
			curVec = pDimVecs[i];
			for (int j = 0; j < pDim; j++)
			{
				curVec[j] = R.nextDouble() * maxMutualDist;
			}
		}
		return pDimVecs;
	}

	/**
	 * This method projects <code>Centroids</code> to projection space and tries
	 * to get them to such a configuration that distances in projection space
	 * are near to curvilinear distances in N-space.
	 * For every epoch first calculate correspondig alpha and
	 * neighbourhood-radius and the do:
	 * <pre>
	 * FOR each Centroid x[j]
	 * 	FOR each Vector x[i] != x[j]
	 * 		delta(x[i]) = alpha(t) * (D(i,j)/d(i,j) - 1) *
	 * 		F(nhradius(t) - d(i,j)) * (x[i] - x[j]);
	 * 		x[i] = x[i] + delta(x[i]);
	 * 	END FOR
	 * END FOR
	 * </pre><br>
	 * Where D(i,j) is curvilinear distance between <code>Centroids</code>
	 * i and j in N-space and d euclidean distance between same points in
	 * projection space.
	 * F is scaling function used to prefer shorter distances over longer. In
	 * this algorithm we use Heaviside unit-step function as F. x[i] - x[j] is
	 * the vector-difference in projection space between two centroids.<br>
	 *
	 * @param epoch <code>int</code> stating which iteration round is going on.
	 * This parameter is used to calculate alpha and lambda.
	 * @see CDA#calcAlpha(int, int)
	 * @see CDA#calcLambda(int, int)
	 */
	private void projectCentroids(final int epoch)
	{
		double alpha;
		double nhRadius;
		double mCentroid[];
		double sCentroid[];

		int permArray[];
		int curIndex;

		alpha = calcAlpha(epoch, tlen);
		nhRadius = magicCoeff * maxMutualDist * calcLambda(epoch, tlen);

		permArray = genRandPermutation(pSpaceCent.length);
		for (int i = 0; i < pSpaceCent.length; i++)
		{
			if (projectionStatus.isCanceled()) return;
			
			curIndex = permArray[i];
			sCentroid = pSpaceCent[curIndex];
			for (int j = 0; j < pSpaceCent.length; j++)
			{
				double dMap;
				double dOrig;
				double coeff;

				if (curIndex != j)
				{
					mCentroid = pSpaceCent[j];
					dMap = dist(sCentroid, mCentroid);
					if (dMap == 0.0 || dMap == -0.0)
					{
						continue;
					}
					coeff = nhFunction(dMap, nhRadius);
					if (coeff == 0.0 || coeff == -0.0)
					{
						continue;
					}
					dOrig = curvidistances[curIndex][j];
					coeff *= alpha * (dOrig / dMap - 1.0);
					applyDelta(coeff, mCentroid, sCentroid);
				}
			}
		}
	}

	/**
	 * This method projects <code>Vectors</code> to projection space and tries
	 * to get them to such a configuration that distances in projection space
	 * are near to curvilinear distances in N-space.
	 * For every epoch first calculate correspondig alpha and
	 * neighbourhood-radius and the do:
	 * <pre>
	 * FOR each Centroid x[j]
	 * 	FOR each Vector x[i] != x[j]
	 * 		delta(x[i]) = alpha(t) * (D(i,j)/d(i,j) - 1) *
	 * 		F(nhradius(t) - d(i,j)) * (x[i] - x[j]);
	 * 		x[i] = x[i] + delta(x[i]);
	 * 	END FOR
	 * END FOR
	 * </pre><br>
	 * Where D(i,j) is curvilinear distance between <code>Centroid</code>
	 * i and <code>Vector</code> j in N-space and d euclidean distance between
	 * same points in projection space.<code>Centroids</code> are iterated
	 * through in a random fashion.
	 * F is scaling function used to prefer shorter distances over longer. In
	 * this algorithm we use Heaviside unit-step function as F. x[i] - x[j] is
	 * the vector-difference in projection space between two centroids.<br>
	 *
	 * @param epoch <code>int</code> stating which iteration round is going on.
	 * This parameter is used to calculate lambda and alpha.
	 * @see CDA#calcAlpha(int, int)
	 * @see CDA#calcLambda(int, int)
	 */
	private void projectVectors(final int epoch)
	{
		double alpha;
		double nhRadius;
		double curVect[];
		double curCent[];

		int permArray[];
		int centIx;

		alpha = calcAlpha(epoch, tlen);
		nhRadius = magicCoeff * maxMutualDist * calcLambda(epoch, tlen);

		permArray = genRandPermutation(pSpaceCent.length);
		for (int i = 0; i < pSpaceCent.length; i++)
		{
			centIx = permArray[i];
			curCent = pSpaceCent[centIx];
			for (int vecIx = 0; vecIx < pSpaceVect.length; vecIx++)
			{
				double dMap;
				double dOrig;
				double coeff;


				curVect = pSpaceVect[vecIx];
				dMap = dist(curCent, curVect);
				if (dMap == 0.0 || dMap == -0.0)
				{
					continue;
				}
				coeff = nhFunction(dMap, nhRadius);
				if (coeff == 0.0 || coeff == -0.0)
				{
					continue;
				}
				dOrig = calcCurviDistance(vecIx, centIx, nhRadius);
				coeff *= alpha * (dOrig / dMap - 1.0);
				applyDelta(coeff, curVect, curCent);
			}
		}
	}

	/**
	 * Find curvilinear distance from vector denoted by vecIx to centroid
	 * denoted by centIx so that the distance is smallest and the centroid
	 * nearest to vector vecIx is within nhRadius.
	 * @param vecIx
	 * @param centIx
	 * @param nhRadius
	 * @return
	 */
	private double calcCurviDistance(int vecIx, int centIx, double nhRadius)
	{
		int closestCentroidIx;
		double closestDist;
		double tempDist;

		closestDist = Double.MAX_VALUE;
		closestCentroidIx = -1;
		/*
		 * Find nearest centroid to the current data-point such that
		 * dist(curVect, nearestCent) + curvidist(nearestCent, curCent)
		 * is minimal and dist(curVect, nearestCent) <= nhRadius.
		 */
		for (int centroidCandIx = 0; centroidCandIx < pSpaceCent.length; centroidCandIx++)
		{
			tempDist = preDistances[vecIx][centroidCandIx];
			if (tempDist > nhRadius)
			{
				continue;
			}
			if (centroidCandIx != centIx)
			{
				tempDist += curvidistances[centIx][centroidCandIx];
			}
			if (tempDist < closestDist)
			{
				closestDist = tempDist;
				closestCentroidIx = centroidCandIx;
			}
		}
		if (closestCentroidIx < 0)
		{
			closestDist = preDistances[vecIx][centIx];
		}
		return closestDist;
	}

	private AdjacencyList linkKNeighbours()
	{
		FibNode fnArray[];
		int centCount;
		int nearestArray[];
		double nearestDistArray[];

		centCount = nSpaceCent.length;

		nearestArray = new int[centCount];
		nearestDistArray = new double[centCount];
		fnArray = new FibNode[centCount];

		nHoodSize = Math.min(nHoodSize, nSpaceCent.length);

		for (int i = 0; i < centCount; i++)
		{
			fnArray[i] = new FibNode(i);
		}
		neighList = new AdjacencyList(fnArray);

		for (int i = 0; i < centCount; i++)
		{
			for (int j = 0; j < nHoodSize; j++)
			{
				nearestArray[j] = -1;
				nearestDistArray[j] = Double.MAX_VALUE;
			}
			for (int j = 0; j < centCount; j++)
			{
				int newPlace = -1;
				boolean addThisValue = false;
				double curDist;

				if (i == j)
				{
					continue;
				}
				curDist = dist(nSpaceCent[i], nSpaceCent[j]);
				for (int k = 0; k < nHoodSize; k++)
				{
					if (curDist < nearestDistArray[k])
					{
						addThisValue = true;
						newPlace = k;
					}
					else
					{
						break;
					}
				}
				if (addThisValue)
				{
					for (int k = 1; k <= newPlace; k++)
					{
						nearestDistArray[k - 1] = nearestDistArray[k];
						nearestArray[k - 1] = nearestArray[k];
					}
					nearestDistArray[newPlace] = curDist;
					nearestArray[newPlace] = j;
				}
			}
			for (int j = 0; j < nHoodSize; j++)
			{
				neighList.addBiConnection(fnArray[i], fnArray[nearestArray[j]],
						nearestDistArray[j]);
			}
		}
		return neighList;
	}

	/**
	 * Function used to determine weight for given distance with given
	 * neighbourhood-radius. Now it is a heaviside step function so that
	 * all distances under nhRadius contribute fully and all outside that
	 * do not contribute at all.
	 * @param dMap distance between two vectors in mapping dimension
	 * @param nhRadius current radius of neighbourhood
	 * @return
	 */
	private double nhFunction(double dMap, double nhRadius)
	{
		return (dMap <= nhRadius) ? 1 : 0;
	}

	private void applyDelta(double coeff, final double mVec[],
			final double sVec[])
	{
		for (int i = 0, n = mVec.length; i < n; i++)
		{
			mVec[i] += coeff * (mVec[i] - sVec[i]);
		}
	}

	private void moveTowards(double alpha, final double mVec[],
			final double sVec[])
	{
		for (int i = 0; i < mVec.length; i++)
		{
			mVec[i] += alpha * (sVec[i] - mVec[i]);
		}
	}

	private double[][] createDistanceGraph(int numOfNeigh)
	{
		FibNode nodes[] = new FibNode[nSpaceCent.length];
		for (int i = 0; i < nSpaceCent.length; i++)
		{
			nodes[i] = new FibNode(i);
		}
		AdjacencyList allConnections = new AdjacencyList(nodes);
		for (int i = 0, n = nSpaceCent.length; i < n; i++)
		{
			for (int j = i + 1; j < n ; j++)
			{
				if (projectionStatus.isCanceled()) return null;
				
				allConnections.addBiConnection(nodes[i], nodes[j],
						dist(nSpaceCent[i], nSpaceCent[j]));
			}
		}
		PrimMST graphCreator = new PrimMST(allConnections);
		AdjacencyList adlist = graphCreator.generateSpanningTrees(numOfNeigh);
		ShortestPathDijkstra dijk = new ShortestPathDijkstra(adlist);
		return dijk.allPairShortestPaths(false, true);
	}
}
