package jmprojection;

/**
 * Preprocesses data through various methods. For every methods, input-data is
 * <code>double[][]</code> matrix which should be of form:
 * <pre>
 *    X1  X2  X3  X4  X5
 *  1 a   b   c   d   e
 *  2 f   g   h   i   j
 *  3 k   l   m   n   o
 *  4 p   q   r   s   t</pre>
 *  (The above matrix is double[4][5])
 * @author Jarkko Miettinen
 */
public final class Preprocess
{

	public final static double[][] scaleToUnityVarianceDefensiveCopy(double[][] data)
	{
		double newData[][];
		newData = new double[data.length][];
		for (int i = 0; i < data.length; i++)
		{
			newData[i] = new double[data[i].length];
			System.arraycopy(data[i], 0, newData[i], 0, data[i].length);
		}
		scaleToUnityVariance(newData);
		return newData;
	}

	public final static double[][] sigmoidScaleDefensiveCopy(double[][] data)
	{
		double newData[][];
		newData = new double[data.length][];
		for (int i = 0; i < data.length; i++)
		{
			newData[i] = new double[data[i].length];
			for (int j = 0; j < newData[i].length; j++)
			{
				newData[i][j] = 1 / (1.0 + Math.exp(-data[i][j]));
			}
		}
		return newData;
	}

	public final static void sigmoidScale(double[][] data)
	{
		for (int i = 0; i < data.length; i++)
		{
			for (int j = 0; j < data[i].length; j++)
			{
				data[i][j] = 1 / (1.0 + Math.exp(-data[i][j]));
			}
		}
	}

	public final static void scaleToUnityVariance(double[][] data)
	{
		int cols;
		int rows;

		double mean;
		double s;
		double delta;
		double temp;

		rows = data.length;
		cols = data[0].length;

		for (int i = 0; i < cols; i++)
		{
			mean = 0.0;
			s = 0.0;
			delta = 0.0;

			for (int j = 0; j < rows; j++)
			{
				temp = data[j][i];
				delta = temp - mean;
				mean += delta / (j + 1);
				s += delta * (temp - mean);
			}
			s = Math.sqrt(s / (rows - 1));
			for (int j = 0; j < rows; j++)
			{
				data[j][i] -= mean;
				data[j][i] /= s;
			}
		}
	}

	public static double[][] transpose(double[][] m)
	{
		double transposed[][];
		int rows;
		int cols;

		cols = m.length;
		rows = m[0].length;
		transposed = new double[rows][cols];

		for (int i = 0; i < rows; i++)
		{
			for (int j = 0; j < cols; j++)
			{
				transposed[i][j] = m[j][i];
			}
		}
		return transposed;
	}
}
